#include "mbed.h"

#include "C12832.h"
#include "USBHostMSD.h"
#include "wave_player.h"
#include "Servo.h"
#include "Ping.h"

#define BIAS 0.2030303
#define THRES 17

/* States enumated and compartmentalized into functions */
enum States { PATROL, SEARCH, ALERT };
States patrol_state();
States search_state();
States alert_state();

/* Global Variables and components */
DigitalOut board_led1(LED1);
DigitalOut board_led2(LED2);
DigitalOut board_led3(LED3);
DigitalOut board_led4(LED4);
C12832 lcd(p5, p7, p6, p8, p11);
PwmOut r_led(p23);
PwmOut g_led(p24);
PwmOut b_led(p25);

Serial pc(USBTX, USBRX);
USBHostMSD* msd = 0;

enum MOVE {BAD = -1, STOP, FORW, BACK, LTUR, RTUR, LPIV,RPIV };
MOVE g_movement = BAD;
Servo rightServo(p21);
Servo leftServo(p22);

DigitalIn pir1(p29,PullUp);
DigitalIn pir2(p30, PullUp);
Ping ultrasonic_in(p16);
AnalogIn microphone_in(p17);

AnalogOut DACout(p18);
PwmOut board_speaker(p23);//(p26);
wave_player waver(&DACout,&board_speaker);
Serial bluetooth(p9, p10);

/* Helper Functions */

    /*                      *
     *  VISUAL FUNCTIONS    *
     *                      */  
void set_boardleds(int one, int two = board_led2, int thr = board_led3, int fou = board_led4) {
    board_led1 = (one <= 0 ? 0 : 1);
    board_led2 = (two <= 0 ? 0 : 1);
    board_led3 = (thr <= 0 ? 0 : 1);
    board_led4 = (fou <= 0 ? 0 : 1);
}

void lcd_newmessage(const char* const message) {
    lcd.cls();
    lcd.locate(0,3);
    lcd.printf(message);
}

void set_rgb_f(float r, float g, float b) {
    r_led = r;
    g_led = g;
    b_led = b;
}

void set_rgb(unsigned int r, unsigned int g, unsigned int b) {
    r_led = (float)r / 255.f;
    g_led = (float)g / 255.f;
    b_led = (float)b / 255.f;
}

    /*                      *
     *     USB FUNCTIONS    *
     *                      */  
void setup_usb() {
    msd = new USBHostMSD("usb");
    while(!msd->connect()) {
        Thread::wait(500);
    }  
}

void play_file(const char * const filepath) {
    FILE *wave_file;
    wave_file=fopen(filepath,"r");
    if(wave_file == NULL) {
        set_boardleds(1, 1, 1, 1);
        return;
    }
    waver.play(wave_file);
    fclose(wave_file);
}

void play_usbfile(const char * const filename) {
    char filepath[80];
    strcpy(filepath, "/usb/");
    strcat(filepath, filename);
    play_file(filepath);
}

    /*                      *
     *   DRIVE FUNCTIONS    *
     *                      */    
void drive_stop() {
    if(g_movement != STOP) {
        rightServo = 0.5;
        leftServo = 0.5;
        g_movement = STOP;
    }
}
void drive_forward() {
    if(g_movement != FORW) {
        rightServo = 0;
        leftServo = 1;
        g_movement = FORW;
    }
}
void drive_backward() {
    if(g_movement != BACK) {
        rightServo = 1;
        leftServo = 0;
        g_movement = BACK;
    }
}
void drive_turnLeft() {
    if(g_movement != LTUR) {
        rightServo = 0;
        leftServo = 0.5;
        g_movement = LTUR;
    }
}
void drive_turnRight() {
    if(g_movement != RTUR) {
        rightServo = 0.5;
        leftServo = 1;
        g_movement = RTUR;
    }
}
void drive_pivotLeft() {
    if(g_movement != LPIV) {
        rightServo = 0;
        leftServo = 0;
        g_movement = LPIV;
    }
}
void drive_pivotRight() {
    if(g_movement != RPIV) {
        rightServo = 1;
        leftServo = 1;
        g_movement = RPIV;
    }
}

    /*                      *
     *   SENSOR FUNCTIONS   *
     *                      */  
int read_distance() {
    int distance = -1;
    ultrasonic_in.Send();
    wait_ms(30);
    distance = ultrasonic_in.Read_cm();
    return distance;   
}
int motion_detected() {
    return !pir1 || !pir2; //pir pulls low when motion is detected    
}

int hear_sound() {
    if((microphone_in - BIAS)*500 > THRES) {//70% volume 
       return 1;
    } else {
        return 0;
    }
}

    /*                      *
     * WIRELESS FUNCTIONS   *
     *                      */  
void send_email() {
    bluetooth.printf("Send Email\n");
}
