/* mbed Microcontroller Library
 * Copyright (c) 2006-2014 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#include "Hardware.h"
#include "ble/BLE.h"
#include "ble/Gap.h"
#include "ble/services/BatteryService.h"

const uint16_t LIGHTBULB_SERVICE = 0xff10;
const uint16_t SWITCH_CHARACTERISTIC = 0xff11;
const uint16_t DIMMER_CHARACTERISTIC = 0xff12;

Hardware myHardware = Hardware();
static BatteryService* batteryServicePtr;
static DFUService* dfuServicePtr;
static uint8_t batteryLevel = 50;
int bleCmdLen = 0;

const static char     *DEVICE_NAME = "BleDev";
static const uint16_t uuid16_list[] = {GattService::UUID_BATTERY_SERVICE, LIGHTBULB_SERVICE};

/* Setup Lightbulb service */
// Set Up custom Characteristics
static uint8_t dimmerValue[10] = {0};
ReadWriteArrayGattCharacteristic<uint8_t, sizeof(dimmerValue)> dimmer(DIMMER_CHARACTERISTIC, dimmerValue);

static uint8_t switchValue[10] = {0};
ReadWriteArrayGattCharacteristic<uint8_t, sizeof(switchValue)> _switch(SWITCH_CHARACTERISTIC, switchValue);

// Set up custom service
GattCharacteristic *characteristics[] = {&_switch, &dimmer};
GattService customService(LIGHTBULB_SERVICE, characteristics, sizeof(characteristics) / sizeof(GattCharacteristic *));

EventQueue *eventQueue;
//EventQueue eventQueue2(32 * EVENTS_EVENT_SIZE);
Thread *t;

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params) {
    BLE::Instance().gap().startAdvertising();
}

void updateSensorValue() {
    batteryLevel++;
    if (batteryLevel > 100) {
        batteryLevel = 20;
    }

    batteryServicePtr->updateBatteryLevel(batteryLevel);
}

void blinkCallback(void) {
    //*(myHardware.led1) = !*(myHardware.led1); /* Do blinky on LED1 while we're waiting for BLE events */

    BLE &ble = BLE::Instance();
    if (ble.gap().getState().connected) {
        eventQueue->call(updateSensorValue);
    }
}

float ledVal = 0.0f;

void setLedCb(void) {
  if(ledVal > 1.0f)
    ledVal = 0.0f;
  (myHardware.led)->write(ledVal);
  ledVal += 0.05f;    
}

/**
 * This function is called when the ble initialization process has failled
 */
void onBleInitError(BLE &ble, ble_error_t error) {
    /* Initialization error handling should go here */
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params) {
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);

    /* Setup primary service */
    batteryServicePtr = new BatteryService(ble, batteryLevel);
    
    /* Enable over-the-air firmware updates. Instantiating DFUSservice introduces a
    * control characteristic which can be used to trigger the application to
    * handover control to a resident bootloader. */
    dfuServicePtr = new DFUService(ble);

    /* Setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *) uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                     (const uint8_t *)UARTServiceUUID_reversed, sizeof(UARTServiceUUID_reversed));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *) DEVICE_NAME, sizeof(DEVICE_NAME));

    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(200); /* 200ms */
    
    ble.addService(customService);
    
    ble.gap().startAdvertising();
}

void scheduleBleEventsProcessing(BLE::OnEventsToProcessCallbackContext* context) {
    BLE &ble = BLE::Instance();
    eventQueue->call(Callback<void()>(&ble, &BLE::processEvents));
}
    
void processBleCli(void) {
    for(int i = 0; i < bleCmdLen; i++){        
        (myHardware.bleCli)->process();
    }
    (myHardware.bleCli)->doEnter();
}

void postToQueue(); 
 
void processCli() {
    (myHardware.myCli)->process();
    (myHardware.pc)->attach(postToQueue, Serial::RxIrq);
}
 
void postToQueue() {
    (myHardware.pc)->attach(NULL, Serial::RxIrq);
    eventQueue->call(processCli);
}

/*ISR when ble uart data arraived. scedules cli proscess*/
void bleDataWritten(const GattWriteCallbackParams *params) {
    if (params->handle == (myHardware.bleSerial)->getTXCHandle()) {
        bleCmdLen = params->len;
        eventQueue->call(processBleCli);
    }
    if(params->handle == dimmer.getValueHandle()) {
        // toggle LED if only 1 byte is written
        if(params->len == 1) {
            //led = params->data[0];
            float ledVal = (float) params->data[0] / 255.0f;
            (myHardware.led)->write(ledVal);
            //(myHardware.pc)->printf("\n\r %d", params->data[0]);
        }
        // print the data if more than 1 byte is written
        else {
            (myHardware.pc)->printf("\n\r Data received: length = %d, data = 0x",params->len); 
        }
        // update the readChar with the value of writeChar
        BLE::Instance(BLE::DEFAULT_INSTANCE).gattServer().write(dimmer.getValueHandle(), params->data, params->len);
    }
}

int main()
{
    eventQueue = new EventQueue(32 * EVENTS_EVENT_SIZE);
    t = new Thread(osPriorityNormal);
    
    //(myHardware.pc)->printf("INIT BLE\n");
    BLE &ble = BLE::Instance();
    ble.onEventsToProcess(scheduleBleEventsProcessing);
    ble.init(bleInitComplete);
    
    ble.gattServer().onDataWritten(bleDataWritten);//chaining ist von vorn nach hinten!
    
    myHardware.init();
    
    //(myHardware.pc)->printf("INIT RTOS\n");
    t->start(callback(eventQueue, &EventQueue::dispatch_forever));
    eventQueue->call_every(500, blinkCallback);
    //eventQueue->call_every(100, setLedCb);
    (myHardware.pc)->attach(postToQueue, Serial::RxIrq);
    
    (myHardware.led)->write((float)dimmerValue[0]/255.0f);
    
    while(true) sleep();

    return 0;
}
