#include <vector>
#include <numeric>
#include <algorithm>
#include "mbed.h"
#include "Matrix.h"
#include "MODSERIAL.h"
#include "HIDScope.h"
#include "QEI.h"
#include "FastPWM.h"
#include "refGen.h"
#include "controller.h"
#include "motorConfig.h"
#include "errorFetch.h"
#include "BiQuad.h"
#include "inverseKinematics.h"

// ADJUSTABLE PARAMETERS
// robot dimensions
const float L1 = 0.391;
const float L2 = 0.391;

// controller ticker time interval
const float Ts = 0.008;

// Defining an inverse-kinematics calculator
inverseKinematics robotKinematics(L1,L2,Ts);

// EMG filter parameters
// calibration time
const int calSamples = 1000;

// KINEMATICS reference motor position
volatile double Mp1C = 0;
volatile double Mp2C = 0;

// Initialize average and max EMG value for calibration to 0 and 1 respectively
volatile float avgEMGvalue = 0;
volatile double maxEMGvalue = 1;

// Controller parameters
const float k_p = 0.01;
const float k_i = 0; // Still needs a reasonable value
const float k_d = 0; // Again, still need to pick a reasonable value

// Defining motor gear ratio - for BOTH motors as this is the same in the current configuration
const float gearRatio = 131;

// LOGISTICS
// Declaring finite-state-machine states
enum robotStates {KILLED, ACTIVE};
volatile robotStates currentState = KILLED;
volatile bool stateChanged = true;

// Declaring a controller ticker and volatile variables to store encoder counts and revs
Ticker controllerTicker;
volatile int m1counts = 0;
volatile int m2counts = 0;
volatile float m1revs = 0.00;
volatile float m2revs = 0.00;

// PWM settings
float pwmPeriod = 1.0/5000.0;
int frequency_pwm = 10000; //10kHz PWM

// Initializing encoder
QEI Encoder1(D12,D13,NC,64, QEI::X4_ENCODING);
QEI Encoder2(D11,D10,NC,64, QEI::X4_ENCODING);
MODSERIAL pc(USBTX, USBRX);
HIDScope scope(5);

// Defining inputs
InterruptIn sw2(SW2);
InterruptIn sw3(SW3);
InterruptIn button1(D2);
InterruptIn button2(D3);

// Defining LED outputs to indicate robot state-us
DigitalOut ledG(LED_GREEN);
DigitalOut ledR(LED_RED);
DigitalOut ledB(LED_BLUE);

// Setting up HIDscope
volatile float x;
volatile float y;  
volatile float z; 
volatile float q;
volatile float k;
volatile float w;


void sendDataToPc(float data1, float data2, float data3, float data4){
        // Capture data
        x = data1;
        y = data2;
        z = data3;
        q = data4;
        scope.set(0, x);   
        scope.set(1, y); 
        scope.set(2, z);
        scope.set(3, q);
        scope.set(4, z);
        scope.set(5, w);
        scope.send(); // send what's in scope memory to PC
}


// REFERENCE PARAMETERS
int posRevRange = 1; // describes the ends of the position range in complete motor output shaft revolutions in both directions
const float maxAngle = 1*3.14*posRevRange; // max angle in radians


// References based on potmeter 1 and 2
// Set Vx using pot1 = A5
// Set Vy using pot2 = A4
refGen ref1(A4, 0.1);
refGen ref2(A3, 0.1);

// readEncoder reads counts and revs and logs results to serial window
errorFetch e1(gearRatio, Ts);
errorFetch e2(gearRatio, Ts);

// Generate a PID controller with the specified values of k_p, k_d and k_i
controller motorController1(k_p, k_d, k_i);
controller motorController2(k_p, k_d, k_i);

motorConfig motor1(D4,D5);
motorConfig motor2(D7,D6);

// PROBLEM: if I'm processing the state machine in the endless while loop, how can I adjust robot behavior in the ticker (as it'll keep running)? Do I need to also implement it there? If so, why bother with the while(1) in the main function in the first place?
void measureAndControl(){
    // Read encoders and potmeter signal (unnfiltered reference)
    m1counts = Encoder1.getPulses();
    m2counts = Encoder2.getPulses();
    
    double m1position = e1.fetchMotorPosition(m1counts);
    double m2position = e2.fetchMotorPosition(m2counts) - m1position;
    
    double pot1 = ref1.getReference();
    double pot2 = ref2.getReference();
    
    // Finite state machine
    switch(currentState){
            case KILLED:
                {
                // Initialization of KILLED state: cut power to both motors
                if(stateChanged){
                    motor1.kill();
                    motor2.kill();
                    pc.printf("Killed state \r\n");
                    stateChanged = false;               
                    }
                
                // Set LED to red
                ledR = 0;
                ledG = 1;
                ledB = 1;

                sendDataToPc(pot1, pot2, m1counts, m2counts); // just send the EMG signal value to HIDscope
                
                double Mp1Xa = -L1*sin(m1position) - L2*sin(m1position+m2position);
                double Mp2Xa = L1*cos(m1position) + L2*cos(m1position+m2position);
                
//                pc.printf("Mp1Xa is %.2f \r\n", Mp1Xa);
//                pc.printf("Mp2Xa is %.2f \r\n", Mp2Xa);
//                
//                pc.printf("theta1 is %.2f \r\n", m1position);
//                pc.printf("theta2 is %.2f \r\n", m2position);
                
                break;
                }
                
                
            case ACTIVE:
                {
                if(stateChanged){
                    pc.printf("Active state \r\n");
                    Mp1C = m1position;
                    Mp2C = m2position;
                    stateChanged = false;
                    }
                    
                // Using potmeter signals to define a desired end-effector velocity;
                
                double vx = pot1;
                double vy = pot2;
                
                // Translating vx and vy to angular velocities
                Matrix q_dot = robotKinematics.computeAngularVelocities(vx,vy,Mp1C,Mp2C);
                double q_dot1 = q_dot(1,1);
                double q_dot2 = q_dot(2,1);
                
                // Computing position setpoint for next ticker tick using desired end-effector velocity
                double Mp1N = Mp1C + Ts*q_dot1;
                double Mp2N = Mp2C + Ts*q_dot2;
                
                double Mp1X = -L1*sin(Mp1N) - L2*sin(Mp1N+Mp2N);
                double Mp2X = L1*cos(Mp1N) + L2*cos(Mp1N+Mp2N);
                
                double Mp1Xa = -L1*sin(m1position) - L2*sin(m1position+m2position);
                double Mp2Xa = L1*cos(m1position) + L2*cos(m1position+m2position);
                
                // Compute error between actual CURRENT motor position and NEXT position setpoint
                e1.fetchError(m1position, Mp1N);
                e2.fetchError(m2position, Mp2N);
                
                // Compute motor value using controller and set motor
                float motorValue1 = motorController1.control(e1.e_pos, e1.e_int, e1.e_der);
                float motorValue2 = motorController2.control(e2.e_pos, e2.e_int, e2.e_der);
                motor1.setMotor(motorValue1);
                motor2.setMotor(motorValue2);
                
                // Send data to HIDscope
                //sendDataToPc(vx, vy, Mp1N, Mp2N);
                
//                pc.printf("Mp1X is %.2f \r\n", Mp1X);
//                pc.printf("Mp2X is %.2f \r\n", Mp2X);
//                pc.printf("Mp1Xa is %.2f \r\n", Mp1Xa);
//                pc.printf("Mp2Xa is %.2f \r\n", Mp2Xa);
//                pc.printf("vx is %.2f \r\n", vx);
//                pc.printf("vy is %.2f \r\n", vy);
                
                // Prepare for next round
                Mp1C = Mp1N;
                Mp2C = Mp2N;
                
                // Set LED to blue
                ledR = 1;
                ledG = 1;
                ledB = 0;
                // NOTE: state transition is handled using buttons triggering functions motorConfig::kill() and motorConfig::turnMotorOn
                break;
                }
        }
    }

void r1SwitchDirection(){
    ref1.r_direction = !ref1.r_direction;
    pc.printf("Switched reference direction! \r\n");
    }
    
void r2SwitchDirection(){
    ref2.r_direction = !ref2.r_direction;
    pc.printf("Switched reference direction! \r\n");
    }

void killSwitch(){
    currentState = KILLED;
    stateChanged = true;
    }
    
void activateRobot(){
    currentState = ACTIVE;
    stateChanged = true;
    }
    

int main()
    {
    pc.baud(115200);
    pc.printf("Main function");
    
    // Attaching state change functions to buttons;
    sw2.fall(&killSwitch);
    sw3.fall(&activateRobot);
    button1.rise(&r1SwitchDirection);
    button2.rise(&r2SwitchDirection);
    
    controllerTicker.attach(measureAndControl, Ts);
    pc.printf("Encoder ticker attached and baudrate set");    
    }
    
