/**
@file    lcd_functions.h
@author  Tu Tri Huynh
@date    January 13, 2021
@brief   This file contains functions for the LCD display. The display is visually divided into four equal parts to output different things.
@section Useful information
LCD size is (480, 272)
Available font sizes are Font8, Font12, Font16, Font20 or Font24
*/

/**
This function will initialize the LCD display for use
1/13/2021
*/
void lcd_initialize()
{
    BSP_LCD_Init();
    BSP_LCD_LayerDefaultInit(LTDC_ACTIVE_LAYER, LCD_FB_START_ADDRESS);
    BSP_LCD_SelectLayer(LTDC_ACTIVE_LAYER);
}



/**
This function will run a screen in LCD display that tells user to configure the unit
1/13/2021
*/
void lcd_show_setup_screen()
{    
    /// Clears the LCD with specified color
    BSP_LCD_Clear(LCD_COLOR_BLACK);
    BSP_LCD_SetBackColor(LCD_COLOR_BLACK);
    BSP_LCD_SetTextColor(LCD_COLOR_WHITE);
    BSP_LCD_SetFont(&Font24);
    BSP_LCD_DisplayStringAt(0, 50, (uint8_t *)"SETUP", CENTER_MODE);
    BSP_LCD_SetFont(&Font16);
    BSP_LCD_DisplayStringAt(0, 200, (uint8_t *)"Please configure the unit", CENTER_MODE);
}

/**
This function will run the startup screen with project- and developername
1/13/2021
*/
void lcd_show_startup_screen()
{    
    BSP_LCD_Clear(LCD_COLOR_BLACK);
    BSP_LCD_SetBackColor(LCD_COLOR_BLACK);
    BSP_LCD_SetTextColor(LCD_COLOR_ORANGE);
    BSP_LCD_DisplayStringAt(0, 100, (uint8_t *)"Light Control System", CENTER_MODE);
    BSP_LCD_SetTextColor(LCD_COLOR_WHITE);
    BSP_LCD_DisplayStringAt(0, 200, (uint8_t *)"by Tu Tri Huynh", CENTER_MODE);
    HAL_Delay(3000);
}

/**
Sets up the upper left part of screen
1/18/2021
*/
void lcd_upper_left()
{
    BSP_LCD_SetTextColor(LCD_COLOR_ORANGE);
    BSP_LCD_FillRect(0, 0, 240, 136);
}

/**
Update the content of the upper left part of display.
1/18/2021
*/
void lcd_update_upper_left(char current_time[])
{
    BSP_LCD_SetBackColor(LCD_COLOR_ORANGE);
    BSP_LCD_SetTextColor(LCD_COLOR_BLUE);
    BSP_LCD_SetFont(&Font24);
    BSP_LCD_DisplayStringAt(50, 55, (uint8_t *)current_time, LEFT_MODE);
    BSP_LCD_SetTextColor(LCD_COLOR_ORANGE);
    /// Used to hide some weird artifacts in time string
    BSP_LCD_FillRect(185, 52, 30, 30);
}

/**
Sets up the upper right part of display
1/18/2021
*/
void lcd_upper_right()
{
    BSP_LCD_SetBackColor(LCD_COLOR_WHITE);
    BSP_LCD_SetTextColor(LCD_COLOR_WHITE);
    BSP_LCD_FillRect(240, 0, 240, 136);
    BSP_LCD_SetBackColor(LCD_COLOR_WHITE);
    BSP_LCD_SetTextColor(LCD_COLOR_DARKYELLOW);
    BSP_LCD_SetFont(&Font20);
    BSP_LCD_DisplayStringAt(5, 16, (uint8_t *)"Light settings", RIGHT_MODE);
}

/** 
Update the content of the upper right part of display.
@param light_reading The intensity level of registered light in percentage
@param rotary_reading The current position of rotary sensor in percentage
1/18/2021
*/
void lcd_update_upper_right(int light_reading, int rotary_reading)
{
    BSP_LCD_SetBackColor(LCD_COLOR_WHITE);
    BSP_LCD_SetTextColor(LCD_COLOR_DARKYELLOW);
    BSP_LCD_SetFont(&Font16);
    char light_to_lcd[30];
    char rotary_to_lcd[30];
    sprintf(light_to_lcd, "Light intensity: %3i%%", light_reading);
    sprintf(rotary_to_lcd, "Rotary turned: %3i%%", rotary_reading);
    
    BSP_LCD_DisplayStringAt(0, 46, (uint8_t *)light_to_lcd, RIGHT_MODE);
    BSP_LCD_DisplayStringAt(0, 70, (uint8_t *)rotary_to_lcd, RIGHT_MODE);
}

/**
Sets up the lower left part of display
1/13/2021
*/
void lcd_lower_left()
{   
    BSP_LCD_SetBackColor(LCD_COLOR_DARKMAGENTA);
    BSP_LCD_SetTextColor(LCD_COLOR_DARKMAGENTA);
    BSP_LCD_FillRect(0, 136, 240, 136);
    BSP_LCD_SetTextColor(LCD_COLOR_WHITE);
    BSP_LCD_SetFont(&Font20);
    BSP_LCD_DisplayStringAt(5, 160, (uint8_t *)"Location", LEFT_MODE);
}

/**
Updates the content of the lower left area of the display
@param building[] Name of the building
@param room[] Name of the room
1/18/2021
*/
void lcd_update_lower_left(char building[], char room[])
{   
    BSP_LCD_SetBackColor(LCD_COLOR_DARKMAGENTA);
    BSP_LCD_SetTextColor(LCD_COLOR_WHITE);
    BSP_LCD_SetFont(&Font16);
    BSP_LCD_DisplayStringAt(35, 200, (uint8_t *)building, LEFT_MODE);
    BSP_LCD_DisplayStringAt(35, 220, (uint8_t *)room, LEFT_MODE);
}

/**
Sets up the lower right part of display
1/18/2021
*/
void lcd_lower_right()
{
    BSP_LCD_SetTextColor(LCD_COLOR_CYAN);
    BSP_LCD_FillRect(240, 136, 240, 136);
    BSP_LCD_SetBackColor(LCD_COLOR_CYAN);
    BSP_LCD_SetTextColor(LCD_COLOR_ORANGE);
    BSP_LCD_SetFont(&Font20);
    BSP_LCD_DisplayStringAt(5, 160, (uint8_t *)"Temperature", RIGHT_MODE);
    BSP_LCD_SetFont(&Font16);
}

/** Update the content of the lower right part of display.
@param temp The temperature reading from the thermal sensor
@param is_fahrenheit The bool value to determine if temperature unit should switch to fahrenheit or not
1/18/2021
*/
void lcd_update_lower_right(float temp, bool is_fahrenheit)
{
    char temp_text[30];
    if (is_fahrenheit == true)
    {
        float fahrenheit = helper_convert_celsius_to_fahrenheit(temp);
        sprintf(temp_text, "%3.0f F", fahrenheit);
    }
    else
    {
        sprintf(temp_text, "%2.0f C", temp);
    }
    BSP_LCD_SetBackColor(LCD_COLOR_CYAN);
    BSP_LCD_SetTextColor(LCD_COLOR_ORANGE);
    BSP_LCD_SetFont(&Font24);
    BSP_LCD_DisplayStringAt(35, 210, (uint8_t *)temp_text, RIGHT_MODE);
    BSP_LCD_DrawCircle(421, 214, 3);
    BSP_LCD_SetFont(&Font16);
}

/**
This function will show the default main screen on display
1/19/2021
*/
void lcd_show_main_screen()
{
    BSP_LCD_Clear(LCD_COLOR_DARKBLUE); // Clear with color
    lcd_upper_left();
    lcd_lower_left();    
    lcd_lower_right();
    lcd_upper_right();
}

/**
This function will print LCD size in console terminal
1/13/2021
*/
void lcd_get_screen_size()
{
    printf("LCD X size : %zu\n\r",BSP_LCD_GetXSize()); //result: 480
    printf("LCD Y size : %zu\n\r",BSP_LCD_GetYSize()); //result: 272
}