/*
sensor_light_control.h

This file contains code for controlling the blinking/brightness using light and rotary sensors.

"The AnalogIn API reads the voltage as a fraction of the system voltage.
The value is a floating point from 0.0(VSS) to 1.0(VCC). 
For example, if you have a 3.3V system and the applied voltage is 1.65V, then AnalogIn() reads 0.5 as the value."
Above quote from: https://os.mbed.com/docs/mbed-os/v6.6/apis/i-o-apis.html

I'm using the above information to read the light sensor and rotary values, as they are analog input devices.
*/

AnalogIn sensor(A0);
AnalogIn  rotary(A1);
PwmOut led(D3);

// Checks to see if the rotary sensor has been turned
bool rotary_on()
{

    // If rotary has been turned
    if (rotary.read() > 0.10f) {
        return true;
    }
    // If rotary is in position zero return false
    return false;
}

// Checks the angle position of rotary to determine LED blink speed
void control_light_rotary()
{
    float rotaryvalue = rotary.read();

    // If only turned a bit only use a little light (slow blinking)
    if (rotaryvalue >= 0.0f && rotaryvalue <= 0.10f) {
        led.period(1.50f);
        led.write(0.10f);
    }
    // If turned more than above, increase brightness (blink faster) etc...
    else if (rotaryvalue > 0.11f && rotaryvalue <= 0.20f) {
        led.period(1.00f);
        led.write(0.10f);
    } else if (rotaryvalue > 0.21f && rotaryvalue <= 0.40f) {
        led.period(1.00f);
        led.write(0.30f);
    } else if (rotaryvalue > 0.41f && rotaryvalue <= 0.60f) {
        led.period(1.00f);
        led.write(0.50f);
    } else if (rotaryvalue > 0.61f && rotaryvalue <= 0.80f) {
        led.period(0.30f);
        led.write(0.70f);
    }
    // Max brightness
    else if (rotaryvalue > 0.81f && rotaryvalue <= 1.00f) {
        led.period(1.00f);
        led.write(1.00f);
    }
    printf("Rotary reading: %1.2f\r\n", rotaryvalue);
}

// Checks light level to determine brightness of LED
void control_light_sensor()
{
    float sensorvalue = sensor.read();

    // Turn on LED if dark (deactivate energy saving)
    if (sensorvalue > 0.00f) {
        led.resume();
    }

    // Check how much light there is and increase brightness accordingly
    if (sensorvalue < 0.04f) {
        // Turn on completely if very dark
        led.period(1.00f);
        led.write(1.00f);
    }
    // Blink if not completely dark yet not very bright
    else if (sensorvalue >= 0.04f && sensorvalue <= 1.0f) {
        led.period(1.00f);
        led.write(0.50f);
    }
    // If very bright in room, turn off LED (activate energy saving)
    else {
        led.period(1.00f);
        led.write(0.00f);
        led.suspend();
    }
    printf("Light sensor reading: %1.2f\r\n", sensorvalue);
}

// Determine if the rotary or the light sensor controls LED light and run that control
void run_sensors()
{
// If rotary has been turned
    if (rotary_on()) {
        // Activate LED light (power saving function off)
        led.resume();
        // Determine light level (change blinking pattern depending on how much rotary is turned)
        control_light_rotary();
    }
    // If rotary is in position zero (e.g. not turned), let the light sensor control light
    else {
        // Light sensor control activation
        control_light_sensor();
    }
}

// Return the value that the light sensor registers multiplied with 100 to make it easy for percent conversion
float get_light_level()
{
    return sensor.read()*100;
}

// Return the value that the rotary sensor registers multiplied with 100 to make it easy for percent conversion
float get_rotary_turnlevel()
{
    return rotary.read()*100;
}