/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
//This simple program bridges the BLE UART service to the PC Serial conection available over USB
// Based on the work of team UCL-IoT for BLE_UART_LEDCOntrol
// https://developer.mbed.org/teams/UCL-IoT/code/BLE_UART_LEDControl_Echo/

#include "mbed.h"
#include "BLEDevice.h"

#include "UARTService.h"

#define NEED_CONSOLE_OUTPUT 1 /* Set this if you need debug messages on the console;
                               * it will have an impact on code-size and power consumption. */

#if NEED_CONSOLE_OUTPUT
#define DEBUG(...) { printf(__VA_ARGS__); }  //Defaults to stdio without having to wirte pcUart explicitly
#else
#define DEBUG(...) /* nothing */
#endif /* #if NEED_CONSOLE_OUTPUT */

Serial pcUart(USBTX, USBRX); // tx, rx Still required to read data coming from the PC

BLEDevice  ble;                               // Create Bluetooth object

UARTService *uartServicePtr;

/* BLE disconnected callback */
void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    DEBUG("Disconnected!\n\r");
    DEBUG("Restarting the advertising process\n\r");
    ble.startAdvertising();
}

/* BLE UART data received callback */
void onDataWritten(const GattCharacteristicWriteCBParams *params)
{
    if ((uartServicePtr != NULL) && (params->charHandle == uartServicePtr->getTXCharacteristicHandle())) {     //If characters received over BLE
        uint16_t bytesRead = params->len;
        DEBUG("received %u bytes\n\r", bytesRead);
        DEBUG("Received string: '");
        //Note the size of data expands to the largest string received. Need to use bytesRead to resize.
        for (int i=0;i<bytesRead; i++) {
          DEBUG("%c",params->data[i]);      
        }
        DEBUG("'\n\r");       
       // ble.updateCharacteristicValue(uartServicePtr->getRXCharacteristicHandle(), params->data,bytesRead);   // Echo received characters back over BLE
    }
}

/* pcUART read callback */
void rxInterrupt(void)
{
     while (pcUart.readable()) {
         if (pcUart.readable()) {
            char s[2];
            s[1]=0;
            s[0]=pcUart.getc();
     //       DEBUG("-%c_",s[0]);   
            uartServicePtr->write(s,1);
         }
    }  
}

int main(void)
{
    pcUart.attach(&rxInterrupt,Serial::RxIrq);  //interrupt for incoming data available from PC connection

    DEBUG("Initialising the nRF51822\n\r");
    ble.init();
    ble.onDisconnection(disconnectionCallback);                                            // Define callback function for BLE disconnection event
    ble.onDataWritten(onDataWritten);                                                      // Define callback function for BLE Data received event

    /* setup advertising */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);             // Indicate that Legacy Bluetooth in not supported
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                     (const uint8_t *)"BLE UART", sizeof("BLE UART") - 1);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                     (const uint8_t *)UARTServiceUUID_reversed, sizeof(UARTServiceUUID_reversed));

    ble.setAdvertisingInterval(Gap::MSEC_TO_ADVERTISEMENT_DURATION_UNITS(1000));          // Set advertising interval to 1 second
    ble.startAdvertising();                                                               // Start advertising

    UARTService uartService(ble);                                                         // Create BLE UART service
    uartServicePtr = &uartService;                                                        // Initalise pointer to point to UART Service

    while (true) {
        ble.waitForEvent();                                                               // Wait for BLE events
    }
}
