#ifndef SEEED_FOUR_DIGIT_DISP_H
#define SEEED_FOUR_DIGIT_DISP_H
#include "mbed.h"
#include "DataClockPair.h"

class SeeedFourDigitDisp {
  private:
    int digitTable_[17] = {0x3f, 0x06, 0x5b, 0x4f,
                          0x66, 0x6d, 0x7d, 0x07,
                          0x7f, 0x6f, 0x77, 0x7c,
                          0x39, 0x5e, 0x79, 0x71,
                          0x00
                         };
    static const int onByte_ = 0x88;
    static const int fixedAddrByte_ = 0x44;
    static const int positionBit_ = 0xc0;
    static const int nullDigit_ = 16;
    static const int colonBit_ = 0x80;
    void start_cmd();
    void stop_cmd();
    DigitalOut datPin_;
    DigitalOut clkPin_;
    void pin_delay(int delay_us = 1);
    void send_byte(int byte);
  public:
    int brightness = 7;
    bool colonFlag = false;
    void set_digit(int pos, int digit);
    void set_integer(int value);
    void clear_display();
    void turn_on();
    SeeedFourDigitDisp (DataClockPair pins);
};

SeeedFourDigitDisp::SeeedFourDigitDisp(DataClockPair pins) : datPin_(pins.dataPin, 1), clkPin_(pins.clockPin, 1) {
    clear_display();
}

void SeeedFourDigitDisp::pin_delay(int delay_us) {
    wait_us(delay_us);
}

void SeeedFourDigitDisp::start_cmd() {
    datPin_ = !datPin_;
}

void SeeedFourDigitDisp::stop_cmd() {
    datPin_ = 0;
    clkPin_ = !clkPin_;
    pin_delay();
    clkPin_ = !clkPin_;
    datPin_ = !datPin_;
}

void SeeedFourDigitDisp::send_byte(int byte) {
    byte |= 0x100; // bring data high for ack after 8 bits
    for (int i = 0; i < 9; ++i) {
        pin_delay();
        clkPin_ = !clkPin_;
        pin_delay();
        datPin_ = byte & 1;
        byte >>= 1;
        pin_delay();
        clkPin_ = !clkPin_;
    }
}

void SeeedFourDigitDisp::set_digit(int pos, int digit) {
    int flaggedDigit = digitTable_[digit] | (colonFlag ? colonBit_ : 0);
    start_cmd();          //start signal sent to TM1637 from MCU
    send_byte(fixedAddrByte_);
    stop_cmd();
    start_cmd();
    send_byte(pos|positionBit_);
    send_byte(flaggedDigit);
    stop_cmd();
    start_cmd();
    send_byte(onByte_ + brightness);
    stop_cmd();
}

void SeeedFourDigitDisp::turn_on() {
    start_cmd();
    send_byte(onByte_+brightness);
    stop_cmd();
}

void SeeedFourDigitDisp::clear_display() {
    set_digit(0,nullDigit_);
    set_digit(1,nullDigit_);
    set_digit(2,nullDigit_);
    set_digit(3,nullDigit_);
}


void SeeedFourDigitDisp::set_integer(int value) {
    clear_display();
    if( value < 0 ) {
        colonFlag = true;
        set_digit(0, 0);
        return;
    }
    if( value > 9999 ) {
        colonFlag = true;
        set_digit(0, 15);
        return;
    }
    for (int i = 3; i >= 0; --i) {
        int digit = value % 10;
        set_digit(i, digit);
        value -= digit;
        if(value < 10) return;
        value /= 10;
    }
}


#endif
