/**
 * @file    MPR121.h
 * @brief   Device driver - MPR121 capactiive touch IC
 * @author  sam grove
 * @version 1.0
 * @see     http://cache.freescale.com/files/sensors/doc/data_sheet/MPR121.pdf
 *
 * Copyright (c) 2013
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#ifndef MPR121_H
#define MPR121_H

#include "mbed.h"
/* #include "rtos.h"
 */

/** Using the Sparkfun SEN-10250 BoB
 *
 * Example:
 * @code
 *  #include "mbed.h"
 *  #include "MPR121.h"
 *  
 *  Serial pc(USBTX, USBRX);
 *  DigitalOut myled(LED1);
 *  
 *  #if defined TARGET_LPC1768 || TARGET_LPC11U24
 *    I2C i2c(p28, p27);
 *    InterruptIn irq(p26);
 *    MPR121 touch_pad(i2c, irq, MPR121::ADDR_VSS);
 *  
 *  #elif defined TARGET_KL25Z
 *    I2C i2c(PTC9, PTC8);
 *    InterruptIn irq(PTA5);
 *    MPR121 touch_pad(i2c, irq, MPR121::ADDR_VSS);
 *  
 *  #else
 *    #error TARGET NOT TESTED
 *  #endif
 *  
 *  int main()  
 *  {       
 *      touch_pad.init();
 *      touch_pad.enable();
 *      
 *      while(1)
 *      {
 *          if(touch_pad.isPressed())
 *          {
 *              uint16_t button_val = touch_pad.buttonPressed();
 *              printf("button = 0x%04x\n", button_val);
 *              myled = (button_val>0) ? 1 : 0;
 *          }            
 *      }
 *  }
 * @endcode
 */

/**
 *  @class MPR121
 *  @brief API for the MPR121 capacitive touch IC
 */ 
class MPR121
{
private:
    
    I2C         *_i2c;
    uint8_t      _i2c_addr;

public:
    
    /** Create the MPR121 object
     *  @param i2c - A defined I2C object
     *  @param i2c_addr - Connection of the address line
     */    
    MPR121(I2C &i2c);
    
    /** Determine if a new button press event occured
     *  Upon calling the state is cleared until another press is detected
     *  @return 1 if a press has been detected since the last call, 0 otherwise
     */
    bool isPressed(void);
    
    /** Get the electrode status (ELE12 ... ELE0 -> b15 xxx b11 ... b0
     *  The buttons are bit mapped. ELE0 = b0 ... ELE11 = b11 b12 ... b15 undefined
     *  @return The state of all buttons
     */
    uint16_t buttonPressed(void);
};

#endif

