#include "mbed.h"
#include "rtos.h"

RawSerial  dev(PA_9, PA_10);
SPI spiOne(PB_5, PB_4, PB_3);
SPI spiTwo(PA_7, PA_6, PA_5);
const int nLedsA = 455;
const int nLedsB = 455;
enum Init {brightness = 4, frequency = 500000, red = 35, green = 3, blue = 0, baud = 9600};

class DotStar {
    SPI* const spi_;
    const int ledNum_;
    int brightness_;
    int colors_[3];
  public:
    enum { red = 0, green = 1, blue = 2 };
    enum { redIndex = 2, greenIndex = 3, blueIndex = 4 };
    enum {off = 0, dim = 8, half = 16, brightest = 31};
    enum {pause = 100, sleep = 2000};
    enum cmdType : char {
        setColor = '!',
        onOff = 'o',
        switchStrip = 's'
    };
    void set_color(const int color, const int val);
    void set_leds();
    DotStar(SPI* const spi, const int nLeds);
    void set_brightness(const int brightness);
};

struct RedGreenBlue {
    int red, green, blue;
};

class DotStarPair {
    DotStar* left;
    DotStar* right;
  public:
    DotStarPair(DotStar* l, DotStar* r);
    void set_brightness(const int brightness);
    void set_rgb(const RedGreenBlue& rgb);
};

DotStarPair::DotStarPair(DotStar* l, DotStar* r) : left(l), right(r) {}

void DotStarPair::set_brightness(const int brightness) {
    left->set_brightness(brightness);
    left->set_leds();
    right->set_brightness(brightness);
    right->set_leds();
};

void DotStarPair::set_rgb(const RedGreenBlue& rgb) {
    left->set_color(DotStar::blue, rgb.blue);
    left->set_color(DotStar::red, rgb.red);
    left->set_color(DotStar::green, rgb.green);
    left->set_leds();
    right->set_color(DotStar::blue, rgb.blue);
    right->set_color(DotStar::red, rgb.red);
    right->set_color(DotStar::green, rgb.green);
    right->set_leds();
};

DotStar::DotStar(SPI* const spi, const int nLeds) : spi_(spi), ledNum_(nLeds),
    brightness_(Init::brightness) {
    spi_->frequency(Init::frequency);
    colors_[DotStar::red] = Init::red;
    colors_[DotStar::blue] = Init::blue;
    colors_[DotStar::green] = Init::green;
    set_leds();
}


void DotStar::set_leds() {
    const int brightnessFrame = (7<<5)|brightness_;
    const int blueFrame = (colors_[DotStar::blue] ) & 0xff;
    const int greenFrame = (colors_[DotStar::green]) & 0xff;
    const int redFrame = colors_[DotStar::red] & 0xff;
    int i;
    Thread::wait(DotStar::pause);
    for (i = 4; i --> 0; spi_->write(0)) { } // start frame
    for (i = 0; i < ledNum_ ; ++i) {
        spi_->write(brightnessFrame); // led frame
        spi_->write(blueFrame); // B
        spi_->write(greenFrame); // G
        spi_->write(redFrame); // R
    }
    for (i = 4; i --> 0; spi_->write(1)) { } // end frame
}

void DotStar::set_brightness(const int brightness) {
    brightness_ = brightness;
}

void DotStar::set_color(const int c, const int val) {
    colors_[c] = val;
};

DotStar dotStarA(&spiOne, nLedsA);
DotStar dotStarB(&spiTwo, nLedsB);

DotStarPair strips(&dotStarA, &dotStarB);

int main() {
    dev.baud(Init::baud);
    for (int i = 1; i < DotStar::brightest; i += 4) {
        strips.set_brightness(i);
    }
    bool sleeping = false;
#define STR_BUFF_SIZE_VAR 6
    char bleCmdString[STR_BUFF_SIZE_VAR] = {};
    for (; ; ) {
        for (int i = 0; i < STR_BUFF_SIZE_VAR; i++) {
            bleCmdString[i] = 0;
        }
        dev.putc('a');
        strips.set_brightness(DotStar::half);
        int i = 0;
        int j = 0;
        while (i < STR_BUFF_SIZE_VAR) {
            while(dev.readable()) {
                bleCmdString[i] = dev.getc();
                ++i;
            }
            if(j++ > 30000000) break;
        }
        /* strips.set_brightness(DotStar::dim); */
        auto cmdPrefix = bleCmdString[0];
        if (cmdPrefix == DotStar::cmdType::setColor) {
            RedGreenBlue rgb;
            rgb.red = (int)(bleCmdString[DotStar::redIndex]);
            rgb.green = (int)(bleCmdString[DotStar::greenIndex]);
            rgb.blue = (int)(bleCmdString[DotStar::blueIndex]);
            strips.set_rgb(rgb);
            /* strips.set_brightness(DotStar::half); */
            /* Thread::wait(DotStar::sleep); */
        } else if (cmdPrefix == DotStar::cmdType::onOff) {
            sleeping = true;
            strips.set_brightness(DotStar::dim);
            Thread::wait(DotStar::pause);
        }
        while (sleeping) {
            strips.set_brightness(DotStar::off);
            Thread::wait(DotStar::sleep);
            while (dev.readable()) {
                if (dev.getc() == DotStar::cmdType::onOff) {
                    sleeping = false;
                    strips.set_brightness(DotStar::dim);
                    Thread::wait(DotStar::pause);
                }
            }
        }
    }

}
