#include "rtos.h"


struct RedGreenBlue {
    int red, green, blue;
};

class DotStar {
    enum Init {brightness = 4, frequency = 500000, r = 35, g = 3, b = 0};
    SPI* const spi_;
    const int ledNum_;
    int brightness_;
    int colors_[3];
  public:
    enum { red = 0, green = 1, blue = 2 };
    enum { redIndex = 2, greenIndex = 3, blueIndex = 4 };
    enum {off = 0, dim = 8, half = 16, brightest = 31};
    enum {pause = 100, sleep = 2000};
    enum cmdType : char {
        setColor = '!',
        onOff = 'o',
    };
    void set_color(const int color, const int val);
    void set_rgb(const RedGreenBlue& rgb);
    void set_leds();
    DotStar(SPI* const spi, const int nLeds);
    void set_brightness(const int brightness);
};

class DotStarPair {
    DotStar* left;
    DotStar* right;
  public:
    DotStarPair(DotStar* l, DotStar* r);
    void set_brightness(const int brightness);
    void set_rgb(const RedGreenBlue& rgb);
};

DotStarPair::DotStarPair(DotStar* l, DotStar* r) : left(l), right(r) {}

void DotStarPair::set_brightness(const int brightness) {
    left->set_brightness(brightness);
    left->set_leds();
    right->set_brightness(brightness);
    right->set_leds();
};

void DotStarPair::set_rgb(const RedGreenBlue& rgb) {
    left->set_color(DotStar::blue, rgb.blue);
    left->set_color(DotStar::red, rgb.red);
    left->set_color(DotStar::green, rgb.green);
    left->set_leds();
    right->set_color(DotStar::blue, rgb.blue);
    right->set_color(DotStar::red, rgb.red);
    right->set_color(DotStar::green, rgb.green);
    right->set_leds();
};

DotStar::DotStar(SPI* const spi, const int nLeds) : spi_(spi), ledNum_(nLeds),
    brightness_(Init::brightness) {
    spi_->frequency(Init::frequency);
    colors_[DotStar::red] = Init::r;
    colors_[DotStar::blue] = Init::b;
    colors_[DotStar::green] = Init::g;
    set_leds();
}

void DotStar::set_leds() {
    const int brightnessFrame = (7<<5)|brightness_;
    const int blueFrame = (colors_[DotStar::blue] ) & 0xff;
    const int greenFrame = (colors_[DotStar::green]) & 0xff;
    const int redFrame = colors_[DotStar::red] & 0xff;
    int i;
    Thread::wait(DotStar::pause);
    for (i = 4; i --> 0; spi_->write(0)) { } // start frame
    for (i = 0; i < ledNum_ ; ++i) {
        spi_->write(brightnessFrame); // led frame
        spi_->write(blueFrame); // B
        spi_->write(greenFrame); // G
        spi_->write(redFrame); // R
    }
    for (i = 4; i --> 0; spi_->write(1)) { } // end frame
}

void DotStar::set_brightness(const int brightness) {
    brightness_ = brightness;
    set_leds();
}

void DotStar::set_rgb(const RedGreenBlue& rgb) {
    set_color(DotStar::blue, rgb.blue);
    set_color(DotStar::red, rgb.red);
    set_color(DotStar::green, rgb.green);
    set_leds();
};

void DotStar::set_color(const int c, const int val) {
    colors_[c] = val;
};
