#include "mbed.h"
#include "LoraTransport.h"

Serial pc(PA_2, PA_3);

uint8_t responsePackage[MAX_PAYLOAD_SIZE + 4] = {0};

uint8_t buf[MAX_PAYLOAD_SIZE] = {0};
uint8_t counter = 0;
uint16_t addr = 0;
uint8_t length = 0;

enum Command {
    SETADDR = 0,
    SENDTO = 1
};

enum State {
    NO_CMD = -1,

    GOT_CMD_SETADDR,
    GOT_SETADDR_B1,

    GOT_CMD_SENDTO,
    GOT_SENDTO_ADDR_B1,
    GOT_SENDTO_ADDR_B2,
    GOT_SENDTO_LEN,
    GOT_SENDTO_MSG
} state = NO_CMD;

void setCmd(char byte) {
    switch (byte) {
        case SETADDR: {
            state = GOT_CMD_SETADDR;
            break;
        }
        case SENDTO: {
            state = GOT_CMD_SENDTO;
            break;
        }
    }
}

void writeToSerial(const char* msg, const size_t length) {
    for (size_t i = 0; i < length; i++) {
        pc.putc(msg[i]);
    }
}

void RXHandler() {
    while (pc.readable()) {
        char byte = pc.getc();
        switch (state) {
            case NO_CMD: {
                setCmd(byte);
                break;
            }

            case GOT_CMD_SETADDR: {
                addr = byte << 8;
                state = GOT_SETADDR_B1;
                break;
            }
            case GOT_SETADDR_B1: {
                addr += byte;
                LoraTransport::setAddress(addr);
                writeToSerial("\x80\x01", 2);
                state = NO_CMD;
                break;
            }

            case GOT_CMD_SENDTO: {
                addr = byte << 8;
                state = GOT_SENDTO_ADDR_B1;
                break;
            }
            case GOT_SENDTO_ADDR_B1: {
                addr += byte;
                state = GOT_SENDTO_ADDR_B2;
                break;
            }
            case GOT_SENDTO_ADDR_B2: {
                length = byte;
                state = GOT_SENDTO_LEN;
                counter = 0;
                break;
            }
            case GOT_SENDTO_LEN: {
                buf[counter++] = byte;

                if (counter == length) {
                    counter = 0;
                    state = GOT_SENDTO_MSG;
                }
                break;
            }
        }
    }
}

int main() {
    LoraTransport::init();
    pc.attach(&RXHandler);

    while (1) {
        if (state == GOT_SENDTO_MSG) {
            LoraTransport::send(addr, buf, length);
            for (int i = 0; i < MAX_PAYLOAD_SIZE; i++) {
                buf[i] = 0;
            }

            while (!LoraTransport::done()) {
                wait_ms(10);
            }

            if (LoraTransport::success()) {
                writeToSerial("\x81\x01", 2);
            } else {
                writeToSerial("\x81\x00", 2);
            }
            state = NO_CMD;
        } else {
            LoraTransport::recv();

            while (!LoraTransport::done()) {
                wait_ms(10);
            }

            if (LoraTransport::success()) {
                LoraTransport::Packet &pkg = LoraTransport::getPacket();
                responsePackage[0] = 255;
                responsePackage[1] = pkg.from_addr >> 8;
                responsePackage[2] = pkg.from_addr & 0x0f;
                responsePackage[3] = pkg.payload_size;
                memcpy(responsePackage + 4, pkg.payload, pkg.payload_size);
                writeToSerial((const char*) responsePackage, pkg.payload_size + 4);
            }
        }
    }
}
