#include "mbed.h"

//------------------------------------
// 9600 baud, 8-bit data, no parity
//------------------------------------

#define BYTEBINPATTERN "%d%d%d%d%d%d%d%d"
#define BYTETOBIN(byte)  \
  (byte & 0x80 ? 1 : 0), \
  (byte & 0x40 ? 1 : 0), \
  (byte & 0x20 ? 1 : 0), \
  (byte & 0x10 ? 1 : 0), \
  (byte & 0x08 ? 1 : 0), \
  (byte & 0x04 ? 1 : 0), \
  (byte & 0x02 ? 1 : 0), \
  (byte & 0x01 ? 1 : 0)

Serial ser(SERIAL_TX, SERIAL_RX);

DigitalIn in[16] = {
    PB_12, PA_11, PA_12, PC_5,
    PC_6, PC_8, PC_9, PB_8,
    PB_9, PA_6, PA_7, PB_6,
    PC_7, PA_9, PA_8, PB_10
};
    
DigitalInOut out[16] = {
    PD_2, PC_11, PC_10, PC_12,
    PH_1, PC_2, PC_3, PC_0,
    PC_1, PB_0, PA_4, PA_1,
    PA_0, PB_3, PB_5, PB_4
};

DigitalOut userled(LED1);

//------------------------------------

void input_state_print()
{
    ser.printf("i");
        
    for (int i = 0; i < 16; i++) {
        ser.printf("%c", in[i] ? 0x31 : 0x30);
    }
        
    ser.printf("\r\n");
}

void setup(void)
{
  RCC_OscInitTypeDef RCC_OscInitStruct;

  // use internal osc - we don't need the external one for any reason
  // and we want to also use an osc pin
  HAL_RCC_DeInit();
  
  RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSI;
  RCC_OscInitStruct.HSIState = RCC_HSI_ON;
  RCC_OscInitStruct.HSICalibrationValue = 16;
  RCC_OscInitStruct.PLL.PLLState = RCC_PLL_NONE;
  HAL_RCC_OscConfig(&RCC_OscInitStruct);
  
  // HAL_SYSTICK_Config(HAL_RCC_GetHCLKFreq()/1000);

  HAL_SYSTICK_CLKSourceConfig(SYSTICK_CLKSOURCE_HCLK);
  
  SystemCoreClockUpdate();
};

int main()
{
    setup();
    
    // reinitialize the serial port since the clock changed
    Serial ser(SERIAL_TX, SERIAL_RX);
    ser.baud(9600);
    
    // initialize the input ticker
    Ticker input_autoprint;
    
    ser.printf("\r\ntrue Nucleo IOReg for dwangoAC v0.1\r\n");
    ser.printf("Running at %d Hz\r\n", SystemCoreClock);
    ser.printf("Use command \"i\" to print input pin state\r\n");
    ser.printf("Use command \"oXX\" to set outputs where XX is raw 16bit value\r\n");
    ser.printf("  1 = short pin to GND, 0 = HiZ pin\r\n");
    
    while (1) {
        if (ser.readable()) {
            switch (ser.getc()) {
                case 'i': {
                    input_state_print();
                    break;
                }
                case 'I': {     // automatically send input from 0.1 to 9.9 seconds
                    uint8_t rate0 = ser.getc();
                    uint8_t rate1 = ser.getc();
                    
                    if (rate0 >= 0x30 && rate0 <= 0x39) {
                        rate0 -= 0x30;
                        if (rate1 >= 0x30 && rate1 <= 0x39) {
                            rate1 -= 0x30;
                            uint8_t rate = (rate0 * 10) + rate1;
                            
                            if (rate == 0) {
                                input_autoprint.detach();
                            } else {
                                input_autoprint.attach(&input_state_print, 0.1 * rate);   
                            }    
                        }    
                    }
                    
                    break;
                }
                
                case 'o': {
                    uint8_t bh = ser.getc();
                    uint8_t bl = ser.getc();
                    
                    // set light confirming command receipt
                    userled = 1;
                    
                    // we can give feedback to our program what is being set
                    printf("o"BYTEBINPATTERN BYTEBINPATTERN"\r\n", BYTETOBIN(bh), BYTETOBIN(bl));
                    
                    // actually set pins = 1 sets active low, 0 sets hiz
                    uint16_t bits = bh << 8 | bl;
                    
                    for (int i = 0; i < 16; i++) {
                        if (bits & (1 << i)) {
                            // active low output
                            out[i] = 0;
                            out[i].output();
                        } else {
                            // hi-z input
                            out[i].input();
                        }
                    }
                    
                    // wait a moment, unset light
                    wait(0.2);
                    userled = 0;
                    break;
                }
            }
        }
    }
}