#ifndef _D7A_COM_H_
#define _D7A_COM_H_

#include "mbed.h"
#include "rtos.h"
#include "d7a_common.h"
#include "d7a.h"


typedef enum
{
    HAL_ERROR_NONE                      =    0,
    HAL_ERROR_DEFAULT                   =   -1,
    HAL_ERROR_MEM_FULL                  =   -2,
    HAL_ERROR_MEM_EMPTY                 =   -3,
    HAL_ERROR_MODULE_BUSY               =   -4,
    HAL_ERROR_MODULE_OFF                =   -5,
    HAL_ERROR_SERIAL_BAD_SYNC           = -100,
    HAL_ERROR_SERIAL_BAD_SEQ            = -101,
    HAL_ERROR_I2C_BAD_DEVID             = -200,
    HAL_ERROR_I2C_NACK                  = -201,
    HAL_ERROR_I2C_ARB_LOST              = -202,

} hal_error_t;

// +--------------+--------+--------+--------+---- - - - - - - - - - - --------+
// |     SYNC     |   LEN  |   SEQ  |   ID   |             PAYLOAD             |
// +--------------+--------+--------+--------+---- - - - - - - - - - - --------+
//
//      2 bytes     1 byte   1 byte   1 byte              LEN bytes
// |<------------>|<------>|<------>|<------>|<--- - - - - - - - - - - ------->|


// first byte of the sync word
// (ASCII start of heading)
#define KAL_COM_SYNC_BYTE_0          0x01

// second byte of the sync word
// (ASCII group separator)
#define KAL_COM_SYNC_BYTE_1          0x1F

// message header length in byte
#define KAL_COM_HEADER_LEN           5


//======================================================================
// d7a_com_fid_t
//----------------------------------------------------------------------
// Enumerator of serial Flow-ids
//======================================================================
typedef enum
{
    // Trace channel
    KAL_COM_FLOWID_TRC = 0,
    // General purpose Command channel
    KAL_COM_FLOWID_CMD,
    // ALP channel
    KAL_COM_FLOWID_ALP,
    // System notifications
    KAL_COM_FLOWID_SYS,
    // File System channel
    KAL_COM_FLOWID_FS,

    KAL_COM_FLOWID_QTY

} d7a_com_fid_t;

#define KAL_COM_FLOW(fid,type)      ((((fid)&0x7)<<4) | ((type)&0xF))
#define KAL_COM_FLOWID(id)          (((id)>>4)&0x7)
#define KAL_COM_FLOWID_REDIRECT     0x80

//======================================================================
// d7a_com_flow_t
//----------------------------------------------------------------------
// Enumerator of serial flows
//======================================================================
typedef enum
{
    // Default printf type
    KAL_COM_FLOW_PRINTF                     = KAL_COM_FLOW(KAL_COM_FLOWID_TRC,0),
    // Substitute the string by a codeword
    // interpreted by the PC com tool
    KAL_COM_FLOW_PRINTF_COMPRESSED          = KAL_COM_FLOW(KAL_COM_FLOWID_TRC,1),
    // Display the payload as hex data
    KAL_COM_FLOW_PRINT_HEX                  = KAL_COM_FLOW(KAL_COM_FLOWID_TRC,2),

    // AT command
    KAL_COM_FLOW_AT_CMD                     = KAL_COM_FLOW(KAL_COM_FLOWID_ALP,0),
    // AT command response
    KAL_COM_FLOW_AT_RESP                    = KAL_COM_FLOW(KAL_COM_FLOWID_ALP,1),
    // AT unsolicited message
    KAL_COM_FLOW_AT_UNS                     = KAL_COM_FLOW(KAL_COM_FLOWID_ALP,2),
    // AT unsolicited error
    KAL_COM_FLOW_AT_ERR                     = KAL_COM_FLOW(KAL_COM_FLOWID_ALP,3),

    // Remote Commands
    KAL_COM_FLOW_CMD                        = KAL_COM_FLOW(KAL_COM_FLOWID_CMD,0),

    // Remote System reset
    KAL_COM_FLOW_SYS_RST                    = KAL_COM_FLOW(KAL_COM_FLOWID_SYS,0),
    // Button Emulator
    KAL_COM_FLOW_SYS_BUTTON                 = KAL_COM_FLOW(KAL_COM_FLOWID_SYS,1),
    // Dump Debug parameters
    KAL_COM_FLOW_SYS_INFO                   = KAL_COM_FLOW(KAL_COM_FLOWID_SYS,2),
    // CUP signalisation
    KAL_COM_FLOW_SYS_CUP                    = KAL_COM_FLOW(KAL_COM_FLOWID_SYS,3),
    // Ping distant COM
    KAL_COM_FLOW_SYS_PING                   = KAL_COM_FLOW(KAL_COM_FLOWID_SYS,4),
    // Pong from distant COM
    KAL_COM_FLOW_SYS_PONG                   = KAL_COM_FLOW(KAL_COM_FLOWID_SYS,5),
    // Enable system config from distant COM
    KAL_COM_FLOW_SYS_CFG                    = KAL_COM_FLOW(KAL_COM_FLOWID_SYS,6),
    // Configure Output Trace level from distant COM
    KAL_COM_FLOW_SYS_TLEV                   = KAL_COM_FLOW(KAL_COM_FLOWID_SYS,7),
    // Configure COM port redirection
    KAL_COM_FLOW_SYS_REDIR                  = KAL_COM_FLOW(KAL_COM_FLOWID_SYS,8),
    // Flow control signalling
    KAL_COM_FLOW_SYS_XON                    = KAL_COM_FLOW(KAL_COM_FLOWID_SYS,9),
    KAL_COM_FLOW_SYS_XOFF                   = KAL_COM_FLOW(KAL_COM_FLOWID_SYS,10),
    KAL_COM_FLOW_SYS_XACK                   = KAL_COM_FLOW(KAL_COM_FLOWID_SYS,11),

    // File System Command/Response
    KAL_COM_FLOW_FS_CMD                     = KAL_COM_FLOW(KAL_COM_FLOWID_FS,0),
    KAL_COM_FLOW_FS_RESP                    = KAL_COM_FLOW(KAL_COM_FLOWID_FS,1),
    
} d7a_com_flow_t;


//======================================================================
// d7a_com_tx_msg_t
//----------------------------------------------------------------------
// Transmit message structure
//======================================================================
typedef struct
{
    // identifier d7a_com_flow_t
    uint8_t  id;
    // length of the string buffer defined by a pointer
    uint8_t  plen;
    // length of the allocated buffer
    uint8_t  alen;
    // pointer to a string buffer that does not need to be freed
    uint8_t* pbuf;
    // pointer to argument that does not need to be freed
    uint8_t* abuf;

} d7a_com_tx_msg_t;

//======================================================================
// d7a_com_rx_msg_t
//----------------------------------------------------------------------
// Receive message structure
//======================================================================
typedef struct
{
    // error message
    hal_error_t err;
    // length of the log (string) buffer
    uint8_t  blen;
    // identifier d7a_com_flow_t
    uint8_t  id;
    // Com port where the message came from
    uint8_t com_id;
    // pointer to the log buffer
    uint8_t  buffer[1];

} d7a_com_rx_msg_t;


d7a_errors_t d7a_com_open( const d7a_com_config_t* config );
d7a_errors_t d7a_com_close();
void d7a_com_post_msg(d7a_com_tx_msg_t* msg);
void d7a_com_dump(uint8_t* buf, uint8_t len, d7a_com_flow_t flow);
d7a_com_rx_msg_t* d7a_com_wait_pkt(uint32_t millisec = osWaitForever);
void d7a_com_start_rx(void);
void d7a_com_stop_rx(void);
void d7a_com_start_tx(void);
void d7a_com_stop_tx(void);
void d7a_com_restart(void);
void d7a_com_print_stats(void);

#endif // _D7A_COM_H_