#ifndef _D7A_FS_H_
#define _D7A_FS_H_

#include "mbed.h"
#include "rtos.h"
#include "dbg.h"
#include "d7a_common.h"
#include "d7a_com.h"
#include "d7a.h"

#define TO_FS       (8000)     // Long time to give the modem time to reboot in case of assert
#define FS_MAX_CONCURRENCY  16 // Maximum number of concurrent access

// PROP Byte
#define FS_BFO_STORAGE_CLASS    0
#define FS_BFO_ACT_COND         4
#define FS_BFO_ACT_EN           7
#define FS_BFS_STORAGE_CLASS    2
#define FS_BFS_ACT_COND         3
#define FS_BFS_ACT_EN           1

//======================================================================
// Attributes and macros
//======================================================================
// "Piped" File: rd/wr are not stored but just sent on IF
#define FS_TRANSIENT         (TRANSIENT  << FS_BFO_STORAGE_CLASS)
// "RAM" File: rd/wr to a volatile buffer.
#define FS_VOLATILE          (VOLATILE   << FS_BFO_STORAGE_CLASS)
// "Mirrored" File: loaded from NVM, cached/used in RAM. Flushable to NVM.
#define FS_RESTORABLE        (RESTORABLE << FS_BFO_STORAGE_CLASS)
// "Normal" File: rd/wr from/to NVM
#define FS_PERMANENT         (PERMANENT  << FS_BFO_STORAGE_CLASS)
#define FS_ACT_COND(c)       ((c & 0x7) << FS_BFO_ACT_COND)
#define FS_ACT_EN            (1  << FS_BFO_ACT_EN)

#define FS_RUNNABLE          (1  << FS_BFO_RUNNABLE)
#define FS_ENCRYPTED         (1  << FS_BFO_ENCRYPTED)

// XXX Old ALP Hack
#define NACK                 FS_RUNNABLE
#define NOTIF                FS_ENCRYPTED

// D7AactP Enabled File properties
#if 0 // Condition is not used (nor usable)
    #define FS_TRANSIENT_NOTIF(c) (FS_TRANSIENT  | FS_ACT_EN | FS_ACT_COND(c))
    #define FS_VOLATILE_NOTIF(c)  (FS_VOLATILE   | FS_ACT_EN | FS_ACT_COND(c))
    #define FS_RESTORABLE_NOTIF(c)(FS_RESTORABLE | FS_ACT_EN | FS_ACT_COND(c))
    #define FS_PERMANENT_NOTIF(c) (FS_PERMANENT  | FS_ACT_EN | FS_ACT_COND(c))
#else
    #define FS_TRANSIENT_NOTIF    (FS_TRANSIENT  | FS_ACT_EN)
    #define FS_VOLATILE_NOTIF     (FS_VOLATILE   | FS_ACT_EN)
    #define FS_RESTORABLE_NOTIF   (FS_RESTORABLE | FS_ACT_EN)
    #define FS_PERMANENT_NOTIF    (FS_PERMANENT  | FS_ACT_EN)
#endif

// COM-FS Operation-codes. Mimics ALP ones.
#define FS_OP_NULL      0
#define FS_OP_RD        1
#define FS_OP_WR        4
#define FS_OP_SYNC      6  // Write File Properties
#define FS_OP_TOUCH     7  // Not in ALP but '7' is still in 'write' actions area
#define FS_OP_STAT      16
#define FS_OP_DSTAT     30 // Not in ALP: Variant of STAT for distant files
#define FS_OP_CREATE    17
#define FS_OP_FLUSH     20
#define FS_OP_EXECUTE   31
#define FS_OP_RETSTAT   32
#define FS_OP_RETDATA   34


// FS/COM-FS Status-codes. Mimics ALP ones.
enum {
    FS_STAT_PENDING                =  1,
    FS_STAT_OK                     =  0,
    FS_STAT_ERR_FID_NOEXIST        = -1,    // FF
    FS_STAT_ERR_FID_ALREADYEXIST   = -2,    // FE
    FS_STAT_ERR_NOTRESTORABLE      = -3,    // FD
    FS_STAT_ERR_PERMISSION         = -4,    // FC
    FS_STAT_ERR_LENGTH_OVFL        = -5,    // FB
    FS_STAT_ERR_ALLOC_OVFL         = -6,    // FA
    FS_STAT_ERR_START_OFFSET_OVFL  = -7,    // F9
    FS_STAT_ERR_WRITE_OVFL         = -8,    // F8
    FS_STAT_ERR_WRITE_PROTECTED    = -9,    // F7
    FS_STAT_ERR_UNKNOWN_OPERATION  = -10,   // F6
    FS_STAT_ERR_INCOMPLETE_OPERAND = -11,   // F5
    FS_STAT_ERR_WRONG_OPERAND      = -12,   // F4
    FS_STAT_ERR_UNKNOWN            = -127,  // 80
};


// Memory "TYPE"
// ------------------------------------------------------
// |  Sync   | Distant |    Com       |   MemType       |
// ------------------------------------------------------
// |   b7    |   b6    |   b5-b4      |     b3-b0       |
// ------------------------------------------------------
#define FS_TYPE_SYNC        0x80
#define FS_TYPE_DISTANT     0x40
#define FS_MEM_TYPE(t)      ((t)&0x0F)
#define FS_COM_TYPE(t)      ((t)&0x30)
#define FS_DISTANT_TYPE(t)  ((t)&0x70)

enum {
    // Markers
        NOFILE  = 0,
        ERASED  = 0xff,
    // Physical Type
        PFLASH  = 1,
        EEPROM  = 2,
        RAM     = 3,
        OTP     = 4,
        UNKNOWN = 0x0F,
        // PSRAM
        // DDR
        // ...etc

        FS_COM_UART    = (0<<4),
        FS_COM_SPI     = (1<<4),
        FS_COM_I2C     = (2<<4),
    // DISTANT Full Types
        // Symetric HST-COM Files:
        // - Header on Host side (i.e. need syncro on STAT)
        // - Data on Host side
        // - Same/Similar memory physical types
        HOSTNS  = FS_TYPE_SYNC + FS_TYPE_DISTANT + FS_COM_UART,
        // Synchronised Symetric HST / COM Files
        // - Idem Symetric HST-COM Files excepted:
        // - Local header is assumed to be syncronised (through FS:SYNC)
        HOSTS   = FS_TYPE_DISTANT + FS_COM_UART,
        HOST    = FS_TYPE_DISTANT + FS_COM_UART, // Common Alias for FS_DISTANT_TYPE(t) usage
};


// D7AactP "trigger" TODO: is it really applicable ?
enum { LIST=0, READ, WRITE, WRITE_FLUSH};


// Utils
#define D7A_READ(fid,offset,len,data)    do {\
        d7a_errors_t status = d7a_alp_read_file((fid), (offset), (len), (uint8_t*)(data));\
        ASSERT(status >= 0, "Read file error %d (FID:%d OFF:%d LEN:%d)\r\n", status, (fid), (offset), (len));\
    } while(0);

d7a_errors_t d7a_fs_open(WriteFileFunction wf, ReadFileFunction rf);
d7a_errors_t d7a_fs_close(void);
void d7a_fs_new_pkt(d7a_com_rx_msg_t* pkt);
d7a_com_rx_msg_t* d7a_fs_wait_pkt( uint32_t millisec = osWaitForever );
void d7a_fs_notif_done(const uint8_t file_id, const uint8_t error);

#endif