#include "mbed.h"
#include "rtos.h"
#include "dbg.h"
#include "d7a_com.h"
#include "d7a_common.h"
#include "d7a_fs.h"
#include "d7a_sys.h"
#include "d7a.h"
#include "d7a_typedefs.h"

#if 0
    #define SYS_DPRINT(...)         DPRINT(__VA_ARGS__)
    #define SYS_DPRINT_DATA(...)    DPRINT_DATA(__VA_ARGS__)
    #define SYS_FPRINT(...)         FPRINT(__VA_ARGS__)
#else
    #define SYS_DPRINT(...);
    #define SYS_DPRINT_DATA(...);
    #define SYS_FPRINT(...);
#endif

static OS_Thread                         g_sys_thread(osPriorityHigh, 512, NULL);
static OS_Queue<d7a_com_rx_msg_t, 8>   g_sys_pkt_queue;
static OS_Queue<bool, 1>               g_sys_wait_pong;

void d7a_sys_thread();

d7a_errors_t d7a_sys_open(void)
{
    SYS_FPRINT("\r\n");
    
    osStatus err = g_sys_thread.start(d7a_sys_thread);
    ASSERT(err == osOK, "Failed to start d7a_sys_thread (err: %d)\r\n", err);
    
    return D7A_ERR_NONE;
}

d7a_errors_t d7a_sys_close(void)
{
    SYS_FPRINT("\r\n");

    g_sys_thread.terminate();
    
    return D7A_ERR_NONE;
}

void d7a_sys_new_pkt(d7a_com_rx_msg_t* pkt)
{
    SYS_FPRINT("\r\n");
    ASSERT(g_sys_pkt_queue.put(pkt) == osOK, "SYS queue full!\r\n");
}

d7a_com_rx_msg_t* d7a_sys_wait_pkt( uint32_t millisec )
{
    SYS_FPRINT("\r\n");
    osEvent evt = g_sys_pkt_queue.get(millisec);
    return (evt.status == osEventMessage)? (d7a_com_rx_msg_t*)evt.value.p : NULL;
}

void d7a_sys_tlev(uint8_t mod, uint16_t lev)
{
    SYS_FPRINT("\r\n");
    uint8_t buf[3];
    
    buf[0] = mod;
    buf[1] = (uint8_t)(lev << 8);
    buf[2] = (uint8_t)(lev & 0x00FF);
    
    d7a_com_dump(buf, 3, KAL_COM_FLOW_SYS_TLEV);
}

bool d7a_sys_ping_modem(uint32_t millisec)
{
    SYS_FPRINT("\r\n");
    uint8_t buf[5] = "PING";
    
    d7a_com_dump(buf, 4, KAL_COM_FLOW_SYS_PING);
    
    osEvent evt = g_sys_wait_pong.get(millisec);
    
    return (evt.status == osEventMessage)? true: false;
}

void d7a_sys_software_reset(void)
{
    SYS_FPRINT("\r\n");
    uint8_t buf[4] = "RST";

    d7a_com_dump(buf, 3, KAL_COM_FLOW_SYS_RST);
}

void d7a_sys_button(uint8_t button)
{
    SYS_FPRINT("\r\n");
    button += '0';
    
    d7a_com_dump(&button, 1, KAL_COM_FLOW_SYS_BUTTON);
}

void d7a_sys_xack(void)
{
    uint8_t buf[] = "X";
    d7a_com_dump(buf, 1, KAL_COM_FLOW_SYS_XACK);
}

void d7a_sys_thread()
{
    SYS_FPRINT("(id:0x%08x)\r\n", osThreadGetId());
    d7a_com_rx_msg_t* pkt;
    
    while (true)
    {
        pkt = d7a_sys_wait_pkt();
        ASSERT(pkt != NULL, "SYS NULL pkt\r\n");

        switch(pkt->id)
        {
            case KAL_COM_FLOW_SYS_RST:
                SYS_DPRINT("KAL_COM_FLOW_SYS_RST\r\n");
                FLUSH();
                NVIC_SystemReset();
                break;
            case KAL_COM_FLOW_SYS_BUTTON:
                SYS_DPRINT("KAL_COM_FLOW_SYS_BUTTON\r\n");
                break;
            case KAL_COM_FLOW_SYS_INFO:
                SYS_DPRINT("KAL_COM_FLOW_SYS_INFO\r\n");
                break;
            case KAL_COM_FLOW_SYS_CUP:
                SYS_DPRINT("KAL_COM_FLOW_SYS_CUP\r\n");
                break;
            case KAL_COM_FLOW_SYS_PING:
                SYS_DPRINT("KAL_COM_FLOW_SYS_PING\r\n");
                uint8_t buf[5] = "PONG";
                d7a_com_dump(buf, 4, KAL_COM_FLOW_SYS_PONG);
                break;
            case KAL_COM_FLOW_SYS_PONG:
                SYS_DPRINT("KAL_COM_FLOW_SYS_PONG\r\n");
                g_sys_wait_pong.put(NULL);
                break;
            case KAL_COM_FLOW_SYS_CFG:
                SYS_DPRINT("KAL_COM_FLOW_SYS_CFG\r\n");
                break;
            case KAL_COM_FLOW_SYS_TLEV:
                SYS_DPRINT("KAL_COM_FLOW_SYS_TLEV\r\n");
                break;
            case KAL_COM_FLOW_SYS_REDIR:
                SYS_DPRINT("KAL_COM_FLOW_SYS_REDIR\r\n");
                break;
            default:
                EPRINT("SYS Unknown Flow ID 0x%02X\r\n", pkt->id);
                break;
        }
        
        FREE(pkt);
    }
}
