#include "mbed.h"
#include "rtos.h"
#include "dbg.h"
#include "d7a_com.h"
#include "d7a_fs.h"
#include "d7a_modem.h"
#include "d7a_sys.h"
#include "d7a_common.h"
#include "d7a.h"
#include "d7a_alp.h"
#include "d7a_typedefs.h"

#if 1
    #define MODEM_DPRINT(...)       DPRINT(__VA_ARGS__)
    #define MODEM_DPRINT_DATA(...)  DPRINT_DATA(__VA_ARGS__)
    #define MODEM_FPRINT(...)       FPRINT(__VA_ARGS__)
#else
    #define MODEM_DPRINT(...);
    #define MODEM_DPRINT_DATA(...);
    #define MODEM_FPRINT(...);
#endif

#define MODEM_TO    (20000)

static NotifDoneFunction                g_modem_notif_done;
static bool                             g_modem_booted;

static MBED_PinName                     g_modem_reset_pin;

static OS_Thread                        g_modem_thread(osPriorityHigh, 512, NULL);
static OS_Queue<void, 8>                g_modem_ready;
static OS_Queue<void, 2>                g_modem_boot;
static OS_Queue<d7a_com_rx_msg_t, 8>    g_modem_pkt_queue;


void d7a_modem_thread();

d7a_errors_t d7a_modem_open(MBED_PinName reset_pin, NotifDoneFunction nd)
{
    MODEM_FPRINT("\r\n");
    
    d7a_errors_t err;
        
    osStatus error = g_modem_thread.start(d7a_modem_thread);
    ASSERT(error == osOK, "Failed to start d7a_modem_thread (err: %d)\r\n", error);
    
    g_modem_booted = false;
    g_modem_notif_done = nd;
    g_modem_reset_pin = reset_pin;

    if (g_modem_reset_pin != NC)
    {
        DigitalIn rst(g_modem_reset_pin);
    }

    err = d7a_modem_reset();
    
    return err;
}

d7a_errors_t d7a_modem_close(void)
{
    MODEM_FPRINT("\r\n");
        
    g_modem_thread.terminate();
    
    if (g_modem_reset_pin != NC)
    {   // Release reset
        DigitalIn rst(g_modem_reset_pin);
    }
    
    return D7A_ERR_NONE;
}

d7a_errors_t d7a_modem_wait_boot( uint32_t millisec )
{
    MODEM_FPRINT("(%d)\r\n", millisec);
    osEvent evt = g_modem_boot.get(millisec);
    return (evt.status == osEventMessage)? (d7a_errors_t)(int32_t)evt.value.p : D7A_ERR_CMD_TO;
}

d7a_errors_t d7a_modem_wait_ready( uint32_t millisec )
{
    MODEM_FPRINT("(%d)\r\n", millisec);
    osEvent evt = g_modem_ready.get(millisec);
    return (evt.status == osEventMessage)? (d7a_errors_t)(int32_t)evt.value.p : D7A_ERR_CMD_TO;
}

static void d7a_modem_soft_reset(void)
{
    MODEM_FPRINT("\r\n");
    IPRINT("MODEM Soft Reset.\r\n");
            
    // Clean buffer and queues
    d7a_com_restart();
    
    // Try software reset
    d7a_sys_software_reset();
}

static void d7a_modem_hard_reset(void)
{
    MODEM_FPRINT("\r\n");
        
    IPRINT("MODEM Hard Reset.\r\n");
    
    // Use hardware reset
    if (g_modem_reset_pin != NC)
    {   // Clear reset
        DigitalOut rst(g_modem_reset_pin, 0);
    }
    
    // Clean buffer and queues
    d7a_com_restart();
    OS_Thread  ::wait(100);
    
    if (g_modem_reset_pin != NC)
    {   // Release reset
        DigitalIn rst(g_modem_reset_pin);
    }
}

d7a_errors_t d7a_modem_reset(void)
{
    MODEM_FPRINT("\r\n");
    d7a_errors_t err = D7A_ERR_UNKNOWN;
    g_modem_booted = false;

    //d7a_modem_soft_reset();
    //err = d7a_modem_wait_boot(5000);
    
    if (err)
    {
        d7a_modem_hard_reset();
        err = d7a_modem_wait_boot(MODEM_TO);
    }
    ASSERT(!err, "MODEM BOOT err %d\r\n", err);
    g_modem_booted = true;
    
    err = d7a_modem_wait_ready(MODEM_TO);
    ASSERT(!err, "MODEM READY err %d\r\n", err);
    
    IPRINT("MODEM Ready.\r\n");
    
    return err;
}

void d7a_modem_new_pkt(d7a_com_rx_msg_t* pkt)
{
    MODEM_FPRINT("\r\n");
    ASSERT(g_modem_pkt_queue.put(pkt) == osOK, "MODEM queue full!\r\n");
}

d7a_com_rx_msg_t* d7a_modem_wait_pkt(uint32_t millisec)
{
    MODEM_FPRINT("(millisec:%d)\r\n", millisec);
    osEvent evt = g_modem_pkt_queue.get(millisec);
    return (evt.status == osEventMessage)? (d7a_com_rx_msg_t*)evt.value.p : NULL;
}

void d7a_modem_msg(uint8_t value, uint8_t* buf, uint8_t len)
{
    MODEM_FPRINT("(value:0x%02X, buf:0x%08x, len:%d)\r\n", value, buf, len);
    d7a_com_tx_msg_t msg;
    uint8_t val = value;
    msg.id = KAL_COM_FLOW_CMD;
    msg.pbuf = &val;
    msg.abuf = buf;
    msg.plen = 1;
    msg.alen = len;
    d7a_com_post_msg(&msg);
}

d7a_errors_t d7a_modem_register(register_file_param_t* file_infos)
{
    d7a_errors_t err;
    
    d7a_modem_msg(WM_CMD_REGISTER_FILE, (uint8_t*)file_infos, sizeof(register_file_param_t));
    err = d7a_modem_wait_ready(TO_FS);
            
    return err;
}

d7a_errors_t d7a_modem_notify(notify_file_param_t* notif)
{
    d7a_errors_t err;
    
    d7a_modem_msg(WM_CMD_NOTIFY_FILE, (uint8_t*)notif, sizeof(notify_file_param_t));
    err = d7a_modem_wait_ready(TO_FS);
            
    return err;
}

d7a_errors_t d7a_modem_start(void)
{
    d7a_errors_t err;
    
    MODEM_DPRINT("Start modem.\r\n");

    d7a_modem_msg(WM_CMD_D7A_STACK_START, NULL, 0);
    err = d7a_modem_wait_ready(TO_FS);
    
    WARNING(!err, "Start modem err %d\r\n", err);
        
    MODEM_DPRINT("Start done.\r\n");
    
    return err;
}

d7a_errors_t d7a_modem_stop(void)
{
    d7a_errors_t err;
    
    MODEM_DPRINT("Stop modem.\r\n");

    d7a_modem_msg(WM_CMD_D7A_STACK_STOP, NULL, 0);
    err = d7a_modem_wait_ready(TO_FS);
            
    MODEM_DPRINT("Stop done.\r\n");
    
    return err;
}

void d7a_modem_thread()
{
    MODEM_FPRINT("(id:0x%08x)\r\n", osThreadGetId());
    d7a_com_rx_msg_t* pkt;
    
    while (true)
    {
        pkt = d7a_modem_wait_pkt();
        ASSERT(pkt != NULL, "MODEM NULL pkt\r\n");

        switch(pkt->id)
        {
            case KAL_COM_FLOW_CMD:
                uint8_t cmd = pkt->buffer[0];
                if (cmd == WM_OK)
                {
                    MODEM_DPRINT("Modem ready\r\n");
                    g_modem_ready.put((void*)D7A_ERR_NONE);
                }
                else if (cmd == WM_BOOT)
                {
                    boot_status_t* bs = (boot_status_t*)&(pkt->buffer[1]);
                    WARNING(false, "Modem booted CAUSE:%d NB_BOOT:%d\r\n", bs->bf.cause, bs->bf.nb_boot);
                    
                    if (g_modem_booted == true)
                    {
                        // Do something?
                    }
                    else
                    {
                        g_modem_boot.put(NULL);
                    }
                }
                else if (cmd == WM_ERROR)
                {
                    int8_t err = pkt->buffer[1];
                    int8_t ret;
                    
                    WARNING(false, "Modem cmd error %d\r\n", err);
                    switch (err)
                    {
                        case WM_ERR_NOT_READY:          ret = D7A_ERR_NOT_READY; break;
                        case WM_ERR_COM_LINK:           ret = D7A_ERR_COM_LINK; break;
                        case WM_ERR_ILLEGAL_FID:        ret = D7A_ERR_ILLEGAL_FID; break;
                        case WM_ERR_ILLEGAL_FILE_DEF:   ret = D7A_ERR_ILLEGAL_FILE_DEF; break;
                        case WM_ERR_UNKNOWN:            ret = D7A_ERR_UNKNOWN; break;
                        default:                        ret = D7A_ERR_NONE; break;
                    }
                    
                    g_modem_ready.put((void*)ret);
                }
                else if (cmd == WM_NOTIF_DONE)
                {
                    if (g_modem_notif_done != NULL)
                    {
                        g_modem_notif_done(pkt->buffer[1], pkt->buffer[2]);
                    }
                }
                else
                {
                    EPRINT("MODEM Unknown cmd %d\r\n", cmd);
                }
                break;
            default:
                EPRINT("MODEM Unknown Flow ID 0x%02X\r\n", pkt->id);
                break;
        }
        
        FREE(pkt);
    }
}



