#include "mbed.h"
#include <stdio.h>
#include "Speaker.h"
#include "PinDetect.h"
#include "BuzzyGraphics.h"
#include "uLCD_4DGL.h"

#include "Buzzy.h"
#include "Ghosts.h"

DigitalOut myled1(LED1);
DigitalOut myled2(LED2);
DigitalOut myled3(LED3);
DigitalOut myled4(LED4);
////////////////////////////////////////
// Setup instance of LCD display
uLCD_4DGL guLCD(p9, p10, p11); // serial tx, serial rx, reset pin;
////////////////////////////////////////
// Setup instances of push button pins
PinDetect gPB_left(p21); 
PinDetect gPB_right(p22); 
PinDetect gPB_up(p23);
PinDetect gPB_down(p24);
// Create Buzzy and Ghosts
Buzzy gBuzzy;
Ghosts gGhosts[NUM_GHOSTS];
// Variable indicates if game is paused or running
int gGameState = GAME_PAUSED;
// Declare and initialize the speaker
Speaker gSpeakerOut(p26);

////////////////////////////////////////////////////////
// This is the maze that changes as the game is played
char gDynaMaze[MAZE_NUM_ROW][MAZE_NUM_COL]; 

//////////////////////////
// Prototype functions
void DrawMaze();

//////////////////////////////////////////////////////////////////////
// Interrupt routine
// used to output next analog sample whenever a timer interrupt occurs
void Sample_timer_interrupt(void)
{
    // Call speaker function to play next value
}
//---------------------------------------------------------------------------------------------------
// Callback routine is interrupt activated by a debounced pb_left hit
void pb_left_hit_callback (void)
{
    // Update game state and tell Buzzy to go left
    gBuzzy.SetDesiredDirectionToMove(Sprite::LEFT_DIR);
}
//---------------------------------------------------------------------------------------------------
// Callback routine is interrupt activated by a debounced pb_right hit
void pb_right_hit_callback (void)
{
    // Update game state and tell Buzzy to go right
    gBuzzy.SetDesiredDirectionToMove(Sprite::RIGHT_DIR);
}
//---------------------------------------------------------------------------------------------------
// Callback routine is interrupt activated by a debounced pb_up hit
void pb_up_hit_callback (void)
{
    // Update game state and tell Buzzy to go up
    gBuzzy.SetDesiredDirectionToMove(Sprite::UP_DIR);
}
//---------------------------------------------------------------------------------------------------
// Callback routine is interrupt activated by a debounced pb_down hit
void pb_down_hit_callback (void)
{
    // Update game state and tell Buzzy to go down
    gBuzzy.SetDesiredDirectionToMove(Sprite::DOWN_DIR);
}
//---------------------------------------------------------------------------------------------------
int main()
{
        
    // Setup push buttons
    gPB_left.mode(PullUp);
    gPB_right.mode(PullUp);
    gPB_up.mode(PullUp);
    gPB_down.mode(PullUp);
    // Delay for initial pullup to take effect
    wait(.01);
    // Setup Interrupt callback functions for a pb hit
    gPB_left.attach_deasserted(&pb_left_hit_callback);
    gPB_right.attach_deasserted(&pb_right_hit_callback);
    gPB_up.attach_deasserted(&pb_up_hit_callback);
    gPB_down.attach_deasserted(&pb_down_hit_callback);
    // Setup speaker
    gSpeakerOut.period(1.0/200000.0);  
    // set up a timer to be used for sample rate interrupts
    Ticker Sample_Period;      
    Sample_Period.attach(&Sample_timer_interrupt, 1.0/(20000.0));

    //Setup LCD display
    guLCD.display_control(PORTRAIT);
    guLCD.background_color(BLACK);
    guLCD.cls();
    guLCD.baudrate(BAUD_3000000); //jack up baud rate to max for fast display
    wait(1.0);
  
    // Start sampling pb inputs using interrupts
    gPB_left.setSampleFrequency();
    gPB_right.setSampleFrequency();
    gPB_up.setSampleFrequency();
    gPB_down.setSampleFrequency();
    //////////////////////////////////////
    // Everything should be ready to start playing the game.
    
    while(1)
    {
        guLCD.cls();
        // Ask the user if they would like to play a game. 
        guLCD.printf("Would you like to play??");
        // Wait for a button to be pressed
        while(gPB_left == 1){
            wait(1);
        }
        guLCD.cls();
        // Initialize needed parts
        for (int ii = 0 ; ii < MAZE_NUM_ROW ; ii++)
        {
            for (int jj = 0 ; jj < MAZE_NUM_COL ; jj++)
            {
                gDynaMaze[ii][jj] = gCnstMaze[ii][jj];
            }
        }
        // Reset the Ghosts and Buzzy states
        gBuzzy.SetDesiredDirectionToMove(Sprite::RIGHT_DIR);
        // Start up new game
        gBuzzy.honey = 0;
        
        // Play introduction sounds while drawing the Maze
        gSpeakerOut.SwitchSound(Speaker::BEGIN);
        DrawMaze();  
        // Start Game loop
        gGameState = GAME_RUNNING;
        while (gGameState == GAME_RUNNING)
        {
            // Move Buzzy and any active ghosts
            gBuzzy.Move();
            // Check to see if Buzzy has eaten all the honey drops
            if(gBuzzy.honey == 10){
                gGameState = 0;
            }
            // Break out of loop if all honey drops are consumed
            
            wait(0.001);
        }
        
        guLCD.cls();
        guLCD.printf("GAME OVER!!");
        wait(2);

        gGameState = GAME_PAUSED;
    }

} //end main