#include <stdint.h>
extern void printhex(uint16_t val);
extern void serout(uint8_t value);

#define NULL (void *) 0
#define RAM_SIZE 1536

//6502 defines
//#define UNDOCUMENTED //when this is defined, undocumented opcodes are handled.
//otherwise, they're simply treated as NOPs.

//#define USE_TIMING //slower, but allows you to specify number of cycles to run for exec6502
//rather than simply a number of instructions. also uses a little more
//program memory when enabled.

#define FLAG_CARRY     0x01
#define FLAG_ZERO      0x02
#define FLAG_INTERRUPT 0x04
#define FLAG_DECIMAL   0x08
#define FLAG_BREAK     0x10
#define FLAG_CONSTANT  0x20
#define FLAG_OVERFLOW  0x40
#define FLAG_SIGN      0x80

#define BASE_STACK     0x100

#define saveaccum(n) a = (uint8_t)((n) & 0x00FF)

//flag modifier macros
#define setcarry() cpustatus |= FLAG_CARRY
#define clearcarry() cpustatus &= (~FLAG_CARRY)
#define setzero() cpustatus |= FLAG_ZERO
#define clearzero() cpustatus &= (~FLAG_ZERO)
#define setinterrupt() cpustatus |= FLAG_INTERRUPT
#define clearinterrupt() cpustatus &= (~FLAG_INTERRUPT)
#define setdecimal() cpustatus |= FLAG_DECIMAL
#define cleardecimal() cpustatus &= (~FLAG_DECIMAL)
#define setoverflow() cpustatus |= FLAG_OVERFLOW
#define clearoverflow() cpustatus &= (~FLAG_OVERFLOW)
#define setsign() cpustatus |= FLAG_SIGN
#define clearsign() cpustatus &= (~FLAG_SIGN)


//flag calculation macros
#define zerocalc(n) { if ((n) & 0x00FF) clearzero(); else setzero(); }

#define signcalc(n) { if ((n) & 0x0080) setsign(); else clearsign(); }

#define carrycalc(n) { if ((n) & 0xFF00) setcarry(); else clearcarry(); }

#define overflowcalc(n, m, o) { if (((n) ^ (uint16_t)(m)) & ((n) ^ (o)) & 0x0080) setoverflow(); else clearoverflow(); }


//6502 CPU registers
uint16_t pc;
uint8_t sp, a, x, y, cpustatus;


//helper variables
uint32_t instructions = 0; //keep track of total instructions executed
int32_t clockticks6502 = 0, clockgoal6502 = 0;
uint16_t oldpc, ea, reladdr, value, result;
uint8_t opcode, oldcpustatus, useaccum;

uint8_t RAM[RAM_SIZE];
uint8_t BIOS[10496] = {
    0xA0,0x04,0xB9,0xF6,0xE0,0x99,0x00,0x02,0x88,0x10,0xF7,
    0xA2,0xFF,0x86,0x88,0x9A,0xA9,0x4C,0x85,0xA1,0xA2,0x1C,
    0xBD,0xFA,0xE0,0x95,0xBB,0xCA,0xD0,0xF8,0xA2,0x12,0xBD,
    0x17,0xE1,0x95,0x00,0xCA,0x10,0xF8,0xA9,0x00,0x85,0xDC,
    0x85,0xDF,0x85,0xB2,0x85,0x67,0xA9,0x0E,0x85,0x64,0xA9,
    0x03,0x85,0xA0,0xA2,0x68,0x86,0x65,0x20,0x92,0xC8,0xA9,
    0x2A,0xA0,0xE1,0x20,0xD3,0xC8,0x20,0x40,0xC2,0x86,0xC3,
    0x84,0xC4,0x20,0xC2,0x00,0xD0,0x1F,0xA0,0x00,0xE6,0x11,
    0xD0,0x08,0xE6,0x12,0xA5,0x12,0xC9,0xC0,0xF0,0x1D,0xA9,
    0x55,0x91,0x11,0xD1,0x11,0xD0,0x15,0x0A,0x91,0x11,0xD1,
    0x11,0xF0,0xE5,0xD0,0x0C,0x20,0xA6,0xD9,0xA5,0xAC,0xC9,
    0x98,0xB0,0xA2,0x20,0xF7,0xD4,0xA5,0x11,0xA4,0x12,0xC0,
    0x01,0x90,0x97,0x85,0x85,0x84,0x86,0x85,0x81,0x84,0x82,
    0xA0,0x00,0xA2,0x03,0x84,0x79,0x86,0x7A,0x98,0x91,0x79,
    0xE6,0x79,0x20,0x92,0xC8,0x20,0x50,0xC3,0xA5,0x85,0x38,
    0xE5,0x79,0xAA,0xA5,0x86,0xE5,0x7A,0x20,0x82,0xDA,0xA9,
    0x39,0xA0,0xE1,0x20,0xD3,0xC8,0xA9,0x5A,0xA0,0xC1,0x85,
    0x01,0x84,0x02,0x6C,0x01,0x00,0x20,0x0B,0xC1,0x85,0x7F,
    0x84,0x80,0x38,0xA5,0xA6,0xE5,0xAA,0xA8,0xA5,0xA7,0xE5,
    0xAB,0xAA,0xE8,0x98,0xF0,0x24,0x38,0x49,0xFF,0x65,0xA6,
    0x85,0xA6,0xB0,0x03,0xC6,0xA7,0x38,0x98,0x49,0xFF,0x65,
    0xA4,0x85,0xA4,0xB0,0x08,0xC6,0xA5,0x90,0x04,0xB1,0xA6,
    0x91,0xA4,0x88,0xD0,0xF9,0xB1,0xA6,0x91,0xA4,0xC6,0xA7,
    0xC6,0xA5,0xCA,0xD0,0xF2,0x60,0x85,0x78,0xBA,0xE4,0x78,
    0x90,0x30,0x60,0xC4,0x82,0x90,0x2A,0xD0,0x04,0xC5,0x81,
    0x90,0x24,0x48,0xA2,0x08,0x98,0x48,0xB5,0xA3,0xCA,0x10,
    0xFA,0x20,0xDF,0xD1,0xA2,0x00,0x68,0x95,0xA4,0xE8,0xE0,
    0x08,0x30,0xF8,0x68,0xA8,0x68,0xC4,0x82,0x90,0x06,0xD0,
    0x05,0xC5,0x81,0xB0,0x01,0x60,0xA2,0x0C,0x20,0x92,0xC8,
    0xBD,0xA9,0xE6,0xBC,0xAA,0xE6,0x20,0xD3,0xC8,0x20,0x89,
    0xC3,0xA9,0xE6,0xA0,0xE7,0x20,0xD3,0xC8,0xA4,0x88,0xC8,
    0xF0,0x03,0x20,0x77,0xDA,0xA9,0x00,0x85,0xDF,0x85,0xDC,
    0xA9,0xF7,0xA0,0xE7,0x20,0xD3,0xC8,0x20,0x4D,0xC2,0x86,
    0xC3,0x84,0xC4,0x20,0xC2,0x00,0xF0,0xF4,0xA2,0xFF,0x86,
    0x88,0x90,0x06,0x20,0x7E,0xC2,0x4C,0xF3,0xC4,0x20,0x55,
    0xC7,0x20,0x7E,0xC2,0x84,0x5D,0x20,0x24,0xC3,0x90,0x44,
    0xA0,0x01,0xB1,0xAA,0x85,0x72,0xA5,0x7B,0x85,0x71,0xA5,
    0xAB,0x85,0x74,0xA5,0xAA,0x88,0xF1,0xAA,0x18,0x65,0x7B,
    0x85,0x7B,0x85,0x73,0xA5,0x7C,0x69,0xFF,0x85,0x7C,0xE5,
    0xAB,0xAA,0x38,0xA5,0xAA,0xE5,0x7B,0xA8,0xB0,0x03,0xE8,
    0xC6,0x74,0x18,0x65,0x71,0x90,0x03,0xC6,0x72,0x18,0xB1,
    0x71,0x91,0x73,0xC8,0xD0,0xF9,0xE6,0x72,0xE6,0x74,0xCA,
    0xD0,0xF2,0xAD,0x21,0x02,0xF0,0x3F,0xA5,0x85,0xA4,0x86,
    0x85,0x81,0x84,0x82,0xA5,0x7B,0x85,0xA6,0xA4,0x7C,0x84,
    0xA7,0x65,0x5D,0x90,0x01,0xC8,0x85,0xA4,0x84,0xA5,0x20,
    0xC1,0xC0,0xA5,0x7F,0xA4,0x80,0x85,0x7B,0x84,0x7C,0xA4,
    0x5D,0x88,0xB9,0x1D,0x02,0x91,0xAA,0x88,0xC0,0x03,0xD0,
    0xF6,0xA5,0x12,0x91,0xAA,0x88,0xA5,0x11,0x91,0xAA,0x88,
    0xA9,0xFF,0x91,0xAA,0x20,0x65,0xC3,0xA6,0x79,0xA5,0x7A,
    0xA0,0x01,0x86,0x71,0x85,0x72,0xB1,0x71,0xF0,0x18,0xA0,
    0x04,0xC8,0xB1,0x71,0xD0,0xFB,0x38,0x98,0x65,0x71,0xAA,
    0xA0,0x00,0x91,0x71,0x98,0x65,0x72,0xC8,0x91,0x71,0x90,
    0xE0,0x4C,0x67,0xC1,0x20,0xEB,0xC8,0x20,0xE8,0xC8,0xD0,
    0x05,0x20,0xED,0xC8,0xCA,0x2C,0xA2,0x00,0x20,0xEA,0xE0,
    0x90,0xFB,0xF0,0xF9,0xC9,0x07,0xF0,0x10,0xC9,0x0D,0xF0,
    0x19,0xE0,0x00,0xD0,0x04,0xC9,0x21,0x90,0xE9,0xC9,0x08,
    0xF0,0xDE,0xE0,0x47,0xB0,0x0C,0x9D,0x21,0x02,0xE8,0x20,
    0xED,0xC8,0xD0,0xD8,0x4C,0x89,0xC8,0xA9,0x07,0xD0,0xF4,
    0xA0,0xFF,0x38,0xA5,0xC3,0xE9,0x21,0xAA,0x86,0x60,0xBD,
    0x21,0x02,0xF0,0x51,0xC9,0x5F,0xB0,0x4D,0xC9,0x3C,0xB0,
    0x0E,0xC9,0x30,0xB0,0x45,0x85,0x5C,0xC9,0x22,0xF0,0x61,
    0xC9,0x2A,0x90,0x3B,0x24,0x60,0x70,0x37,0x86,0x78,0x84,
    0xBA,0xA0,0x10,0x84,0x73,0xA0,0xE3,0x84,0x74,0xA0,0x00,
    0xD1,0x73,0xF0,0x05,0x90,0x21,0xC8,0xD0,0xF7,0x98,0x0A,
    0xAA,0xBD,0x2E,0xE3,0x85,0x73,0xBD,0x2F,0xE3,0x85,0x74,
    0xA0,0xFF,0xA6,0x78,0xC8,0xB1,0x73,0x30,0x08,0xE8,0xDD,
    0x21,0x02,0xF0,0xF5,0xD0,0x2B,0xA4,0xBA,0xE8,0xC8,0x99,
    0x21,0x02,0xC9,0x00,0xF0,0x32,0xE9,0x3A,0xF0,0x04,0xC9,
    0x49,0xD0,0x02,0x85,0x60,0x49,0x57,0xD0,0x93,0x85,0x5C,
    0xBD,0x21,0x02,0xF0,0xE2,0xC5,0x5C,0xF0,0xDE,0xC8,0x99,
    0x21,0x02,0xE8,0xD0,0xF0,0xA6,0x78,0xB1,0x73,0x08,0xC8,
    0x28,0x10,0xF9,0xB1,0x73,0xD0,0xBE,0xBD,0x21,0x02,0x10,
    0xC3,0xC8,0xC8,0x99,0x21,0x02,0xC8,0xC8,0xC8,0xC6,0xC3,
    0x60,0xA5,0x79,0xA6,0x7A,0xA0,0x01,0x85,0xAA,0x86,0xAB,
    0xB1,0xAA,0xF0,0x1A,0xA0,0x03,0xB1,0xAA,0x88,0xC5,0x12,
    0xD0,0x04,0xB1,0xAA,0xC5,0x11,0xB0,0x09,0x88,0xB1,0xAA,
    0xAA,0x88,0xB1,0xAA,0x90,0xDE,0xF0,0x01,0x18,0x60,0xD0,
    0xFD,0xA9,0x00,0xA8,0x91,0x79,0xC8,0x91,0x79,0x18,0xA5,
    0x79,0x69,0x02,0x85,0x7B,0xA5,0x7A,0x69,0x00,0x85,0x7C,
    0x18,0xA5,0x79,0x69,0xFF,0x85,0xC3,0xA5,0x7A,0x69,0xFF,
    0x85,0xC4,0xA5,0x85,0xA4,0x86,0x85,0x81,0x84,0x82,0xA5,
    0x7B,0xA4,0x7C,0x85,0x7D,0x84,0x7E,0x85,0x7F,0x84,0x80,
    0x20,0x44,0xC5,0xA2,0x68,0x86,0x65,0x68,0xAA,0x68,0x8E,
    0xFE,0x01,0x8D,0xFF,0x01,0xA2,0xFD,0x9A,0xA9,0x00,0x85,
    0x8C,0x85,0x61,0x60,0xF0,0xD0,0x60,0x90,0x06,0xF0,0x04,
    0xC9,0xB7,0xD0,0xF4,0x20,0x55,0xC7,0x20,0x24,0xC3,0x20,
    0xC2,0x00,0xF0,0x0C,0xC9,0xB7,0xD0,0x93,0x20,0xBC,0x00,
    0x20,0x55,0xC7,0xD0,0x8B,0xA5,0x11,0x05,0x12,0xD0,0x06,
    0xA9,0xFF,0x85,0x11,0x85,0x12,0xA0,0x01,0x84,0x60,0x20,
    0x92,0xC8,0xB1,0xAA,0xF0,0x3E,0x20,0x14,0xC5,0xC8,0xB1,
    0xAA,0xAA,0xC8,0xB1,0xAA,0xC5,0x12,0xD0,0x04,0xE4,0x11,
    0xF0,0x02,0xB0,0x2A,0x84,0x97,0x20,0x82,0xDA,0xA9,0x20,
    0xA4,0x97,0x29,0x7F,0x20,0xED,0xC8,0xC9,0x22,0xD0,0x06,
    0xA5,0x60,0x49,0xFF,0x85,0x60,0xC8,0xB1,0xAA,0xD0,0x0E,
    0xA8,0xB1,0xAA,0xAA,0xC8,0xB1,0xAA,0x86,0xAA,0x85,0xAB,
    0xD0,0xB7,0x60,0x10,0xDE,0x24,0x60,0x30,0xDA,0xA2,0xE5,
    0x0A,0x0A,0x90,0x02,0xE8,0x18,0x69,0x11,0x90,0x01,0xE8,
    0x85,0x73,0x86,0x74,0x84,0x97,0xA0,0x00,0xB1,0x73,0xAA,
    0xC8,0xB1,0x73,0xCA,0xF0,0xB8,0x20,0xED,0xC8,0xC8,0xB1,
    0x73,0x48,0xC8,0xB1,0x73,0xA0,0x00,0x85,0x74,0x68,0x85,
    0x73,0xB1,0x73,0xCA,0xF0,0xA2,0x20,0xED,0xC8,0xC8,0xD0,
    0xF5,0xA9,0x80,0x85,0x61,0x20,0xB4,0xC7,0x68,0x68,0xA9,
    0x10,0x20,0x03,0xC1,0x20,0x9D,0xC6,0x18,0x98,0x65,0xC3,
    0x48,0xA5,0xC4,0x69,0x00,0x48,0xA5,0x88,0x48,0xA5,0x87,
    0x48,0xA9,0xAD,0x20,0xF1,0xCB,0x20,0xD0,0xCA,0x20,0xCD,
    0xCA,0xA5,0xB0,0x09,0x7F,0x25,0xAD,0x85,0xAD,0xA9,0x97,
    0xA0,0xC4,0x85,0x71,0x84,0x72,0x4C,0x84,0xCB,0xA9,0xE0,
    0xA0,0xE1,0x20,0x7D,0xD8,0x20,0xC2,0x00,0xC9,0xB2,0xD0,
    0x06,0x20,0xBC,0x00,0x20,0xCD,0xCA,0x20,0xE9,0xD8,0x85,
    0xB0,0x20,0x79,0xCB,0xA5,0x98,0x48,0xA5,0x97,0x48,0xA9,
    0x81,0x48,0x20,0x14,0xC5,0xA5,0xC3,0xA4,0xC4,0xA6,0x88,
    0xE8,0xF0,0x04,0x85,0x8B,0x84,0x8C,0xA0,0x00,0xB1,0xC3,
    0xF0,0x07,0xC9,0x3A,0xF0,0x1D,0x4C,0x02,0xCC,0xA0,0x02,
    0xB1,0xC3,0x18,0xF0,0x56,0xC8,0xB1,0xC3,0x85,0x87,0xC8,
    0xB1,0xC3,0x85,0x88,0x98,0x65,0xC3,0x85,0xC3,0x90,0x02,
    0xE6,0xC4,0x20,0xBC,0x00,0x20,0xFC,0xC4,0x4C,0xBC,0xC4,
    0xF0,0x54,0x0A,0xB0,0x03,0x4C,0xB4,0xC7,0xC9,0x56,0xB0,
    0xCE,0xA8,0xB9,0x08,0xE2,0x48,0xB9,0x07,0xE2,0x48,0x4C,
    0xBC,0x00,0x6C,0x03,0x02,0xC9,0x03,0xB0,0x01,0x18,0xD0,
    0x67,0xA5,0xC4,0x49,0x02,0xF0,0x10,0x49,0x02,0xA4,0xC3,
    0x84,0x8B,0x85,0x8C,0xA5,0x87,0xA4,0x88,0x85,0x89,0x84,
    0x8A,0x68,0x68,0x90,0x07,0xA9,0xDE,0xA0,0xE7,0x4C,0x4F,
    0xC1,0x4C,0x5A,0xC1,0xD0,0x0F,0x38,0xA5,0x79,0xE9,0x01,
    0xA4,0x7A,0xB0,0x01,0x88,0x85,0x8F,0x84,0x90,0x60,0x20,
    0x55,0xC7,0x20,0xA0,0xC6,0xA5,0x88,0xC5,0x12,0xB0,0x0B,
    0x98,0x38,0x65,0xC3,0xA6,0xC4,0x90,0x07,0xE8,0xB0,0x04,
    0xA5,0x79,0xA6,0x7A,0x20,0x28,0xC3,0xB0,0x03,0x4C,0x72,
    0xC6,0xA5,0xAA,0xE9,0x01,0xA4,0xAB,0xB0,0xD0,0x90,0xCD,
    0x20,0x8C,0xD4,0x86,0x0D,0x60,0xD0,0xFD,0xA4,0x8C,0xD0,
    0x05,0xA2,0x1E,0x4C,0x3C,0xC1,0xA9,0x93,0x20,0x14,0xDF,
    0xA9,0x93,0x20,0x17,0xDF,0x84,0xC4,0xA5,0x8B,0x85,0xC3,
    0xA5,0x89,0xA4,0x8A,0x85,0x87,0x84,0x88,0x60,0xD0,0x03,
    0x4C,0x65,0xC3,0x20,0x72,0xC3,0xF0,0x2E,0xA9,0x05,0x20,
    0x03,0xC1,0xA5,0xC4,0x48,0xA5,0xC3,0x48,0xA5,0x88,0x48,
    0xA5,0x87,0x48,0xA9,0x9D,0x48,0x20,0xC2,0x00,0x4C,0xBC,
    0xC4,0xA9,0x05,0x20,0x03,0xC1,0xA5,0xC4,0x48,0xA5,0xC3,
    0x48,0xA5,0x88,0x48,0xA5,0x87,0x48,0xA9,0x8D,0x48,0x20,
    0xC2,0x00,0x20,0xEB,0xC5,0x4C,0xBC,0xC4,0x20,0x55,0xC7,
    0x20,0xA0,0xC6,0xA5,0x88,0xC5,0x12,0xB0,0x0B,0x98,0x38,
    0x65,0xC3,0xA6,0xC4,0x90,0x07,0xE8,0xB0,0x04,0xA5,0x79,
    0xA6,0x7A,0x20,0x28,0xC3,0x90,0x67,0xA5,0xAA,0xE9,0x01,
    0x85,0xC3,0xA5,0xAB,0xE9,0x00,0x85,0xC4,0x60,0xA2,0x22,
    0x4C,0x3C,0xC1,0xA8,0xBA,0xBD,0x03,0x01,0xC9,0x9D,0xD0,
    0xF2,0xE8,0xE8,0x9A,0x98,0xF0,0x20,0xC9,0x3A,0xF0,0x1C,
    0xE9,0xB3,0xAA,0xF0,0x04,0xCA,0xD0,0x62,0xCA,0x86,0x98,
    0x20,0xBC,0x00,0x20,0xE1,0xCA,0xA5,0xAC,0xF0,0x02,0xA9,
    0xFF,0xBA,0x45,0x98,0xD0,0x1A,0xBD,0x02,0x01,0x85,0x87,
    0xBD,0x03,0x01,0x85,0x88,0xBD,0x04,0x01,0x85,0xC3,0xBD,
    0x05,0x01,0x85,0xC4,0x20,0xC2,0x00,0x4C,0xBC,0xC4,0xE8,
    0xE8,0xE8,0xE8,0xE8,0x9A,0x4C,0x8C,0xC6,0xA2,0x04,0x2C,
    0xA2,0x0E,0x4C,0x3C,0xC1,0xD0,0x9E,0x68,0x68,0x68,0xC9,
    0x8D,0xD0,0xEF,0x68,0x85,0x87,0x68,0x85,0x88,0x68,0x85,
    0xC3,0x68,0x85,0xC4,0x20,0x9D,0xC6,0x98,0x18,0x65,0xC3,
    0x85,0xC3,0x90,0x02,0xE6,0xC4,0x60,0x4C,0x02,0xCC,0xA2,
    0x3A,0x2C,0xA2,0x00,0xA0,0x00,0x84,0x5C,0x8A,0x45,0x5C,
    0x85,0x5C,0xB1,0xC3,0xF0,0xEA,0xC5,0x5C,0xF0,0xE6,0xC8,
    0xC9,0x22,0xD0,0xF3,0xF0,0xEC,0x20,0xE1,0xCA,0x20,0xC2,
    0x00,0xC9,0xB0,0xF0,0x11,0xC9,0x89,0xD0,0xD2,0xA6,0xC3,
    0xA4,0xC4,0x20,0xBC,0x00,0xB0,0xC9,0x86,0xC3,0x84,0xC4,
    0xA5,0xAC,0xF0,0x1B,0x20,0xBC,0x00,0xB0,0x03,0x4C,0xEB,
    0xC5,0xC9,0x90,0xD0,0x03,0x4C,0xFE,0xC4,0x20,0xFC,0xC4,
    0xA0,0x00,0xB1,0xC3,0xC9,0xAC,0xF0,0x99,0x60,0xA0,0x00,
    0xA2,0x01,0xC8,0xB1,0xC3,0xF0,0x0F,0xC9,0x8B,0xD0,0x03,
    0xE8,0xD0,0xF4,0xC9,0xAC,0xD0,0xF0,0xCA,0xD0,0xED,0xC8,
    0x98,0x18,0x65,0xC3,0x85,0xC3,0x90,0x02,0xE6,0xC4,0x20,
    0xC2,0x00,0x90,0xC3,0x4C,0xFC,0xC4,0x20,0xA0,0xC6,0x4C,
    0x8F,0xC6,0x4C,0x02,0xCC,0xC9,0xA9,0xD0,0x03,0x4C,0x38,
    0xDF,0xC9,0xAA,0xD0,0x03,0x4C,0x3C,0xDF,0x20,0x8C,0xD4,
    0x48,0xC9,0x8D,0xF0,0x04,0xC9,0x89,0xD0,0xE3,0xC6,0xAF,
    0xD0,0x04,0x68,0x4C,0xFE,0xC4,0x20,0xBC,0x00,0x20,0x55,
    0xC7,0xC9,0x2C,0xF0,0xEE,0x68,0x60,0xA2,0x00,0x86,0x11,
    0x86,0x12,0xB0,0xF7,0xE0,0x19,0xA8,0xB0,0xDD,0xE9,0x2F,
    0xA8,0xA5,0x11,0x0A,0x26,0x12,0x0A,0x26,0x12,0x65,0x11,
    0x85,0x11,0x8A,0x65,0x12,0x06,0x11,0x2A,0xAA,0x98,0x65,
    0x11,0x85,0x11,0x90,0x01,0xE8,0x20,0xBC,0x00,0x4C,0x59,
    0xC7,0xA9,0xE4,0x2C,0xA9,0xE0,0x48,0x20,0xAA,0xCD,0xA6,
    0x5F,0x30,0x1E,0x85,0x97,0x84,0x98,0x20,0x7D,0xD8,0x68,
    0x48,0xA0,0xE1,0x20,0xBE,0xD5,0x20,0xA3,0xD8,0x20,0xC2,
    0x00,0xC9,0x2C,0xD0,0xA8,0x20,0xBC,0x00,0x4C,0x8C,0xC7,
    0x4C,0xDC,0xCA,0x20,0xAA,0xCD,0x85,0x97,0x84,0x98,0xA9,
    0xC1,0x20,0xF1,0xCB,0xA5,0x5F,0x48,0x20,0xE1,0xCA,0x68,
    0x2A,0x20,0xD3,0xCA,0xD0,0x03,0x4C,0xA3,0xD8,0xA0,0x02,
    0xB1,0xAE,0xC5,0x82,0x90,0x17,0xD0,0x07,0x88,0xB1,0xAE,
    0xC5,0x81,0x90,0x0E,0xA4,0xAF,0xC4,0x7C,0x90,0x08,0xD0,
    0x0D,0xA5,0xAE,0xC5,0x7B,0xB0,0x07,0xA5,0xAE,0xA4,0xAF,
    0x4C,0x0C,0xC8,0xA0,0x00,0xB1,0xAE,0x20,0x32,0xD1,0xA5,
    0x9E,0xA4,0x9F,0x85,0xB8,0x84,0xB9,0x20,0x11,0xD3,0xA9,
    0xAC,0xA0,0x00,0x85,0x9E,0x84,0x9F,0x20,0x73,0xD3,0xA0,
    0x00,0xB1,0x9E,0x91,0x97,0xC8,0xB1,0x9E,0x91,0x97,0xC8,
    0xB1,0x9E,0x91,0x97,0x60,0x20,0xAA,0xCD,0x85,0x97,0x84,
    0x98,0x20,0x00,0xDF,0xA6,0x5F,0x30,0x07,0xA8,0x20,0x66,
    0xD0,0x4C,0xA3,0xD8,0x48,0xA9,0x01,0xB0,0x01,0x68,0x20,
    0x3A,0xD1,0xF0,0x05,0x68,0xA0,0x00,0x91,0xAD,0x20,0x85,
    0xD1,0x4C,0xD0,0xC7,0x20,0xD6,0xC8,0x20,0xC2,0x00,0xF0,
    0x3B,0xC9,0xAB,0xF0,0x56,0xC9,0xAF,0xF0,0x52,0xC9,0x2C,
    0xF0,0x38,0xC9,0x3B,0xF0,0x66,0x20,0xE1,0xCA,0x24,0x5F,
    0x30,0xE1,0x20,0x95,0xDA,0x20,0x44,0xD1,0xA0,0x00,0xA5,
    0x0F,0xF0,0x0A,0x38,0xE5,0x0E,0xF1,0xAE,0xB0,0x03,0x20,
    0x92,0xC8,0x20,0xD6,0xC8,0xF0,0xC9,0xA9,0x00,0x9D,0x21,
    0x02,0xA2,0x21,0xA0,0x02,0xA9,0x0D,0x20,0xED,0xC8,0xA9,
    0x0A,0xD0,0x52,0xA5,0x0E,0xC5,0x10,0x90,0x05,0x20,0x92,
    0xC8,0xD0,0x27,0x38,0xE5,0x64,0xB0,0xFC,0x49,0xFF,0x69,
    0x01,0xD0,0x12,0x48,0x20,0x89,0xD4,0xC9,0x29,0xD0,0x7B,
    0x68,0xC9,0xAB,0xD0,0x06,0x8A,0xE5,0x0E,0x90,0x0A,0xAA,
    0x8A,0xF0,0x06,0x20,0xE8,0xC8,0xCA,0xD0,0xFA,0x20,0xBC,
    0x00,0xD0,0x85,0x60,0x20,0x44,0xD1,0x20,0x3E,0xD3,0xA0,
    0x00,0xAA,0xF0,0x49,0xB1,0x71,0x20,0xED,0xC8,0xC8,0xCA,
    0xD0,0xF7,0x60,0xA9,0x20,0x2C,0xA9,0x3F,0xC9,0x20,0x90,
    0x19,0x48,0xA5,0x0F,0xD0,0x0A,0xA5,0x0E,0xE5,0x64,0xD0,
    0x0B,0x85,0x0E,0xF0,0x07,0xC5,0x0E,0xD0,0x03,0x20,0x92,
    0xC8,0xE6,0x0E,0x68,0x20,0xED,0xE0,0xC9,0x0D,0xD0,0x14,
    0x86,0x78,0xA6,0x0D,0xF0,0x0A,0xA9,0x00,0x20,0xED,0xC8,
    0xCA,0xD0,0xFA,0xA9,0x0D,0x86,0x0E,0xA6,0x78,0x29,0xFF,
    0x60,0xA5,0x62,0x10,0x0B,0xA5,0x8D,0xA4,0x8E,0x85,0x87,
    0x84,0x88,0x4C,0x02,0xCC,0xA9,0x12,0xA0,0xE8,0x20,0xD3,
    0xC8,0xA5,0x8B,0xA4,0x8C,0x85,0xC3,0x84,0xC4,0x60,0xC9,
    0x22,0xD0,0x0B,0x20,0xBE,0xCB,0xA9,0x3B,0x20,0xF1,0xCB,
    0x20,0xD6,0xC8,0x20,0x6A,0xD0,0x20,0x40,0xC2,0xA9,0x00,
    0xCD,0x21,0x02,0xD0,0x0A,0x18,0x4C,0x2C,0xC5,0xA6,0x8F,
    0xA4,0x90,0xA9,0x80,0x85,0x62,0x86,0x91,0x84,0x92,0x20,
    0xAA,0xCD,0x85,0x97,0x84,0x98,0xA5,0xC3,0xA4,0xC4,0x85,
    0x11,0x84,0x12,0xA6,0x91,0xA4,0x92,0x86,0xC3,0x84,0xC4,
    0x20,0xC2,0x00,0xD0,0x11,0x24,0x62,0x30,0x65,0x20,0xEB,
    0xC8,0x20,0x40,0xC2,0x86,0xC3,0x84,0xC4,0x20,0xC2,0x00,
    0x24,0x5F,0x10,0x24,0x85,0x5B,0xC9,0x22,0xF0,0x07,0xA9,
    0x3A,0x85,0x5B,0xA9,0x2C,0x18,0x85,0x5C,0xA5,0xC3,0xA4,
    0xC4,0x69,0x00,0x90,0x01,0xC8,0x20,0x4A,0xD1,0x20,0xCF,
    0xD4,0x20,0xD0,0xC7,0x4C,0xCE,0xC9,0x20,0xA6,0xD9,0x20,
    0xA3,0xD8,0x20,0xC2,0x00,0xF0,0x0A,0xC9,0x2C,0xF0,0x03,
    0x4C,0x28,0xC9,0x20,0xBC,0x00,0xA5,0xC3,0xA4,0xC4,0x85,
    0x91,0x84,0x92,0xA5,0x11,0xA4,0x12,0x85,0xC3,0x84,0xC4,
    0x20,0xC2,0x00,0xF0,0x2C,0x20,0xFE,0xCB,0x4C,0x73,0xC9,
    0x20,0x9D,0xC6,0xC8,0xAA,0xD0,0x12,0xA2,0x06,0xC8,0xB1,
    0xC3,0xF0,0x73,0xC8,0xB1,0xC3,0x85,0x8D,0xC8,0xB1,0xC3,
    0xC8,0x85,0x8E,0xB1,0xC3,0xC8,0xAA,0x20,0x8F,0xC6,0xE0,
    0x83,0xF0,0x81,0xD0,0xDA,0xA5,0x91,0xA4,0x92,0xA6,0x62,
    0x10,0x03,0x4C,0x4E,0xC5,0xA0,0x00,0xB1,0x91,0xD0,0x01,
    0x60,0xA9,0x01,0xA0,0xE8,0x4C,0xD3,0xC8,0xBA,0xE8,0xE8,
    0xE8,0xE8,0xBD,0x01,0x01,0xC9,0x81,0xD0,0x21,0xA5,0x98,
    0xD0,0x0A,0xBD,0x02,0x01,0x85,0x97,0xBD,0x03,0x01,0x85,
    0x98,0xDD,0x03,0x01,0xD0,0x07,0xA5,0x97,0xDD,0x02,0x01,
    0xF0,0x07,0x8A,0x18,0x69,0x10,0xAA,0xD0,0xD8,0x60,0xD0,
    0x04,0xA0,0x00,0xF0,0x03,0x20,0xAA,0xCD,0x85,0x97,0x84,
    0x98,0x20,0x37,0xCA,0xF0,0x04,0xA2,0x00,0xF0,0x63,0x9A,
    0x8A,0x38,0xE9,0xF7,0x85,0x73,0x69,0xFB,0xA0,0x01,0x20,
    0x7D,0xD8,0xBA,0xBD,0x08,0x01,0x85,0xB0,0xA5,0x97,0xA4,
    0x98,0x20,0xBE,0xD5,0x20,0xA3,0xD8,0xA0,0x01,0x20,0x19,
    0xD9,0xBA,0xDD,0x08,0x01,0xF0,0x17,0xBD,0x0D,0x01,0x85,
    0x87,0xBD,0x0E,0x01,0x85,0x88,0xBD,0x10,0x01,0x85,0xC3,
    0xBD,0x0F,0x01,0x85,0xC4,0x4C,0xBC,0xC4,0x8A,0x69,0x0F,
    0xAA,0x9A,0x20,0xC2,0x00,0xC9,0x2C,0xD0,0xF1,0x20,0xBC,
    0x00,0x20,0x6B,0xCA,0x20,0xE1,0xCA,0x18,0x24,0x38,0x24,
    0x5F,0x30,0x03,0xB0,0x03,0x60,0xB0,0xFD,0xA2,0x18,0x4C,
    0x3C,0xC1,0xA6,0xC3,0xD0,0x02,0xC6,0xC4,0xC6,0xC3,0xA9,
    0x00,0x48,0xA9,0x02,0x20,0x03,0xC1,0x20,0xCD,0xCB,0xA9,
    0x00,0x85,0x9B,0x20,0xC2,0x00,0x38,0xE9,0xC0,0x90,0x17,
    0xC9,0x03,0xB0,0x13,0xC9,0x01,0x2A,0x49,0x01,0x45,0x9B,
    0xC5,0x9B,0x90,0x67,0x85,0x9B,0x20,0xBC,0x00,0x4C,0xFB,
    0xCA,0xA6,0x9B,0xD0,0x2C,0xB0,0x79,0x69,0x0A,0x90,0x75,
    0xD0,0x07,0x24,0x5F,0x10,0x03,0x4C,0xD4,0xD2,0x85,0x71,
    0x0A,0x65,0x71,0xA8,0x68,0xD9,0xE9,0xE2,0xB0,0x65,0x20,
    0xD0,0xCA,0x48,0x20,0x62,0xCB,0x68,0xA4,0x99,0x10,0x19,
    0xAA,0xF0,0x76,0xD0,0x5D,0x26,0x5F,0x8A,0x85,0x5F,0x2A,
    0xA6,0xC3,0xD0,0x02,0xC6,0xC4,0xC6,0xC3,0xA0,0x24,0x85,
    0x9B,0xD0,0xD5,0xD9,0xE9,0xE2,0xB0,0x44,0x90,0xD7,0xB9,
    0xEB,0xE2,0x48,0xB9,0xEA,0xE2,0x48,0x20,0x79,0xCB,0xA5,
    0x9B,0x48,0xB9,0xE9,0xE2,0x4C,0xEB,0xCA,0x4C,0x02,0xCC,
    0x68,0x85,0x71,0xE6,0x71,0x68,0x85,0x72,0xA5,0xB0,0x48,
    0x20,0xD9,0xD8,0xA5,0xAF,0x48,0xA5,0xAE,0x48,0xA5,0xAD,
    0x48,0xA5,0xAC,0x48,0x6C,0x71,0x00,0xA0,0xFF,0x68,0xF0,
    0x20,0xC9,0x64,0xF0,0x03,0x20,0xD0,0xCA,0x84,0x99,0x68,
    0x4A,0x85,0x63,0x68,0x85,0xB3,0x68,0x85,0xB4,0x68,0x85,
    0xB5,0x68,0x85,0xB6,0x68,0x85,0xB7,0x45,0xB0,0x85,0xB8,
    0xA5,0xAC,0x60,0xA5,0xC3,0xA4,0xC4,0x69,0x00,0x90,0x01,
    0xC8,0x20,0x44,0xD1,0x4C,0xCF,0xD4,0x20,0xBC,0x00,0xB0,
    0x03,0x4C,0xA6,0xD9,0xAA,0x30,0x2F,0xC9,0x24,0xF0,0xF6,
    0xC9,0x25,0xF0,0xF2,0xC9,0x2E,0xF0,0xEE,0xC9,0x22,0xF0,
    0xD6,0xC9,0x28,0xD0,0x4F,0x20,0xE9,0xCA,0xA9,0x29,0xA0,
    0x00,0xD1,0xC3,0xD0,0x0B,0x4C,0xBC,0x00,0xA9,0x28,0xD0,
    0xF3,0xA9,0x2C,0xD0,0xEF,0xA2,0x02,0x4C,0x3C,0xC1,0xC9,
    0xB7,0xF0,0x29,0xC9,0xB6,0xF0,0xBE,0xC9,0xB1,0xD0,0x13,
    0xA0,0x21,0xD0,0x1F,0x20,0xA6,0xCE,0xA5,0xAF,0x49,0xFF,
    0xA8,0xA5,0xAE,0x49,0xFF,0x4C,0x59,0xD0,0xC9,0xAE,0xD0,
    0x03,0x4C,0xB4,0xD0,0xE9,0xC3,0xB0,0x19,0x4C,0x02,0xCC,
    0xA0,0x1E,0x68,0x68,0x4C,0x3A,0xCB,0x20,0xAA,0xCD,0x85,
    0xAE,0x84,0xAF,0xA6,0x5F,0x30,0x03,0x4C,0x7D,0xD8,0x60,
    0x0A,0xA8,0xB9,0xA4,0xE2,0x48,0xB9,0xA3,0xE2,0x48,0xB9,
    0x5E,0xE2,0xF0,0x05,0x48,0xB9,0x5D,0xE2,0x48,0x60,0x20,
    0xEC,0xCB,0x4C,0xD2,0xCA,0x20,0xEC,0xCB,0x4C,0xD0,0xCA,
    0x46,0x5F,0x4C,0xBC,0x00,0x20,0xE9,0xCA,0x20,0xFE,0xCB,
    0x20,0xD2,0xCA,0x68,0xAA,0x68,0xA8,0xA5,0xAF,0x48,0xA5,
    0xAE,0x48,0x98,0x48,0x8A,0x48,0x20,0x8C,0xD4,0x8A,0x60,
    0x20,0xE9,0xCA,0x20,0xD0,0xCA,0xA5,0xAC,0xC9,0x98,0xB0,
    0x20,0x20,0x50,0xD9,0xA2,0x02,0xB5,0xAD,0x95,0x11,0xCA,
    0x10,0xF9,0x20,0xC2,0x00,0xA2,0x00,0xC9,0x29,0xF0,0x0A,
    0x20,0xDE,0xD4,0x20,0xC2,0x00,0xC9,0x29,0xD0,0x01,0x60,
    0x4C,0x29,0xCF,0x20,0xE2,0xCC,0x45,0x5B,0xA8,0xA5,0xAE,
    0x45,0x5C,0x4C,0x59,0xD0,0x20,0xE2,0xCC,0x05,0x5B,0xA8,
    0xA5,0xAE,0x05,0x5C,0x4C,0x59,0xD0,0x20,0xE2,0xCC,0x25,
    0x5B,0xA8,0xA5,0xAE,0x25,0x5C,0x4C,0x59,0xD0,0x20,0xA6,
    0xCE,0xA5,0xAE,0x85,0x5C,0xA5,0xAF,0x85,0x5B,0x20,0xC3,
    0xD5,0x20,0xA6,0xCE,0xA5,0xAF,0x60,0x20,0xD3,0xCA,0xB0,
    0x13,0xA5,0xB7,0x09,0x7F,0x25,0xB4,0x85,0xB4,0xA9,0xB3,
    0xA0,0x00,0x20,0x17,0xD9,0xAA,0x4C,0x3F,0xCD,0x46,0x5F,
    0xC6,0x9B,0x20,0x3E,0xD3,0x85,0xAC,0x86,0xAD,0x84,0xAE,
    0xA5,0xB5,0xA4,0xB6,0x20,0x42,0xD3,0x86,0xB5,0x84,0xB6,
    0xAA,0x38,0xE5,0xAC,0xF0,0x08,0xA9,0x01,0x90,0x04,0xA6,
    0xAC,0xA9,0xFF,0x85,0xB0,0xA0,0xFF,0xE8,0xC8,0xCA,0xD0,
    0x07,0xA6,0xB0,0x30,0x0F,0x18,0x90,0x0C,0xB1,0xB5,0xD1,
    0xAD,0xF0,0xEF,0xA2,0xFF,0xB0,0x02,0xA2,0x01,0xE8,0x8A,
    0x2A,0x25,0x63,0xF0,0x02,0xA9,0xFF,0x4C,0xFA,0xD8,0x20,
    0xFE,0xCB,0xAA,0x20,0xAF,0xCD,0x20,0xC2,0x00,0xD0,0xF4,
    0x60,0x20,0x9F,0xCD,0xA5,0xAE,0xA6,0x78,0xF0,0x22,0xE0,
    0x10,0xB0,0x23,0x06,0xAF,0x2A,0xCA,0xD0,0xFA,0xA4,0xAF,
    0x4C,0x59,0xD0,0x20,0x9F,0xCD,0xA5,0xAE,0xA6,0x78,0xF0,
    0x0A,0xE0,0x10,0xB0,0x0B,0x4A,0x66,0xAF,0xCA,0xD0,0xFA,
    0xA4,0xAF,0x4C,0x59,0xD0,0xA9,0x00,0xA8,0x4C,0x59,0xD0,
    0x20,0x8F,0xD4,0x86,0x78,0x20,0xC3,0xD5,0x4C,0xA6,0xCE,
    0xA2,0x00,0x20,0xC2,0x00,0x86,0x5E,0x85,0x93,0x29,0x7F,
    0x20,0x1E,0xCE,0xB0,0x03,0x4C,0x02,0xCC,0xA2,0x00,0x86,
    0x5F,0x20,0xBC,0x00,0x90,0x05,0x20,0x1E,0xCE,0x90,0x0B,
    0xAA,0x20,0xBC,0x00,0x90,0xFB,0x20,0x1E,0xCE,0xB0,0xF6,
    0xC9,0x24,0xD0,0x0B,0xA9,0xFF,0x85,0x5F,0x8A,0x09,0x80,
    0xAA,0x20,0xBC,0x00,0x86,0x94,0x05,0x61,0xC9,0x28,0xD0,
    0x03,0x4C,0xB8,0xCE,0xA9,0x00,0x85,0x61,0xA5,0x7B,0xA6,
    0x7C,0xA0,0x00,0x86,0xAB,0x85,0xAA,0xE4,0x7E,0xD0,0x04,
    0xC5,0x7D,0xF0,0x2C,0xA5,0x93,0xD1,0xAA,0xD0,0x08,0xA5,
    0x94,0xC8,0xD1,0xAA,0xF0,0x69,0x88,0x18,0xA5,0xAA,0x69,
    0x06,0x90,0xE1,0xE8,0xD0,0xDC,0xC9,0x61,0xB0,0x0A,0xC9,
    0x41,0x90,0x05,0xE9,0x5B,0x38,0xE9,0xA5,0x60,0xE9,0x7B,
    0x38,0xE9,0x85,0x60,0x68,0x48,0xC9,0x3D,0xD0,0x05,0xA9,
    0xE1,0xA0,0xE1,0x60,0xA5,0x7D,0xA4,0x7E,0x85,0xAA,0x84,
    0xAB,0xA5,0x7F,0xA4,0x80,0x85,0xA6,0x84,0xA7,0x18,0x69,
    0x06,0x90,0x01,0xC8,0x85,0xA4,0x84,0xA5,0x20,0xC1,0xC0,
    0xA5,0xA4,0xA4,0xA5,0xC8,0x85,0x7D,0x84,0x7E,0xA0,0x00,
    0xA5,0x93,0x91,0xAA,0xC8,0xA5,0x94,0x91,0xAA,0xA9,0x00,
    0xC8,0x91,0xAA,0xC8,0x91,0xAA,0xC8,0x91,0xAA,0xC8,0x91,
    0xAA,0xA5,0xAA,0x18,0x69,0x02,0xA4,0xAB,0x90,0x01,0xC8,
    0x85,0x95,0x84,0x96,0x60,0xA5,0x5D,0x0A,0x69,0x05,0x65,
    0xAA,0xA4,0xAB,0x90,0x01,0xC8,0x85,0xA4,0x84,0xA5,0x60,
    0x20,0xBC,0x00,0x20,0xCD,0xCA,0xA5,0xB0,0x30,0x0D,0xA5,
    0xAC,0xC9,0x90,0x90,0x09,0xA9,0xE8,0xA0,0xE1,0x20,0x17,
    0xD9,0xD0,0x74,0x4C,0x50,0xD9,0xA5,0x5E,0x48,0xA5,0x5F,
    0x48,0xA0,0x00,0x98,0x48,0xA5,0x94,0x48,0xA5,0x93,0x48,
    0x20,0x9C,0xCE,0x68,0x85,0x93,0x68,0x85,0x94,0x68,0xA8,
    0xBA,0xBD,0x02,0x01,0x48,0xBD,0x01,0x01,0x48,0xA5,0xAE,
    0x9D,0x02,0x01,0xA5,0xAF,0x9D,0x01,0x01,0xC8,0x20,0xC2,
    0x00,0xC9,0x2C,0xF0,0xD2,0x84,0x5D,0x20,0xEF,0xCB,0x68,
    0x85,0x5F,0x68,0x85,0x5E,0xA6,0x7D,0xA5,0x7E,0x86,0xAA,
    0x85,0xAB,0xC5,0x80,0xD0,0x04,0xE4,0x7F,0xF0,0x39,0xA0,
    0x00,0xB1,0xAA,0xC8,0xC5,0x93,0xD0,0x06,0xA5,0x94,0xD1,
    0xAA,0xF0,0x16,0xC8,0xB1,0xAA,0x18,0x65,0xAA,0xAA,0xC8,
    0xB1,0xAA,0x65,0xAB,0x90,0xD7,0xA2,0x10,0x2C,0xA2,0x08,
    0x4C,0x3C,0xC1,0xA2,0x12,0xA5,0x5E,0xD0,0xF7,0x20,0x8B,
    0xCE,0xA5,0x5D,0xA0,0x04,0xD1,0xAA,0xD0,0xE7,0x4C,0xC5,
    0xCF,0x20,0x8B,0xCE,0x20,0x0B,0xC1,0xA0,0x00,0x84,0xBB,
    0xA5,0x93,0x91,0xAA,0xC8,0xA5,0x94,0x91,0xAA,0xA5,0x5D,
    0xA0,0x04,0x84,0xBA,0x91,0xAA,0x18,0xA2,0x0B,0xA9,0x00,
    0x24,0x5E,0x50,0x07,0x68,0x69,0x01,0xAA,0x68,0x69,0x00,
    0xC8,0x91,0xAA,0xC8,0x8A,0x91,0xAA,0x20,0x14,0xD0,0x86,
    0xBA,0x85,0xBB,0xA4,0x71,0xC6,0x5D,0xD0,0xDD,0x65,0xA5,
    0xB0,0x5D,0x85,0xA5,0xA8,0x8A,0x65,0xA4,0x90,0x03,0xC8,
    0xF0,0x52,0x20,0x0B,0xC1,0x85,0x7F,0x84,0x80,0xA9,0x00,
    0xE6,0xBB,0xA4,0xBA,0xF0,0x05,0x88,0x91,0xA4,0xD0,0xFB,
    0xC6,0xA5,0xC6,0xBB,0xD0,0xF5,0xE6,0xA5,0x38,0xA0,0x02,
    0xA5,0x7F,0xE5,0xAA,0x91,0xAA,0xC8,0xA5,0x80,0xE5,0xAB,
    0x91,0xAA,0xA5,0x5E,0xD0,0x53,0xC8,0xB1,0xAA,0x85,0x5D,
    0xA9,0x00,0x85,0xBA,0x85,0xBB,0xC8,0x68,0xAA,0x85,0xAE,
    0x68,0x85,0xAF,0xD1,0xAA,0x90,0x0E,0xD0,0x06,0xC8,0x8A,
    0xD1,0xAA,0x90,0x07,0x4C,0x26,0xCF,0x4C,0x3A,0xC1,0xC8,
    0xA5,0xBB,0x05,0xBA,0xF0,0x0A,0x20,0x14,0xD0,0x8A,0x65,
    0xAE,0xAA,0x98,0xA4,0x71,0x65,0xAF,0x86,0xBA,0xC6,0x5D,
    0xD0,0xCB,0x06,0xBA,0x2A,0x06,0xBA,0x2A,0xA8,0xA5,0xBA,
    0x65,0xA4,0x85,0x95,0x98,0x65,0xA5,0x85,0x96,0xA8,0xA5,
    0x95,0x60,0x84,0x71,0xB1,0xAA,0x85,0x76,0x88,0xB1,0xAA,
    0x85,0x77,0xA9,0x10,0x85,0xA8,0xA2,0x00,0xA0,0x00,0x8A,
    0x0A,0xAA,0x98,0x2A,0xA8,0xB0,0xB3,0x06,0xBA,0x26,0xBB,
    0x90,0x0B,0x18,0x8A,0x65,0x76,0xAA,0x98,0x65,0x77,0xA8,
    0xB0,0xA2,0xC6,0xA8,0xD0,0xE3,0x60,0xA5,0x5F,0x10,0x03,
    0x20,0x3E,0xD3,0x20,0xDF,0xD1,0x38,0xA5,0x81,0xE5,0x7F,
    0xA8,0xA5,0x82,0xE5,0x80,0x46,0x5F,0x85,0xAD,0x84,0xAE,
    0xA2,0x90,0x4C,0x02,0xD9,0xA4,0x0E,0xA9,0x00,0xF0,0xEF,
    0xA6,0x88,0xE8,0xD0,0xA4,0xA2,0x16,0x4C,0x3C,0xC1,0x20,
    0xA5,0xD0,0x85,0x9C,0x84,0x9D,0x20,0x6A,0xD0,0x20,0xFA,
    0xCB,0xA9,0x80,0x85,0x61,0x20,0xAA,0xCD,0x20,0xD0,0xCA,
    0x20,0xEF,0xCB,0xA9,0xC1,0x20,0xF1,0xCB,0xA5,0x96,0x48,
    0xA5,0x95,0x48,0xA5,0xC4,0x48,0xA5,0xC3,0x48,0x20,0x8C,
    0xC6,0x4C,0x14,0xD1,0xA9,0xAE,0x20,0xF1,0xCB,0x09,0x80,
    0x85,0x61,0x20,0xB1,0xCD,0x4C,0xD0,0xCA,0x20,0xA5,0xD0,
    0x48,0x98,0x48,0x20,0xFA,0xCB,0x20,0xE1,0xCA,0x20,0xEF,
    0xCB,0x20,0xD0,0xCA,0x68,0x85,0x9D,0x68,0x85,0x9C,0xA2,
    0x20,0xA0,0x03,0xB1,0x9C,0xF0,0x9D,0x85,0x96,0x88,0xB1,
    0x9C,0x85,0x95,0xAA,0xC8,0xB1,0x95,0x48,0x88,0x10,0xFA,
    0xA4,0x96,0x20,0xA7,0xD8,0xA5,0xC4,0x48,0xA5,0xC3,0x48,
    0xB1,0x9C,0x85,0xC3,0xC8,0xB1,0x9C,0x85,0xC4,0xA5,0x96,
    0x48,0xA5,0x95,0x48,0x20,0xCD,0xCA,0x68,0x85,0x9C,0x68,
    0x85,0x9D,0x20,0xC2,0x00,0xF0,0x03,0x4C,0x02,0xCC,0x68,
    0x85,0xC3,0x68,0x85,0xC4,0xA0,0x00,0x68,0x91,0x9C,0xC8,
    0x68,0x91,0x9C,0xC8,0x68,0x91,0x9C,0xC8,0x68,0x91,0x9C,
    0x60,0x20,0xD0,0xCA,0x20,0x95,0xDA,0xA9,0xF0,0xA0,0x00,
    0xF0,0x12,0xA6,0xAE,0xA4,0xAF,0x86,0x9E,0x84,0x9F,0x20,
    0xAD,0xD1,0x86,0xAD,0x84,0xAE,0x85,0xAC,0x60,0xA2,0x22,
    0x86,0x5B,0x86,0x5C,0x85,0xB8,0x84,0xB9,0x85,0xAD,0x84,
    0xAE,0xA0,0xFF,0xC8,0xB1,0xB8,0xF0,0x0C,0xC5,0x5B,0xF0,
    0x04,0xC5,0x5C,0xD0,0xF3,0xC9,0x22,0xF0,0x01,0x18,0x84,
    0xAC,0x98,0x65,0xB8,0x85,0xBA,0xA6,0xB9,0x90,0x01,0xE8,
    0x86,0xBB,0xA5,0xB9,0xC9,0x03,0xB0,0x0B,0x98,0x20,0x32,
    0xD1,0xA6,0xB8,0xA4,0xB9,0x20,0x1F,0xD3,0xA6,0x65,0xE0,
    0x71,0xD0,0x05,0xA2,0x1C,0x4C,0x3C,0xC1,0xA5,0xAC,0x95,
    0x00,0xA5,0xAD,0x95,0x01,0xA5,0xAE,0x95,0x02,0xA0,0x00,
    0x86,0xAE,0x84,0xAF,0x88,0x84,0x5F,0x86,0x66,0xE8,0xE8,
    0xE8,0x86,0x65,0x60,0x46,0x60,0x48,0x49,0xFF,0x38,0x65,
    0x81,0xA4,0x82,0xB0,0x01,0x88,0xC4,0x80,0x90,0x11,0xD0,
    0x04,0xC5,0x7F,0x90,0x0B,0x85,0x81,0x84,0x82,0x85,0x83,
    0x84,0x84,0xAA,0x68,0x60,0xA2,0x0C,0xA5,0x60,0x30,0xB8,
    0x20,0xDF,0xD1,0xA9,0x80,0x85,0x60,0x68,0xD0,0xD0,0xA6,
    0x85,0xA5,0x86,0x86,0x81,0x85,0x82,0xA0,0x00,0x84,0x9D,
    0xA5,0x7F,0xA6,0x80,0x85,0xAA,0x86,0xAB,0xA9,0x68,0x85,
    0x71,0x84,0x72,0xC5,0x65,0xF0,0x05,0x20,0x63,0xD2,0xF0,
    0xF7,0x06,0xA0,0xA5,0x7B,0xA6,0x7C,0x85,0x71,0x86,0x72,
    0xE4,0x7E,0xD0,0x04,0xC5,0x7D,0xF0,0x05,0x20,0x5D,0xD2,
    0xF0,0xF3,0x85,0xA4,0x86,0xA5,0xA9,0x04,0x85,0xA0,0xA5,
    0xA4,0xA6,0xA5,0xE4,0x80,0xD0,0x04,0xC5,0x7F,0xF0,0x75,
    0x85,0x71,0x86,0x72,0xA0,0x02,0xB1,0x71,0x65,0xA4,0x85,
    0xA4,0xC8,0xB1,0x71,0x65,0xA5,0x85,0xA5,0xA0,0x01,0xB1,
    0x71,0x10,0xDB,0xA0,0x04,0xB1,0x71,0x0A,0x69,0x05,0x20,
    0x95,0xD2,0xE4,0xA5,0xD0,0x04,0xC5,0xA4,0xF0,0xCD,0x20,
    0x63,0xD2,0xF0,0xF3,0xC8,0xB1,0x71,0x10,0x30,0xC8,0xB1,
    0x71,0xF0,0x2B,0xC8,0xB1,0x71,0xAA,0xC8,0xB1,0x71,0xC5,
    0x82,0x90,0x06,0xD0,0x1E,0xE4,0x81,0xB0,0x1A,0xC5,0xAB,
    0x90,0x17,0xD0,0x04,0xE4,0xAA,0x90,0x11,0x86,0xAA,0x85,
    0xAB,0xA5,0x71,0xA6,0x72,0x85,0x9C,0x86,0x9D,0x88,0x88,
    0x84,0xA2,0x18,0xA5,0xA0,0x65,0x71,0x85,0x71,0x90,0x02,
    0xE6,0x72,0xA6,0x72,0xA0,0x00,0x60,0xC6,0xA0,0xA6,0x9D,
    0xF0,0xF5,0xA4,0xA2,0x18,0xB1,0x9C,0x65,0xAA,0x85,0xA6,
    0xA5,0xAB,0x69,0x00,0x85,0xA7,0xA5,0x81,0xA6,0x82,0x85,
    0xA4,0x86,0xA5,0x20,0xC8,0xC0,0xA4,0xA2,0xC8,0xA5,0xA4,
    0x91,0x9C,0xAA,0xE6,0xA5,0xA5,0xA5,0xC8,0x91,0x9C,0x4C,
    0xE3,0xD1,0xA5,0xAF,0x48,0xA5,0xAE,0x48,0x20,0xCD,0xCB,
    0x20,0xD2,0xCA,0x68,0x85,0xB8,0x68,0x85,0xB9,0xA0,0x00,
    0xB1,0xB8,0x18,0x71,0xAE,0x90,0x05,0xA2,0x1A,0x4C,0x3C,
    0xC1,0x20,0x32,0xD1,0x20,0x11,0xD3,0xA5,0x9E,0xA4,0x9F,
    0x20,0x42,0xD3,0x20,0x23,0xD3,0xA5,0xB8,0xA4,0xB9,0x20,
    0x42,0xD3,0x20,0x85,0xD1,0x4C,0xF8,0xCA,0xA0,0x00,0xB1,
    0xB8,0x48,0xC8,0xB1,0xB8,0xAA,0xC8,0xB1,0xB8,0xA8,0x68,
    0x86,0x71,0x84,0x72,0xAA,0xF0,0x14,0xA0,0x00,0xB1,0x71,
    0x91,0x83,0xC8,0xCA,0xD0,0xF8,0x98,0x18,0x65,0x83,0x85,
    0x83,0x90,0x02,0xE6,0x84,0x60,0x20,0xD2,0xCA,0xA5,0xAE,
    0xA4,0xAF,0x85,0x71,0x84,0x72,0x20,0x73,0xD3,0x08,0xA0,
    0x00,0xB1,0x71,0x48,0xC8,0xB1,0x71,0xAA,0xC8,0xB1,0x71,
    0xA8,0x68,0x28,0xD0,0x13,0xC4,0x82,0xD0,0x0F,0xE4,0x81,
    0xD0,0x0B,0x48,0x18,0x65,0x81,0x85,0x81,0x90,0x02,0xE6,
    0x82,0x68,0x86,0x71,0x84,0x72,0x60,0xC4,0x67,0xD0,0x0C,
    0xC5,0x66,0xD0,0x08,0x85,0x65,0xE9,0x03,0x85,0x66,0xA0,
    0x00,0x60,0x20,0x8F,0xD4,0x8A,0x48,0xA9,0x01,0x20,0x3A,
    0xD1,0x68,0xA0,0x00,0x91,0xAD,0x4C,0x85,0xD1,0x48,0x20,
    0xF7,0xD3,0xD1,0x9E,0x98,0xF0,0x09,0x48,0x20,0xF7,0xD3,
    0x18,0xF1,0x9E,0x49,0xFF,0x90,0x04,0xB1,0x9E,0xAA,0x98,
    0x48,0x8A,0x48,0x20,0x3A,0xD1,0xA5,0x9E,0xA4,0x9F,0x20,
    0x42,0xD3,0x68,0xA8,0x68,0x18,0x65,0x71,0x85,0x71,0x90,
    0x02,0xE6,0x72,0x98,0x20,0x23,0xD3,0x4C,0x85,0xD1,0x48,
    0xA9,0xFF,0x85,0xAF,0x20,0xC2,0x00,0xC9,0x29,0xF0,0x06,
    0x20,0xFE,0xCB,0x20,0x8C,0xD4,0x20,0xF7,0xD3,0xCA,0x8A,
    0x48,0x18,0xA2,0x00,0xF1,0x9E,0xB0,0xC2,0x49,0xFF,0xC5,
    0xAF,0x90,0xBD,0xA5,0xAF,0xB0,0xB9,0x20,0xEF,0xCB,0x68,
    0x85,0xA2,0x68,0x85,0xA3,0x68,0xAA,0x68,0x85,0x9E,0x68,
    0x85,0x9F,0xA0,0x00,0x8A,0xF0,0x79,0xE6,0xA2,0x6C,0xA2,
    0x00,0x20,0x3B,0xD3,0x85,0xAC,0xA8,0xF0,0x38,0x20,0x3A,
    0xD1,0x86,0xAD,0x84,0xAE,0xA8,0x88,0xB1,0x71,0x20,0x22,
    0xCE,0x90,0x02,0x09,0x20,0x91,0x83,0x98,0xD0,0xF1,0xF0,
    0x1F,0x20,0x3B,0xD3,0x85,0xAC,0xA8,0xF0,0x17,0x20,0x3A,
    0xD1,0x86,0xAD,0x84,0xAE,0xA8,0x88,0xB1,0x71,0x20,0x1E,
    0xCE,0x90,0x02,0x29,0xDF,0x91,0x83,0x98,0xD0,0xF1,0x4C,
    0x85,0xD1,0x20,0xBC,0x00,0x20,0xAA,0xCD,0x20,0xEF,0xCB,
    0x20,0xD2,0xCA,0xA0,0x02,0xB1,0x95,0xAA,0x88,0xB1,0x95,
    0xA8,0x8A,0x4C,0x59,0xD0,0x20,0x74,0xD4,0x4C,0x66,0xD0,
    0x20,0x3B,0xD3,0xA8,0x60,0x20,0x74,0xD4,0xF0,0x08,0xA0,
    0x00,0xB1,0x71,0xA8,0x4C,0x66,0xD0,0x4C,0x29,0xCF,0x20,
    0xBC,0x00,0x20,0xCD,0xCA,0x20,0xA2,0xCE,0xA4,0xAE,0xD0,
    0xF0,0xA6,0xAF,0x4C,0xC2,0x00,0x20,0x74,0xD4,0xD0,0x03,
    0x4C,0x50,0xD6,0xA6,0xC3,0xA4,0xC4,0x86,0xBA,0x84,0xBB,
    0xA6,0x71,0x86,0xC3,0x18,0x65,0x71,0x85,0x73,0xA5,0x72,
    0x85,0xC4,0x69,0x00,0x85,0x74,0xA0,0x00,0xB1,0x73,0x48,
    0x98,0x91,0x73,0x20,0xC2,0x00,0x20,0xA6,0xD9,0x68,0xA0,
    0x00,0x91,0x73,0xA6,0xBA,0xA4,0xBB,0x86,0xC3,0x84,0xC4,
    0x60,0x20,0xCD,0xCA,0x20,0xF1,0xD4,0x20,0xFE,0xCB,0xA5,
    0x12,0x48,0xA5,0x11,0x48,0x20,0x8C,0xD4,0x68,0x85,0x11,
    0x68,0x85,0x12,0x60,0xA5,0xAC,0xC9,0x98,0xB0,0x8F,0x20,
    0x50,0xD9,0xA5,0xAE,0xA4,0xAF,0x84,0x11,0x85,0x12,0x60,
    0x20,0xF1,0xD4,0xA2,0x00,0xA1,0x11,0xA8,0x4C,0x66,0xD0,
    0x20,0xD8,0xD4,0x8A,0xA2,0x00,0x81,0x11,0x60,0x20,0xF1,
    0xD4,0xA2,0x00,0xA1,0x11,0xA8,0xE6,0x11,0xD0,0x02,0xE6,
    0x12,0xA1,0x11,0x4C,0x59,0xD0,0x20,0xCD,0xCA,0x20,0xF1,
    0xD4,0x84,0x97,0x85,0x98,0x20,0xFE,0xCB,0x20,0xCD,0xCA,
    0x20,0xF1,0xD4,0x98,0xA2,0x00,0x81,0x97,0xE6,0x97,0xD0,
    0x02,0xE6,0x98,0xA5,0x12,0x81,0x97,0x4C,0xC2,0x00,0x20,
    0xAA,0xCD,0x85,0x97,0x84,0x98,0xA5,0x5F,0x48,0x20,0xFE,
    0xCB,0x20,0xAA,0xCD,0x68,0x45,0x5F,0x10,0x10,0xA0,0x03,
    0xB1,0x97,0xAA,0xB1,0x95,0x91,0x97,0x8A,0x91,0x95,0x88,
    0x10,0xF3,0x60,0x4C,0xDC,0xCA,0x20,0xCD,0xCA,0x20,0xF1,
    0xD4,0xA9,0xD5,0x48,0xA9,0x85,0x48,0x6C,0x11,0x00,0x4C,
    0xC2,0x00,0x20,0xD8,0xD4,0x86,0x97,0xA2,0x00,0x20,0xC2,
    0x00,0xF0,0x03,0x20,0xDE,0xD4,0x86,0x98,0xB1,0x11,0x45,
    0x98,0x25,0x97,0xF0,0xF8,0x60,0x20,0x8B,0xD7,0xA5,0xB0,
    0x49,0xFF,0x85,0xB0,0x45,0xB7,0x85,0xB8,0xA5,0xAC,0x4C,
    0xC1,0xD5,0x20,0xDA,0xD6,0x90,0x4D,0xA9,0xE9,0xA0,0xE1,
    0x20,0x8B,0xD7,0xD0,0x10,0xA5,0xB7,0x85,0xB0,0xA2,0x04,
    0xB5,0xB2,0x95,0xAB,0xCA,0xD0,0xF9,0x86,0xB9,0x60,0xA6,
    0xB9,0x86,0xA3,0xA2,0xB3,0xA5,0xB3,0xA8,0xF0,0xC4,0x38,
    0xE5,0xAC,0xF0,0x24,0x90,0x12,0x84,0xAC,0xA4,0xB7,0x84,
    0xB0,0x49,0xFF,0x69,0x00,0xA0,0x00,0x84,0xA3,0xA2,0xAC,
    0xD0,0x04,0xA0,0x00,0x84,0xB9,0xC9,0xF9,0x30,0xB6,0xA8,
    0xA5,0xB9,0x56,0x01,0x20,0xF1,0xD6,0x24,0xB8,0x10,0x4C,
    0xA0,0xAC,0xE0,0xB3,0xF0,0x02,0xA0,0xB3,0x38,0x49,0xFF,
    0x65,0xA3,0x85,0xB9,0xB9,0x03,0x00,0xF5,0x03,0x85,0xAF,
    0xB9,0x02,0x00,0xF5,0x02,0x85,0xAE,0xB9,0x01,0x00,0xF5,
    0x01,0x85,0xAD,0xB0,0x03,0x20,0x96,0xD6,0xA0,0x00,0x98,
    0x18,0xA6,0xAD,0xD0,0x3E,0xA6,0xAE,0x86,0xAD,0xA6,0xAF,
    0x86,0xAE,0xA6,0xB9,0x86,0xAF,0x84,0xB9,0x69,0x08,0xC9,
    0x18,0xD0,0xE8,0xA9,0x00,0x85,0xAC,0x85,0xB0,0x60,0x65,
    0xA3,0x85,0xB9,0xA5,0xAF,0x65,0xB6,0x85,0xAF,0xA5,0xAE,
    0x65,0xB5,0x85,0xAE,0xA5,0xAD,0x65,0xB4,0x85,0xAD,0xB0,
    0x1A,0x60,0x69,0x01,0x06,0xB9,0x26,0xAF,0x26,0xAE,0x26,
    0xAD,0x10,0xF4,0x38,0xE5,0xAC,0xB0,0xCF,0x49,0xFF,0x69,
    0x01,0x85,0xAC,0x90,0x0C,0xE6,0xAC,0xF0,0x36,0x66,0xAD,
    0x66,0xAE,0x66,0xAF,0x66,0xB9,0x60,0xA5,0xB0,0x49,0xFF,
    0x85,0xB0,0xA5,0xAD,0x49,0xFF,0x85,0xAD,0xA5,0xAE,0x49,
    0xFF,0x85,0xAE,0xA5,0xAF,0x49,0xFF,0x85,0xAF,0xA5,0xB9,
    0x49,0xFF,0x85,0xB9,0xE6,0xB9,0xD0,0x0A,0xE6,0xAF,0xD0,
    0x06,0xE6,0xAE,0xD0,0x02,0xE6,0xAD,0x60,0xA2,0x0A,0x4C,
    0x3C,0xC1,0xA2,0x74,0xB4,0x03,0x84,0xB9,0xB4,0x02,0x94,
    0x03,0xB4,0x01,0x94,0x02,0xA4,0xB2,0x94,0x01,0x69,0x08,
    0x30,0xEC,0xF0,0xEA,0xE9,0x08,0xA8,0xA5,0xB9,0xB0,0x12,
    0x16,0x01,0x90,0x02,0xF6,0x01,0x76,0x01,0x76,0x01,0x76,
    0x02,0x76,0x03,0x6A,0xC8,0xD0,0xEE,0x18,0x60,0x20,0xE9,
    0xD8,0xF0,0x02,0x10,0x03,0x4C,0x29,0xCF,0xA5,0xAC,0xE9,
    0x7F,0x48,0xA9,0x80,0x85,0xAC,0xA9,0x69,0xA0,0xE1,0x20,
    0xBE,0xD5,0xA9,0x6D,0xA0,0xE1,0x20,0x01,0xD8,0xA9,0xE0,
    0xA0,0xE1,0x20,0xA3,0xD5,0xA9,0x5C,0xA0,0xE1,0x20,0x51,
    0xDC,0xA9,0x71,0xA0,0xE1,0x20,0xBE,0xD5,0x68,0x20,0x45,
    0xDA,0xA9,0x75,0xA0,0xE1,0x20,0x8B,0xD7,0xF0,0x4C,0x20,
    0xB1,0xD7,0xA9,0x00,0x85,0x75,0x85,0x76,0x85,0x77,0xA5,
    0xB9,0x20,0x60,0xD7,0xA5,0xAF,0x20,0x60,0xD7,0xA5,0xAE,
    0x20,0x60,0xD7,0xA5,0xAD,0x20,0x65,0xD7,0x4C,0x6E,0xD8,
    0xD0,0x03,0x4C,0xC8,0xD6,0x4A,0x09,0x80,0xA8,0x90,0x13,
    0x18,0xA5,0x77,0x65,0xB6,0x85,0x77,0xA5,0x76,0x65,0xB5,
    0x85,0x76,0xA5,0x75,0x65,0xB4,0x85,0x75,0x66,0x75,0x66,
    0x76,0x66,0x77,0x66,0xB9,0x98,0x4A,0xD0,0xDE,0x60,0x85,
    0x71,0x84,0x72,0xA0,0x03,0xB1,0x71,0x85,0xB6,0x88,0xB1,
    0x71,0x85,0xB5,0x88,0xB1,0x71,0x85,0xB7,0x45,0xB0,0x85,
    0xB8,0xA5,0xB7,0x09,0x80,0x85,0xB4,0x88,0xB1,0x71,0x85,
    0xB3,0xA5,0xAC,0x60,0xA5,0xB3,0xF0,0x1D,0x18,0x65,0xAC,
    0x90,0x04,0x30,0x31,0x18,0x2C,0x10,0x12,0x69,0x80,0x85,
    0xAC,0xD0,0x03,0x4C,0x54,0xD6,0xA5,0xB8,0x85,0xB0,0x60,
    0xA5,0xB0,0x10,0x1B,0x68,0x68,0x4C,0x50,0xD6,0x20,0xCA,
    0xD8,0xAA,0xF0,0xF0,0x18,0x69,0x02,0xB0,0x0B,0xA2,0x00,
    0x86,0xB8,0x20,0xDB,0xD5,0xE6,0xAC,0xD0,0xE0,0x4C,0xC3,
    0xD6,0x20,0xCA,0xD8,0xA9,0xF1,0xA0,0xE1,0xA2,0x00,0x86,
    0xB8,0x20,0x7D,0xD8,0x4C,0x04,0xD8,0x20,0x8B,0xD7,0xF0,
    0x63,0x20,0xD9,0xD8,0xA9,0x00,0x38,0xE5,0xAC,0x85,0xAC,
    0x20,0xB1,0xD7,0xE6,0xAC,0xF0,0xD6,0xA2,0xFF,0xA9,0x01,
    0xA4,0xB4,0xC4,0xAD,0xD0,0x0A,0xA4,0xB5,0xC4,0xAE,0xD0,
    0x04,0xA4,0xB6,0xC4,0xAF,0x08,0x2A,0x90,0x0E,0xA0,0x01,
    0xE8,0xE0,0x02,0x30,0x04,0xD0,0x28,0xA0,0x40,0x95,0x75,
    0x98,0x28,0x90,0x14,0xA8,0xA5,0xB6,0xE5,0xAF,0x85,0xB6,
    0xA5,0xB5,0xE5,0xAE,0x85,0xB5,0xA5,0xB4,0xE5,0xAD,0x85,
    0xB4,0x98,0x06,0xB6,0x26,0xB5,0x26,0xB4,0xB0,0xCF,0x30,
    0xBD,0x10,0xCB,0x4A,0x6A,0x6A,0x85,0xB9,0x28,0x4C,0x6E,
    0xD8,0xA2,0x14,0x4C,0x3C,0xC1,0xA5,0x75,0x85,0xAD,0xA5,
    0x76,0x85,0xAE,0xA5,0x77,0x85,0xAF,0x4C,0x34,0xD6,0x85,
    0x71,0x84,0x72,0xA0,0x03,0xB1,0x71,0x85,0xAF,0x88,0xB1,
    0x71,0x85,0xAE,0x88,0xB1,0x71,0x85,0xB0,0x09,0x80,0x85,
    0xAD,0x88,0xB1,0x71,0x85,0xAC,0x84,0xB9,0x60,0xA2,0xA4,
    0xA0,0x00,0xF0,0x04,0xA6,0x97,0xA4,0x98,0x20,0xD9,0xD8,
    0x86,0x71,0x84,0x72,0xA0,0x03,0xA5,0xAF,0x91,0x71,0x88,
    0xA5,0xAE,0x91,0x71,0x88,0xA5,0xB0,0x09,0x7F,0x25,0xAD,
    0x91,0x71,0x88,0xA5,0xAC,0x91,0x71,0x84,0xB9,0x60,0x20,
    0xD9,0xD8,0xA2,0x05,0xB5,0xAB,0x95,0xB2,0xCA,0xD0,0xF9,
    0x86,0xB9,0x60,0xA5,0xAC,0xF0,0xFB,0x06,0xB9,0x90,0xF7,
    0x20,0xB8,0xD6,0xD0,0xF2,0x4C,0x89,0xD6,0xA5,0xAC,0xF0,
    0x09,0xA5,0xB0,0x2A,0xA9,0xFF,0xB0,0x02,0xA9,0x01,0x60,
    0x20,0xE9,0xD8,0x85,0xAD,0xA9,0x00,0x85,0xAE,0xA2,0x88,
    0xA5,0xAD,0x49,0xFF,0x2A,0xA9,0x00,0x85,0xAF,0x86,0xAC,
    0x85,0xB9,0x85,0xB0,0x4C,0x2F,0xD6,0x46,0xB0,0x60,0x85,
    0x73,0x84,0x74,0xA0,0x00,0xB1,0x73,0xC8,0xAA,0xF0,0xC6,
    0xB1,0x73,0x45,0xB0,0x30,0xC4,0xE4,0xAC,0xD0,0x1A,0xB1,
    0x73,0x09,0x80,0xC5,0xAD,0xD0,0x12,0xC8,0xB1,0x73,0xC5,
    0xAE,0xD0,0x0B,0xC8,0xA9,0x7F,0xC5,0xB9,0xB1,0x73,0xE5,
    0xAF,0xF0,0x28,0xA5,0xB0,0x90,0x02,0x49,0xFF,0x4C,0xEF,
    0xD8,0xA5,0xAC,0xF0,0x4A,0x38,0xE9,0x98,0x24,0xB0,0x10,
    0x09,0xAA,0xA9,0xFF,0x85,0xB2,0x20,0x9C,0xD6,0x8A,0xA2,
    0xAC,0xC9,0xF9,0x10,0x06,0x20,0xDA,0xD6,0x84,0xB2,0x60,
    0xA8,0xA5,0xB0,0x29,0x80,0x46,0xAD,0x05,0xAD,0x85,0xAD,
    0x20,0xF1,0xD6,0x84,0xB2,0x60,0xA5,0xAC,0xC9,0x98,0xB0,
    0x1E,0x20,0x50,0xD9,0x84,0xB9,0xA5,0xB0,0x84,0xB0,0x49,
    0x80,0x2A,0xA9,0x98,0x85,0xAC,0xA5,0xAF,0x85,0x5B,0x4C,
    0x2F,0xD6,0x85,0xAD,0x85,0xAE,0x85,0xAF,0xA8,0x60,0xA0,
    0x00,0x84,0x5F,0xA2,0x09,0x94,0xA8,0xCA,0x10,0xFB,0x90,
    0x7F,0xC9,0x2D,0xD0,0x04,0x86,0xB1,0xF0,0x04,0xC9,0x2B,
    0xD0,0x05,0x20,0xBC,0x00,0x90,0x6E,0xC9,0x24,0xD0,0x03,
    0x4C,0x6E,0xDE,0xC9,0x25,0xD0,0x08,0x4C,0x9C,0xDE,0x20,
    0xBC,0x00,0x90,0x5B,0xC9,0x2E,0xF0,0x2E,0xC9,0x45,0xD0,
    0x30,0x20,0xBC,0x00,0x90,0x17,0xC9,0xB7,0xF0,0x0E,0xC9,
    0x2D,0xF0,0x0A,0xC9,0xB6,0xF0,0x08,0xC9,0x2B,0xF0,0x04,
    0xD0,0x07,0x66,0xAB,0x20,0xBC,0x00,0x90,0x5B,0x24,0xAB,
    0x10,0x0E,0xA9,0x00,0x38,0xE5,0xA9,0x4C,0x11,0xDA,0x66,
    0xAA,0x24,0xAA,0x50,0xC3,0xA5,0xA9,0x38,0xE5,0xA8,0x85,
    0xA9,0xF0,0x12,0x10,0x09,0x20,0xF0,0xD7,0xE6,0xA9,0xD0,
    0xF9,0xF0,0x07,0x20,0xD7,0xD7,0xC6,0xA9,0xD0,0xF9,0xA5,
    0xB1,0x30,0x01,0x60,0x4C,0xF4,0xDB,0x48,0x24,0xAA,0x10,
    0x02,0xE6,0xA8,0x20,0xD7,0xD7,0x68,0x29,0x0F,0x20,0x45,
    0xDA,0x4C,0xD2,0xD9,0x48,0x20,0xCA,0xD8,0x68,0x20,0xFA,
    0xD8,0xA5,0xB7,0x45,0xB0,0x85,0xB8,0xA6,0xAC,0x4C,0xC1,
    0xD5,0xA5,0xA9,0xC9,0x0A,0x90,0x09,0xA9,0x64,0x24,0xAB,
    0x30,0x0E,0x4C,0xC3,0xD6,0x0A,0x0A,0x65,0xA9,0x0A,0xA0,
    0x00,0x71,0xC3,0xE9,0x2F,0x85,0xA9,0x4C,0xF8,0xD9,0xA9,
    0xED,0xA0,0xE7,0x20,0xD3,0xC8,0xA5,0x88,0xA6,0x87,0x85,
    0xAD,0x86,0xAE,0xA2,0x90,0x38,0x20,0x07,0xD9,0xA0,0x00,
    0x98,0x20,0xA2,0xDA,0x4C,0xD3,0xC8,0xA0,0x01,0xA9,0x20,
    0x24,0xB0,0x10,0x02,0xA9,0x2D,0x99,0xEF,0x00,0x85,0xB0,
    0x84,0xBA,0xC8,0xA6,0xAC,0xD0,0x05,0xA9,0x30,0x4C,0xAE,
    0xDB,0xA9,0x00,0xE0,0x81,0xB0,0x09,0xA9,0x81,0xA0,0xE1,
    0x20,0x39,0xD7,0xA9,0xFA,0x85,0xA8,0xA9,0x7D,0xA0,0xE1,
    0x20,0x17,0xD9,0xF0,0x1E,0x10,0x12,0xA9,0x79,0xA0,0xE1,
    0x20,0x17,0xD9,0xF0,0x02,0x10,0x0E,0x20,0xD7,0xD7,0xC6,
    0xA8,0xD0,0xEE,0x20,0xF0,0xD7,0xE6,0xA8,0xD0,0xDC,0x20,
    0xBA,0xD5,0x20,0x50,0xD9,0xA2,0x01,0xA5,0xA8,0x18,0x69,
    0x07,0x30,0x09,0xC9,0x08,0xB0,0x06,0x69,0xFF,0xAA,0xA9,
    0x02,0x38,0xE9,0x02,0x85,0xA9,0x86,0xA8,0x8A,0xF0,0x02,
    0x10,0x13,0xA4,0xBA,0xA9,0x2E,0xC8,0x99,0xEF,0x00,0x8A,
    0xF0,0x06,0xA9,0x30,0xC8,0x99,0xEF,0x00,0x84,0xBA,0xA0,
    0x00,0xA2,0x80,0xA5,0xAF,0x18,0x79,0xF7,0xE1,0x85,0xAF,
    0xA5,0xAE,0x79,0xF6,0xE1,0x85,0xAE,0xA5,0xAD,0x79,0xF5,
    0xE1,0x85,0xAD,0xE8,0xB0,0x04,0x10,0xE5,0x30,0x02,0x30,
    0xE1,0x8A,0x90,0x04,0x49,0xFF,0x69,0x0A,0x69,0x2F,0xC8,
    0xC8,0xC8,0x84,0x95,0xA4,0xBA,0xC8,0xAA,0x29,0x7F,0x99,
    0xEF,0x00,0xC6,0xA8,0xD0,0x06,0xA9,0x2E,0xC8,0x99,0xEF,
    0x00,0x84,0xBA,0xA4,0x95,0x8A,0x49,0xFF,0x29,0x80,0xAA,
    0xC0,0x12,0xD0,0xB2,0xA4,0xBA,0xB9,0xEF,0x00,0x88,0xC9,
    0x30,0xF0,0xF8,0xC9,0x2E,0xF0,0x01,0xC8,0xA9,0x2B,0xA6,
    0xA9,0xF0,0x2E,0x10,0x08,0xA9,0x00,0x38,0xE5,0xA9,0xAA,
    0xA9,0x2D,0x99,0xF1,0x00,0xA9,0x45,0x99,0xF0,0x00,0x8A,
    0xA2,0x2F,0x38,0xE8,0xE9,0x0A,0xB0,0xFB,0x69,0x3A,0x99,
    0xF3,0x00,0x8A,0x99,0xF2,0x00,0xA9,0x00,0x99,0xF4,0x00,
    0xF0,0x08,0x99,0xEF,0x00,0xA9,0x00,0x99,0xF0,0x00,0xA9,
    0xF0,0xA0,0x00,0x60,0xF0,0x42,0xA5,0xB3,0xD0,0x03,0x4C,
    0x52,0xD6,0xA2,0x9C,0xA0,0x00,0x20,0xA7,0xD8,0xA5,0xB7,
    0x10,0x0F,0x20,0x81,0xD9,0xA9,0x9C,0xA0,0x00,0x20,0x17,
    0xD9,0xD0,0x03,0x98,0xA4,0x5B,0x20,0xC5,0xD5,0x98,0x48,
    0x20,0xFB,0xD6,0xA9,0x9C,0xA0,0x00,0x20,0x39,0xD7,0x20,
    0xFF,0xDB,0x68,0x4A,0x90,0x0A,0xA5,0xAC,0xF0,0x06,0xA5,
    0xB0,0x49,0xFF,0x85,0xB0,0x60,0xA9,0x85,0xA0,0xE1,0x20,
    0x39,0xD7,0xA5,0xB9,0x69,0x50,0x90,0x03,0x20,0xE1,0xD8,
    0x85,0xA3,0x20,0xCD,0xD8,0xA5,0xAC,0xC9,0x88,0x90,0x03,
    0x20,0xCE,0xD7,0x20,0x81,0xD9,0xA5,0x5B,0x18,0x69,0x81,
    0xF0,0xF3,0x38,0xE9,0x01,0x48,0xA2,0x04,0xB5,0xB3,0xB4,
    0xAC,0x95,0xAC,0x94,0xB3,0xCA,0x10,0xF5,0xA5,0xA3,0x85,
    0xB9,0x20,0xA6,0xD5,0x20,0xF4,0xDB,0xA9,0x89,0xA0,0xE1,
    0x20,0x67,0xDC,0xA9,0x00,0x85,0xB8,0x68,0x4C,0xB3,0xD7,
    0x85,0xBA,0x84,0xBB,0x20,0x9D,0xD8,0xA9,0xA4,0x20,0x39,
    0xD7,0x20,0x6B,0xDC,0xA9,0xA4,0xA0,0x00,0x4C,0x39,0xD7,
    0x85,0xBA,0x84,0xBB,0xA2,0xA8,0x20,0x9F,0xD8,0xB1,0xBA,
    0x85,0xB1,0xA4,0xBA,0xC8,0x98,0xD0,0x02,0xE6,0xBB,0x85,
    0xBA,0xA4,0xBB,0x20,0x39,0xD7,0xA5,0xBA,0xA4,0xBB,0x18,
    0x69,0x04,0x90,0x01,0xC8,0x85,0xBA,0x84,0xBB,0x20,0xBE,
    0xD5,0xA9,0xA8,0xA0,0x00,0xC6,0xB1,0xD0,0xE4,0x60,0xA5,
    0xAC,0xF0,0x07,0xA2,0xD8,0xA0,0x00,0x20,0xA7,0xD8,0xA2,
    0xAF,0xA0,0x13,0x06,0xD9,0x26,0xDA,0x26,0xDB,0x26,0xD8,
    0x90,0x05,0x8A,0x45,0xD9,0x85,0xD9,0x88,0xD0,0xEE,0xA2,
    0x02,0xB5,0xD9,0x95,0xAD,0xCA,0x10,0xF9,0xA9,0x80,0x85,
    0xAC,0x0A,0x85,0xB0,0x4C,0x34,0xD6,0xA9,0xA6,0xA0,0xE1,
    0x20,0xBE,0xD5,0x20,0xCA,0xD8,0xA9,0xBB,0xA0,0xE1,0xA6,
    0xB7,0x20,0xF9,0xD7,0x20,0xCA,0xD8,0x20,0x81,0xD9,0xA9,
    0x00,0x85,0xB8,0x20,0xA6,0xD5,0xA9,0xED,0xA0,0xE1,0x20,
    0xA3,0xD5,0xA5,0xB0,0x48,0x10,0x0D,0x20,0xBA,0xD5,0xA5,
    0xB0,0x30,0x09,0xA5,0x63,0x49,0xFF,0x85,0x63,0x20,0xF4,
    0xDB,0xA9,0xED,0xA0,0xE1,0x20,0xBE,0xD5,0x68,0x10,0x03,
    0x20,0xF4,0xDB,0xA9,0xAA,0xA0,0xE1,0x4C,0x51,0xDC,0x20,
    0x9D,0xD8,0xA9,0x00,0x85,0x63,0x20,0xD8,0xDC,0xA2,0x9C,
    0xA0,0x00,0x20,0xA7,0xD8,0xA9,0xA4,0xA0,0x00,0x20,0x7D,
    0xD8,0xA9,0x00,0x85,0xB0,0xA5,0x63,0x20,0x49,0xDD,0xA9,
    0x9C,0xA0,0x00,0x4C,0x01,0xD8,0x48,0x4C,0x0A,0xDD,0x20,
    0x0A,0x00,0x4C,0xEF,0xCB,0xA5,0xB0,0x48,0x10,0x03,0x20,
    0xF4,0xDB,0xA5,0xAC,0x48,0xC9,0x81,0x90,0x07,0xA9,0xE0,
    0xA0,0xE1,0x20,0x01,0xD8,0xA9,0xBF,0xA0,0xE1,0x20,0x51,
    0xDC,0x68,0xC9,0x81,0x90,0x07,0xA9,0xA6,0xA0,0xE1,0x20,
    0xA3,0xD5,0x68,0x10,0x16,0x4C,0xF4,0xDB,0x20,0xD8,0xD4,
    0xE0,0x08,0xB0,0x20,0xA9,0x00,0x38,0x2A,0xCA,0x10,0xFC,
    0xE8,0x01,0x11,0x81,0x11,0x60,0x20,0xD8,0xD4,0xE0,0x08,
    0xB0,0x0C,0xA9,0xFF,0x2A,0xCA,0x10,0xFC,0xE8,0x21,0x11,
    0x81,0x11,0x60,0x4C,0x29,0xCF,0x20,0xBC,0x00,0x20,0xD8,
    0xD4,0xE0,0x08,0xB0,0xF3,0x20,0xC2,0x00,0xC9,0x29,0xF0,
    0x03,0x4C,0x02,0xCC,0x20,0xBC,0x00,0xA9,0x00,0x38,0x2A,
    0xCA,0x10,0xFC,0xE8,0x21,0x11,0xF0,0x02,0xA9,0xFF,0x4C,
    0xFA,0xD8,0xE0,0x19,0xB0,0x48,0x86,0x78,0xA9,0x18,0x20,
    0x3A,0xD1,0xA0,0x17,0xA2,0x18,0x46,0x11,0x66,0x12,0x66,
    0x13,0x8A,0x2A,0x91,0xAD,0x88,0x10,0xF3,0xA5,0x78,0xF0,
    0x0A,0xAA,0x38,0x49,0xFF,0x69,0x18,0xF0,0x1C,0xD0,0x0F,
    0xA8,0xB1,0xAD,0xC9,0x30,0xD0,0x07,0xCA,0xF0,0x03,0xC8,
    0x10,0xF4,0xE8,0x98,0x18,0x65,0xAD,0x85,0xAD,0xA9,0x00,
    0x65,0xAE,0x85,0xAE,0x86,0xAC,0x20,0xBC,0x00,0x4C,0x85,
    0xD1,0x4C,0x29,0xCF,0xE0,0x07,0xB0,0xF9,0x86,0x78,0xA9,
    0x06,0x20,0x3A,0xD1,0xA0,0x05,0xF8,0xA5,0x13,0x20,0x51,
    0xDE,0xA5,0x12,0x20,0x51,0xDE,0xA5,0x11,0x20,0x51,0xDE,
    0xD8,0xA2,0x06,0xA5,0x78,0xF0,0xB7,0xAA,0x38,0x49,0xFF,
    0x69,0x06,0xF0,0xC9,0xD0,0xBC,0xAA,0x29,0x0F,0x20,0x5C,
    0xDE,0x8A,0x4A,0x4A,0x4A,0x4A,0xC9,0x0A,0x69,0x30,0x91,
    0xAD,0x88,0x60,0x85,0xAC,0xA9,0x00,0x85,0xB8,0x8A,0x20,
    0x45,0xDA,0x20,0xBC,0x00,0x90,0x0A,0x09,0x20,0xE9,0x61,
    0xC9,0x06,0xB0,0x2A,0x69,0x0A,0x29,0x0F,0xAA,0xA5,0xAC,
    0xF0,0xE4,0x69,0x04,0x90,0xDC,0x4C,0xC3,0xD6,0xAA,0xA5,
    0xAC,0xF0,0x06,0xE6,0xAC,0xF0,0xF4,0xA9,0x00,0x85,0xB8,
    0x8A,0x20,0x45,0xDA,0x20,0xBC,0x00,0x49,0x30,0xC9,0x02,
    0x90,0xE6,0x4C,0x2A,0xDA,0xAD,0x00,0x02,0xD0,0x18,0x20,
    0xEA,0xE0,0x90,0x0B,0x8D,0x01,0x02,0xA2,0x20,0x8E,0x02,
    0x02,0x4C,0x17,0xC5,0xAE,0x02,0x02,0xF0,0x03,0xCE,0x02,
    0x02,0xA2,0xDC,0x20,0xD0,0xDE,0xA2,0xDF,0x20,0xD0,0xDE,
    0x60,0xB5,0x00,0x10,0xFB,0x0A,0x29,0x40,0xF0,0xF6,0x95,
    0x00,0x8A,0xA8,0x68,0x68,0xA9,0x05,0x20,0x03,0xC1,0xA5,
    0xC4,0x48,0xA5,0xC3,0x48,0xA5,0x88,0x48,0xA5,0x87,0x48,
    0xA9,0x8D,0x48,0xB9,0x01,0x00,0x85,0xC3,0xB9,0x02,0x00,
    0x85,0xC4,0x4C,0xBC,0xC4,0x20,0xEA,0xE0,0xB0,0x09,0xAD,
    0x02,0x02,0xF0,0x09,0xAD,0x01,0x02,0x38,0xA2,0x00,0x8E,
    0x02,0x02,0x60,0xA2,0xDF,0x2C,0xA2,0xDC,0xC9,0x93,0xF0,
    0x11,0xC9,0xB5,0xF0,0x07,0x49,0xA2,0xF0,0x0E,0x4C,0x02,
    0xCC,0xA9,0x7F,0x35,0x00,0x10,0x05,0xB5,0x00,0x0A,0x15,
    0x00,0x95,0x00,0x4C,0xBC,0x00,0x58,0xA2,0xDF,0x2C,0xA2,
    0xDC,0x86,0x78,0x20,0xBC,0x00,0x20,0x55,0xC7,0xA5,0x79,
    0xA6,0x7A,0x20,0x28,0xC3,0xB0,0x03,0x4C,0x72,0xC6,0xA6,
    0x78,0xA5,0xAA,0xE9,0x01,0x95,0x01,0xA5,0xAB,0xE9,0x00,
    0x95,0x02,0xA9,0xC0,0x95,0x00,0x60,0xD0,0xFD,0xA5,0xDF,
    0x0A,0x05,0xDF,0x85,0xDF,0x4C,0x79,0xC6,0xD0,0xF1,0xA5,
    0xDC,0x0A,0x05,0xDC,0x85,0xDC,0x4C,0x79,0xC6,0x20,0xE9,
    0xCA,0x4C,0xD0,0xCA,0x20,0xB1,0xDF,0x10,0xFB,0xA5,0xB4,
    0x09,0x80,0x85,0xB4,0x20,0xC3,0xD5,0xF0,0xF0,0x20,0xB1,
    0xDF,0x30,0xFB,0xF0,0xF9,0xA5,0xB4,0x09,0x80,0x85,0xB4,
    0x20,0xC3,0xD5,0xF0,0xEE,0xC9,0x29,0xD0,0x05,0x68,0x68,
    0x4C,0xBC,0x00,0x4C,0x02,0xCC,0x20,0xC2,0x00,0xC9,0x2C,
    0xD0,0xED,0x20,0xD9,0xD8,0xA5,0xB0,0x09,0x7F,0x25,0xAD,
    0x48,0xA5,0xAE,0x48,0xA5,0xAF,0x48,0xA5,0xAC,0x48,0x20,
    0xBC,0x00,0x20,0xCD,0xCA,0x68,0x85,0xB3,0x68,0x85,0xB6,
    0x68,0x85,0xB5,0x68,0x85,0xB4,0x85,0xB7,0xA9,0xB3,0xA0,
    0x00,0x4C,0x17,0xD9,0xC9,0x2C,0xF0,0x1B,0x20,0x8C,0xD4,
    0x8A,0xF0,0x0A,0xE0,0x10,0x90,0x45,0xE4,0x64,0xB0,0x02,
    0x86,0x64,0x86,0x0F,0x20,0xC2,0x00,0xF0,0x1A,0xC9,0x2C,
    0xD0,0xA9,0x20,0x89,0xD4,0x8A,0x30,0x2E,0xE0,0x01,0x90,
    0x2A,0xA5,0x0F,0xF0,0x06,0xE4,0x0F,0xF0,0x02,0xB0,0x20,
    0x86,0x64,0xA5,0x0F,0xF0,0x06,0xC5,0x64,0xB0,0x03,0x85,
    0x64,0x38,0xE5,0x64,0xB0,0xFC,0x65,0x64,0x18,0x65,0x64,
    0x85,0x10,0xA5,0x0F,0x38,0xE5,0x10,0x85,0x10,0x60,0x4C,
    0x29,0xCF,0xA5,0xB0,0x30,0xF9,0xA5,0xAC,0xF0,0xF4,0x20,
    0xCA,0xD8,0xA9,0x00,0x85,0x77,0x85,0x76,0x85,0x75,0x85,
    0x78,0x85,0xAF,0x85,0xAE,0x85,0xAD,0xA2,0x18,0xA5,0xB3,
    0x4A,0xB0,0x0E,0x06,0xB6,0x26,0xB5,0x26,0xB4,0x26,0x77,
    0x26,0x76,0x26,0x75,0x26,0x78,0x06,0xB6,0x26,0xB5,0x26,
    0xB4,0x26,0x77,0x26,0x76,0x26,0x75,0x26,0x78,0x06,0xAF,
    0x26,0xAE,0x26,0xAD,0xA5,0xAF,0x2A,0x85,0x5B,0xA5,0xAE,
    0x2A,0x85,0x5C,0xA5,0xAD,0x2A,0x85,0x5D,0xA9,0x00,0x2A,
    0x85,0x5E,0xA5,0x77,0xE5,0x5B,0x85,0x5B,0xA5,0x76,0xE5,
    0x5C,0x85,0x5C,0xA5,0x75,0xE5,0x5D,0xA8,0xA5,0x78,0xE5,
    0x5E,0x90,0x0E,0x85,0x78,0x84,0x75,0xA5,0x5C,0x85,0x76,
    0xA5,0x5B,0x85,0x77,0xE6,0xAF,0xCA,0xD0,0xA2,0x38,0xA5,
    0xB3,0xE9,0x80,0x6A,0x69,0x00,0x85,0xAC,0x4C,0x34,0xD6,
    0x20,0xBC,0x00,0x20,0xAA,0xCD,0x20,0xEF,0xCB,0xA4,0x95,
    0xA5,0x96,0x4C,0x59,0xD0,0xA9,0xBB,0xA0,0xE1,0x20,0x7D,
    0xD8,0xC6,0xAC,0x60,0xA9,0xBB,0xA0,0xE1,0x4C,0x7D,0xD8,
    0x6C,0x05,0x02,0x6C,0x07,0x02,0x6C,0x09,0x02,0x6C,0x0B,
    0x02,0x00,0x00,0x00,0xA8,0xDE,0xE6,0xC3,0xD0,0x02,0xE6,
    0xC4,0xAD,0xFF,0xFF,0xC9,0xAC,0xF0,0x0E,0xC9,0x3A,0xB0,
    0x0A,0xC9,0x20,0xF0,0xEB,0x38,0xE9,0x30,0x38,0xE9,0xD0,
    0x60,0x4C,0x00,0xC0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x4C,0x29,0xCF,0x00,0x00,0x00,0xF2,0x00,0x03,0x0D,0x0A,
    0x4D,0x65,0x6D,0x6F,0x72,0x79,0x20,0x73,0x69,0x7A,0x65,
    0x20,0x00,0x20,0x42,0x79,0x74,0x65,0x73,0x20,0x66,0x72,
    0x65,0x65,0x0D,0x0A,0x0A,0x45,0x6E,0x68,0x61,0x6E,0x63,
    0x65,0x64,0x20,0x42,0x41,0x53,0x49,0x43,0x20,0x32,0x2E,
    0x32,0x32,0x0A,0x00,0x02,0x80,0x19,0x56,0x62,0x80,0x76,
    0x22,0xF3,0x82,0x38,0xAA,0x40,0x80,0x35,0x04,0xF3,0x81,
    0x35,0x04,0xF3,0x80,0x80,0x00,0x00,0x80,0x31,0x72,0x18,
    0x91,0x43,0x4F,0xF8,0x94,0x74,0x23,0xF7,0x94,0x74,0x24,
    0x00,0x81,0x38,0xAA,0x3B,0x06,0x74,0x63,0x90,0x8C,0x77,
    0x23,0x0C,0xAB,0x7A,0x1E,0x94,0x00,0x7C,0x63,0x42,0x80,
    0x7E,0x75,0xFE,0xD0,0x80,0x31,0x72,0x15,0x81,0x00,0x00,
    0x00,0x81,0x49,0x0F,0xDB,0x04,0x86,0x1E,0xD7,0xFB,0x87,
    0x99,0x26,0x65,0x87,0x23,0x34,0x58,0x86,0xA5,0x5D,0xE1,
    0x83,0x49,0x0F,0xDB,0x08,0x78,0x3A,0xC5,0x37,0x7B,0x83,
    0xA2,0x5C,0x7C,0x2E,0xDD,0x4D,0x7D,0x99,0xB0,0x1E,0x7D,
    0x59,0xED,0x24,0x7E,0x91,0x72,0x00,0x7E,0x4C,0xB9,0x73,
    0x7F,0xAA,0xAA,0x53,0x81,0x00,0x00,0x00,0x81,0x80,0x00,
    0x00,0x90,0x80,0x00,0x00,0x00,0x7F,0x00,0x00,0x00,0x84,
    0x20,0x00,0x00,0xFE,0x79,0x60,0x00,0x27,0x10,0xFF,0xFC,
    0x18,0x00,0x00,0x64,0xFF,0xFF,0xF6,0x00,0x00,0x01,0x1A,
    0xC5,0x57,0xC4,0x64,0xCA,0x8B,0xC6,0x46,0xC9,0x5E,0xCD,
    0x66,0xC9,0xB3,0xC7,0x85,0xC7,0xEA,0xC5,0xA9,0xC5,0xB9,
    0xC6,0x41,0xC5,0xCD,0xC5,0x64,0xDF,0x70,0xDF,0x76,0xC6,
    0x1D,0xC7,0x18,0xC5,0x26,0xC7,0x7F,0xC5,0x88,0xC7,0x88,
    0xD5,0xEF,0xE0,0xF2,0xE0,0x73,0xD0,0x0D,0xD5,0x29,0xD5,
    0x76,0xD5,0xB3,0xC5,0x1C,0xC6,0x54,0xC8,0x85,0xC5,0xA2,
    0xC3,0x9F,0xC3,0x4D,0xC3,0xE5,0xDF,0x23,0xC8,0x4E,0xD5,
    0x81,0xDD,0x95,0xDD,0x13,0xDF,0x16,0xDF,0x64,0xCC,0x64,
    0xCC,0x64,0xCC,0xE8,0xCA,0xEB,0xCB,0xEB,0xCB,0x64,0xCC,
    0x64,0xCC,0x64,0xCC,0x64,0xCC,0x64,0xCC,0x64,0xCC,0x64,
    0xCC,0x64,0xCC,0x64,0xCC,0x64,0xCC,0x00,0x00,0x5E,0xCC,
    0x64,0xCC,0x5E,0xCC,0x5E,0xCC,0x5E,0xCC,0x5E,0xCC,0x64,
    0xCC,0x8B,0xCC,0x8B,0xCC,0x00,0x00,0x7C,0xDF,0x7C,0xDF,
    0x6A,0xCC,0x6A,0xCC,0x00,0x00,0x6F,0xCC,0x6F,0xCC,0x6F,
    0xCC,0xF6,0xD8,0x80,0xD9,0x13,0xD9,0x4C,0xDD,0x44,0xD0,
    0x63,0xD0,0x3B,0xE0,0x9C,0xDC,0xFA,0xD6,0xFE,0xDB,0xD0,
    0xDC,0xD7,0xDC,0x20,0xDD,0x52,0xDD,0x02,0xD5,0x16,0xD5,
    0x54,0xD4,0x6D,0xD4,0x25,0xD1,0x9A,0xD4,0x78,0xD4,0x32,
    0xD4,0x11,0xD4,0x83,0xD3,0x22,0xDE,0xD3,0xDD,0xAB,0xDD,
    0x82,0xDF,0x92,0xDF,0xD8,0xE0,0xE2,0xE0,0xC8,0xE0,0x95,
    0xD3,0x9E,0xD3,0xCD,0xD3,0x79,0xC0,0xD5,0x79,0xA5,0xD5,
    0x7B,0x3B,0xD7,0x7B,0x03,0xD8,0x7F,0xBA,0xDB,0x50,0xD4,
    0xCC,0x46,0xBA,0xCC,0x46,0xC7,0xCC,0x56,0x80,0xCD,0x56,
    0x68,0xCD,0x7D,0xF3,0xDB,0x5A,0x16,0xCC,0x64,0xF5,0xCC,
    0x2A,0x2B,0x2D,0x2F,0x3C,0x3D,0x3E,0x3F,0x41,0x42,0x43,
    0x44,0x45,0x46,0x47,0x48,0x49,0x4C,0x4D,0x4E,0x4F,0x50,
    0x52,0x53,0x54,0x55,0x56,0x57,0x5E,0x00,0x68,0xE3,0x6A,
    0xE3,0x6C,0xE3,0x6E,0xE3,0x70,0xE3,0x74,0xE3,0x76,0xE3,
    0x7A,0xE3,0x7C,0xE3,0x8C,0xE3,0xA5,0xE3,0xBC,0xE3,0xD5,
    0xE3,0xE4,0xE3,0xEE,0xE3,0xFB,0xE3,0x01,0xE4,0x13,0xE4,
    0x38,0xE4,0x46,0xE4,0x58,0xE4,0x60,0xE4,0x75,0xE4,0xA4,
    0xE4,0xCF,0xE4,0xE3,0xE4,0xF4,0xE4,0x00,0xE5,0x0F,0xE5,
    0xB8,0x00,0xB6,0x00,0xB7,0x00,0xB9,0x00,0x3C,0xBF,0xC2,
    0x00,0xC1,0x00,0x3E,0xBE,0xC0,0x00,0x9F,0x00,0x42,0x53,
    0x28,0xC5,0x4E,0x44,0xBB,0x53,0x43,0x28,0xD7,0x54,0x4E,
    0x28,0xD0,0x00,0x49,0x4E,0x24,0x28,0xDC,0x49,0x54,0x43,
    0x4C,0x52,0xA8,0x49,0x54,0x53,0x45,0x54,0xA7,0x49,0x54,
    0x54,0x53,0x54,0x28,0xDD,0x00,0x41,0x4C,0x4C,0x9C,0x48,
    0x52,0x24,0x28,0xDA,0x4C,0x45,0x41,0x52,0xA2,0x4F,0x4E,
    0x54,0xA0,0x4F,0x53,0x28,0xCD,0x00,0x41,0x54,0x41,0x83,
    0x45,0x43,0x88,0x45,0x45,0x4B,0x28,0xD2,0x45,0x46,0x99,
    0x49,0x4D,0x85,0x4F,0x4B,0x45,0x9B,0x4F,0x9D,0x00,0x4C,
    0x53,0x45,0xAC,0x4E,0x44,0x80,0x4F,0x52,0xBC,0x58,0x50,
    0x28,0xCC,0x00,0x4E,0xAE,0x4F,0x52,0x81,0x52,0x45,0x28,
    0xC7,0x00,0x45,0x54,0xA5,0x4F,0x53,0x55,0x42,0x8D,0x4F,
    0x54,0x4F,0x89,0x00,0x45,0x58,0x24,0x28,0xDB,0x00,0x46,
    0x8B,0x4E,0x43,0x95,0x4E,0x50,0x55,0x54,0x84,0x4E,0x54,
    0x28,0xC4,0x52,0x51,0xA9,0x00,0x43,0x41,0x53,0x45,0x24,
    0x28,0xD9,0x45,0x46,0x54,0x24,0x28,0xE3,0x45,0x4E,0x28,
    0xD4,0x45,0x54,0x87,0x49,0x53,0x54,0xA1,0x4F,0x41,0x44,
    0x97,0x4F,0x47,0x28,0xCB,0x4F,0x4F,0x50,0x9E,0x00,0x41,
    0x58,0x28,0xDE,0x49,0x44,0x24,0x28,0xE5,0x49,0x4E,0x28,
    0xDF,0x00,0x45,0x57,0xA3,0x45,0x58,0x54,0x82,0x4D,0x49,
    0xAA,0x4F,0x54,0xB1,0x55,0x4C,0x4C,0x94,0x00,0x46,0x46,
    0xB5,0x4E,0x93,0x52,0xBD,0x00,0x45,0x45,0x4B,0x28,0xD1,
    0x49,0xE0,0x4F,0x4B,0x45,0x9A,0x4F,0x53,0x28,0xC8,0x52,
    0x49,0x4E,0x54,0x9F,0x00,0x45,0x41,0x44,0x86,0x45,0x4D,
    0x91,0x45,0x53,0x54,0x4F,0x52,0x45,0x8C,0x45,0x54,0x49,
    0x52,0x51,0x8E,0x45,0x54,0x4E,0x4D,0x49,0x8F,0x45,0x54,
    0x55,0x52,0x4E,0x90,0x49,0x47,0x48,0x54,0x24,0x28,0xE4,
    0x4E,0x44,0x28,0xCA,0x55,0x4E,0x8A,0x00,0x41,0x44,0x44,
    0x28,0xD3,0x41,0x56,0x45,0x98,0x47,0x4E,0x28,0xC3,0x49,
    0x4E,0x28,0xCE,0x50,0x43,0x28,0xAF,0x51,0x52,0x28,0xC9,
    0x54,0x45,0x50,0xB2,0x54,0x4F,0x50,0x92,0x54,0x52,0x24,
    0x28,0xD5,0x57,0x41,0x50,0xA6,0x00,0x41,0x42,0x28,0xAB,
    0x41,0x4E,0x28,0xCF,0x48,0x45,0x4E,0xB0,0x4F,0xAD,0x57,
    0x4F,0x50,0x49,0xE1,0x00,0x43,0x41,0x53,0x45,0x24,0x28,
    0xD8,0x4E,0x54,0x49,0x4C,0xB3,0x53,0x52,0x28,0xC6,0x00,
    0x41,0x4C,0x28,0xD6,0x41,0x52,0x50,0x54,0x52,0x28,0xE2,
    0x00,0x41,0x49,0x54,0x96,0x48,0x49,0x4C,0x45,0xB4,0x49,
    0x44,0x54,0x48,0xA4,0x00,0xBA,0x00,0x03,0x45,0xD9,0xE3,
    0x03,0x46,0xE6,0xE3,0x04,0x4E,0x49,0xE4,0x04,0x44,0xBC,
    0xE3,0x05,0x49,0x06,0xE4,0x03,0x44,0xCB,0xE3,0x04,0x52,
    0x75,0xE4,0x03,0x4C,0x24,0xE4,0x03,0x44,0xC0,0xE3,0x04,
    0x47,0xF6,0xE3,0x03,0x52,0xA0,0xE4,0x02,0x49,0x01,0xE4,
    0x07,0x52,0x7C,0xE4,0x05,0x47,0xF1,0xE3,0x06,0x52,0x83,
    0xE4,0x06,0x52,0x89,0xE4,0x06,0x52,0x8F,0xE4,0x03,0x52,
    0x79,0xE4,0x04,0x53,0xC1,0xE4,0x02,0x4F,0x5B,0xE4,0x04,
    0x4E,0x53,0xE4,0x03,0x49,0x03,0xE4,0x04,0x57,0x00,0xE5,
    0x04,0x4C,0x2B,0xE4,0x04,0x53,0xA9,0xE4,0x03,0x44,0xC8,
    0xE3,0x04,0x50,0x67,0xE4,0x04,0x44,0xCE,0xE3,0x04,0x43,
    0xA5,0xE3,0x02,0x44,0xD2,0xE3,0x04,0x4C,0x33,0xE4,0x05,
    0x50,0x6F,0xE4,0x04,0x43,0xB3,0xE3,0x04,0x4C,0x27,0xE4,
    0x05,0x43,0xAE,0xE3,0x03,0x4E,0x46,0xE4,0x05,0x57,0x09,
    0xE5,0x03,0x47,0xEE,0xE3,0x04,0x53,0xCA,0xE4,0x06,0x42,
    0x97,0xE3,0x06,0x42,0x91,0xE3,0x03,0x49,0x0F,0xE4,0x03,
    0x4E,0x4D,0xE4,0x04,0x54,0xCF,0xE4,0x04,0x45,0xD5,0xE3,
    0x02,0x54,0xDB,0xE4,0x02,0x46,0xE4,0xE3,0x04,0x53,0xB5,
    0xE4,0x04,0x54,0xD7,0xE4,0x03,0x4E,0x50,0xE4,0x04,0x53,
    0xBD,0xE4,0x05,0x55,0xEA,0xE4,0x05,0x57,0x04,0xE5,0x03,
    0x4F,0x58,0xE4,0x01,0x2B,0x00,0x00,0x01,0x2D,0x00,0x00,
    0x01,0x2A,0x00,0x00,0x01,0x2F,0x00,0x00,0x01,0x5E,0x00,
    0x00,0x03,0x41,0x80,0xE3,0x03,0x45,0xDC,0xE3,0x02,0x4F,
    0x5D,0xE4,0x02,0x3E,0x76,0xE3,0x02,0x3C,0x70,0xE3,0x01,
    0x3E,0x00,0x00,0x01,0x3D,0x00,0x00,0x01,0x3C,0x00,0x00,
    0x04,0x53,0xAD,0xE4,0x04,0x49,0x0B,0xE4,0x04,0x41,0x7C,
    0xE3,0x04,0x55,0xEF,0xE4,0x04,0x46,0xE9,0xE3,0x04,0x50,
    0x6B,0xE4,0x04,0x53,0xB9,0xE4,0x04,0x52,0x9C,0xE4,0x04,
    0x4C,0x2F,0xE4,0x04,0x45,0xDF,0xE3,0x04,0x43,0xB7,0xE3,
    0x04,0x53,0xB1,0xE4,0x04,0x54,0xD3,0xE4,0x04,0x41,0x87,
    0xE3,0x05,0x50,0x60,0xE4,0x05,0x44,0xC3,0xE3,0x05,0x53,
    0xA4,0xE4,0x04,0x4C,0x20,0xE4,0x05,0x53,0xC5,0xE4,0x04,
    0x56,0xF4,0xE4,0x04,0x41,0x83,0xE3,0x07,0x55,0xE3,0xE4,
    0x07,0x4C,0x13,0xE4,0x05,0x43,0xA9,0xE3,0x05,0x48,0xFB,
    0xE3,0x05,0x42,0x8C,0xE3,0x07,0x42,0x9D,0xE3,0x04,0x4D,
    0x38,0xE4,0x04,0x4D,0x41,0xE4,0x02,0x50,0x65,0xE4,0x05,
    0x54,0xDD,0xE4,0x07,0x56,0xF8,0xE4,0x06,0x4C,0x1A,0xE4,
    0x07,0x52,0x95,0xE4,0x05,0x4D,0x3C,0xE4,0xCD,0xE6,0xDE,
    0xE6,0xE5,0xE6,0xFA,0xE6,0x06,0xE7,0x14,0xE7,0x1D,0xE7,
    0x2B,0xE7,0x3F,0xE7,0x4C,0xE7,0x5D,0xE7,0x6C,0xE7,0x7B,
    0xE7,0x89,0xE7,0x99,0xE7,0xAC,0xE7,0xBB,0xE7,0xCE,0xE7,
    0x4E,0x45,0x58,0x54,0x20,0x77,0x69,0x74,0x68,0x6F,0x75,
    0x74,0x20,0x46,0x4F,0x52,0x00,0x53,0x79,0x6E,0x74,0x61,
    0x78,0x00,0x52,0x45,0x54,0x55,0x52,0x4E,0x20,0x77,0x69,
    0x74,0x68,0x6F,0x75,0x74,0x20,0x47,0x4F,0x53,0x55,0x42,
    0x00,0x4F,0x75,0x74,0x20,0x6F,0x66,0x20,0x44,0x41,0x54,
    0x41,0x00,0x46,0x75,0x6E,0x63,0x74,0x69,0x6F,0x6E,0x20,
    0x63,0x61,0x6C,0x6C,0x00,0x4F,0x76,0x65,0x72,0x66,0x6C,
    0x6F,0x77,0x00,0x4F,0x75,0x74,0x20,0x6F,0x66,0x20,0x6D,
    0x65,0x6D,0x6F,0x72,0x79,0x00,0x55,0x6E,0x64,0x65,0x66,
    0x69,0x6E,0x65,0x64,0x20,0x73,0x74,0x61,0x74,0x65,0x6D,
    0x65,0x6E,0x74,0x00,0x41,0x72,0x72,0x61,0x79,0x20,0x62,
    0x6F,0x75,0x6E,0x64,0x73,0x00,0x44,0x6F,0x75,0x62,0x6C,
    0x65,0x20,0x64,0x69,0x6D,0x65,0x6E,0x73,0x69,0x6F,0x6E,
    0x00,0x44,0x69,0x76,0x69,0x64,0x65,0x20,0x62,0x79,0x20,
    0x7A,0x65,0x72,0x6F,0x00,0x49,0x6C,0x6C,0x65,0x67,0x61,
    0x6C,0x20,0x64,0x69,0x72,0x65,0x63,0x74,0x00,0x54,0x79,
    0x70,0x65,0x20,0x6D,0x69,0x73,0x6D,0x61,0x74,0x63,0x68,
    0x00,0x53,0x74,0x72,0x69,0x6E,0x67,0x20,0x74,0x6F,0x6F,
    0x20,0x6C,0x6F,0x6E,0x67,0x00,0x53,0x74,0x72,0x69,0x6E,
    0x67,0x20,0x74,0x6F,0x6F,0x20,0x63,0x6F,0x6D,0x70,0x6C,
    0x65,0x78,0x00,0x43,0x61,0x6E,0x27,0x74,0x20,0x63,0x6F,
    0x6E,0x74,0x69,0x6E,0x75,0x65,0x00,0x55,0x6E,0x64,0x65,
    0x66,0x69,0x6E,0x65,0x64,0x20,0x66,0x75,0x6E,0x63,0x74,
    0x69,0x6F,0x6E,0x00,0x4C,0x4F,0x4F,0x50,0x20,0x77,0x69,
    0x74,0x68,0x6F,0x75,0x74,0x20,0x44,0x4F,0x00,0x0D,0x0A,
    0x42,0x72,0x65,0x61,0x6B,0x00,0x20,0x45,0x72,0x72,0x6F,
    0x72,0x00,0x20,0x69,0x6E,0x20,0x6C,0x69,0x6E,0x65,0x20,
    0x00,0x0D,0x0A,0x52,0x65,0x61,0x64,0x79,0x0D,0x0A,0x00,
    0x20,0x45,0x78,0x74,0x72,0x61,0x20,0x69,0x67,0x6E,0x6F,
    0x72,0x65,0x64,0x0D,0x0A,0x00,0x20,0x52,0x65,0x64,0x6F,
    0x20,0x66,0x72,0x6F,0x6D,0x20,0x73,0x74,0x61,0x72,0x74,
    0x0D,0x0A,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00
};

uint8_t BIOStop[256] = {
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xD8,0xA2,0xFF,0x9A,
    0xA0,0x1C,0xB9,0xBB,0xFF,0x99,0x04,0x02,0x88,0xD0,0xF7,
    0xB9,0xD8,0xFF,0xF0,0x06,0x20,0xED,0xE0,0xC8,0xD0,0xF5,
    0x20,0xEA,0xE0,0x90,0xFB,0x29,0xDF,0xC9,0x57,0xF0,0x07,
    0xC9,0x43,0xD0,0xD7,0x4C,0x00,0xC0,0x4C,0x00,0x00,0x8D,
    0x01,0xF0,0x60,0xAD,0x04,0xF0,0xF0,0x02,0x38,0x60,0x18,
    0x60,0xB3,0xFF,0xAF,0xFF,0xBB,0xFF,0xBB,0xFF,0x48,0xA5,
    0xDF,0x4A,0x05,0xDF,0x85,0xDF,0x68,0x40,0x48,0xA5,0xDC,
    0x4A,0x05,0xDC,0x85,0xDC,0x68,0x40,0x0D,0x0A,0x36,0x35,
    0x30,0x32,0x20,0x45,0x68,0x42,0x41,0x53,0x49,0x43,0x20,
    0x5B,0x43,0x5D,0x6F,0x6C,0x64,0x2F,0x5B,0x57,0x5D,0x61,
    0x72,0x6D,0x20,0x3F,0x00,0x00,0x00,0x00,0x17,0x02,0x80,
    0xFF,0x0D,0x02
};

uint8_t read6502(uint16_t address)
{
    uint16_t BIOSaddr;
    uint8_t tempval = 0;

    if (address == 0xF004) { //EhBASIC simulated ASIC input
        tempval = getkey();
        clearkey();
        return(tempval);
    }

    if (address >= 0xC000) {
        BIOSaddr = address - 0xC000;
        if (BIOSaddr < 0x2900) return ( BIOS[ BIOSaddr ] );
        if (BIOSaddr >= 0x3F00) return ( BIOStop[ BIOSaddr - 0x3F00 ] );
    }

    if (address < RAM_SIZE) return(RAM[address]);
    return(0);
}

void write6502(uint16_t address, uint8_t value)
{
    if (address < RAM_SIZE) RAM[address] = value;
    if (address == 0xF001) { //EhBASIC simulated ASIC output
        serout(value);
    }
}

//a few general functions used by various other functions
void push16(uint16_t pushval)
{
    write6502(BASE_STACK + sp, (pushval >> 8) & 0xFF);
    write6502(BASE_STACK + ((sp - 1) & 0xFF), pushval & 0xFF);
    sp -= 2;
}

void push8(uint8_t pushval)
{
    write6502(BASE_STACK + sp--, pushval);
}

uint16_t pull16()
{
    uint16_t temp16;
    temp16 = read6502(BASE_STACK + ((sp + 1) & 0xFF)) | ((uint16_t)read6502(BASE_STACK + ((sp + 2) & 0xFF)) << 8);
    sp += 2;
    return(temp16);
}

uint8_t pull8()
{
    return (read6502(BASE_STACK + ++sp));
}

void reset6502()
{
    pc = (uint16_t)read6502(0xFFFC) | ((uint16_t)read6502(0xFFFD) << 8);
    a = 0;
    x = 0;
    y = 0;
    sp = 0xFD;
    cpustatus |= FLAG_CONSTANT;
}

//addressing mode functions, calculates effective addresses
void imp()   //implied
{
}

void acc()   //accumulator
{
    useaccum = 1;
}

void imm()   //immediate
{
    ea = pc++;
}

void zp()   //zero-page
{
    ea = (uint16_t)read6502((uint16_t)pc++);
}

void zpx()   //zero-page,X
{
    ea = ((uint16_t)read6502((uint16_t)pc++) + (uint16_t)x) & 0xFF; //zero-page wraparound
}

void zpy()   //zero-page,Y
{
    ea = ((uint16_t)read6502((uint16_t)pc++) + (uint16_t)y) & 0xFF; //zero-page wraparound
}

void rel()   //relative for branch ops (8-bit immediate value, sign-extended)
{
    reladdr = (uint16_t)read6502(pc++);
    if (reladdr & 0x80) reladdr |= 0xFF00;
}

void abso()   //absolute
{
    ea = (uint16_t)read6502(pc) | ((uint16_t)read6502(pc+1) << 8);
    pc += 2;
}

void absx()   //absolute,X
{
    uint16_t startpage;
    ea = ((uint16_t)read6502(pc) | ((uint16_t)read6502(pc+1) << 8));
    startpage = ea & 0xFF00;
    ea += (uint16_t)x;

    pc += 2;
}

void absy()   //absolute,Y
{
    uint16_t startpage;
    ea = ((uint16_t)read6502(pc) | ((uint16_t)read6502(pc+1) << 8));
    startpage = ea & 0xFF00;
    ea += (uint16_t)y;

    pc += 2;
}

void ind()   //indirect
{
    uint16_t eahelp, eahelp2;
    eahelp = (uint16_t)read6502(pc) | (uint16_t)((uint16_t)read6502(pc+1) << 8);
    eahelp2 = (eahelp & 0xFF00) | ((eahelp + 1) & 0x00FF); //replicate 6502 page-boundary wraparound bug
    ea = (uint16_t)read6502(eahelp) | ((uint16_t)read6502(eahelp2) << 8);
    pc += 2;
}

void indx()   // (indirect,X)
{
    uint16_t eahelp;
    eahelp = (uint16_t)(((uint16_t)read6502(pc++) + (uint16_t)x) & 0xFF); //zero-page wraparound for table pointer
    ea = (uint16_t)read6502(eahelp & 0x00FF) | ((uint16_t)read6502((eahelp+1) & 0x00FF) << 8);
}

void indy()   // (indirect),Y
{
    uint16_t eahelp, eahelp2, startpage;
    eahelp = (uint16_t)read6502(pc++);
    eahelp2 = (eahelp & 0xFF00) | ((eahelp + 1) & 0x00FF); //zero-page wraparound
    ea = (uint16_t)read6502(eahelp) | ((uint16_t)read6502(eahelp2) << 8);
    startpage = ea & 0xFF00;
    ea += (uint16_t)y;

}

static uint16_t getvalue()
{
    if (useaccum) return((uint16_t)a);
    else return((uint16_t)read6502(ea));
}

static uint16_t getvalue16()
{
    return((uint16_t)read6502(ea) | ((uint16_t)read6502(ea+1) << 8));
}

void putvalue(uint16_t saveval)
{
    if (useaccum) a = (uint8_t)(saveval & 0x00FF);
    else write6502(ea, (saveval & 0x00FF));
}


//instruction handler functions
void adc()
{
    value = getvalue();
    result = (uint16_t)a + value + (uint16_t)(cpustatus & FLAG_CARRY);

    carrycalc(result);
    zerocalc(result);
    overflowcalc(result, a, value);
    signcalc(result);

#ifndef NES_CPU
    if (cpustatus & FLAG_DECIMAL) {
        clearcarry();

        if ((a & 0x0F) > 0x09) {
            a += 0x06;
        }
        if ((a & 0xF0) > 0x90) {
            a += 0x60;
            setcarry();
        }

        clockticks6502++;
    }
#endif

    saveaccum(result);
}

void op_and()
{
    value = getvalue();
    result = (uint16_t)a & value;

    zerocalc(result);
    signcalc(result);

    saveaccum(result);
}

void asl()
{
    value = getvalue();
    result = value << 1;

    carrycalc(result);
    zerocalc(result);
    signcalc(result);

    putvalue(result);
}

void bcc()
{
    if ((cpustatus & FLAG_CARRY) == 0) {
        oldpc = pc;
        pc += reladdr;
        if ((oldpc & 0xFF00) != (pc & 0xFF00)) clockticks6502 += 2; //check if jump crossed a page boundary
        else clockticks6502++;
    }
}

void bcs()
{
    if ((cpustatus & FLAG_CARRY) == FLAG_CARRY) {
        oldpc = pc;
        pc += reladdr;
        if ((oldpc & 0xFF00) != (pc & 0xFF00)) clockticks6502 += 2; //check if jump crossed a page boundary
        else clockticks6502++;
    }
}

void beq()
{
    if ((cpustatus & FLAG_ZERO) == FLAG_ZERO) {
        oldpc = pc;
        pc += reladdr;
        if ((oldpc & 0xFF00) != (pc & 0xFF00)) clockticks6502 += 2; //check if jump crossed a page boundary
        else clockticks6502++;
    }
}

void op_bit()
{
    value = getvalue();
    result = (uint16_t)a & value;

    zerocalc(result);
    cpustatus = (cpustatus & 0x3F) | (uint8_t)(value & 0xC0);
}

void bmi()
{
    if ((cpustatus & FLAG_SIGN) == FLAG_SIGN) {
        oldpc = pc;
        pc += reladdr;
        if ((oldpc & 0xFF00) != (pc & 0xFF00)) clockticks6502 += 2; //check if jump crossed a page boundary
        else clockticks6502++;
    }
}

void bne()
{
    if ((cpustatus & FLAG_ZERO) == 0) {
        oldpc = pc;
        pc += reladdr;
        if ((oldpc & 0xFF00) != (pc & 0xFF00)) clockticks6502 += 2; //check if jump crossed a page boundary
        else clockticks6502++;
    }
}

void bpl()
{
    if ((cpustatus & FLAG_SIGN) == 0) {
        oldpc = pc;
        pc += reladdr;
        if ((oldpc & 0xFF00) != (pc & 0xFF00)) clockticks6502 += 2; //check if jump crossed a page boundary
        else clockticks6502++;
    }
}

void brk()
{
    pc++;
    push16(pc); //push next instruction address onto stack
    push8(cpustatus | FLAG_BREAK); //push CPU cpustatus to stack
    setinterrupt(); //set interrupt flag
    pc = (uint16_t)read6502(0xFFFE) | ((uint16_t)read6502(0xFFFF) << 8);
}

void bvc()
{
    if ((cpustatus & FLAG_OVERFLOW) == 0) {
        oldpc = pc;
        pc += reladdr;
        if ((oldpc & 0xFF00) != (pc & 0xFF00)) clockticks6502 += 2; //check if jump crossed a page boundary
        else clockticks6502++;
    }
}

void bvs()
{
    if ((cpustatus & FLAG_OVERFLOW) == FLAG_OVERFLOW) {
        oldpc = pc;
        pc += reladdr;
        if ((oldpc & 0xFF00) != (pc & 0xFF00)) clockticks6502 += 2; //check if jump crossed a page boundary
        else clockticks6502++;
    }
}

void clc()
{
    clearcarry();
}

void cld()
{
    cleardecimal();
}

void cli()
{
    clearinterrupt();
}

void clv()
{
    clearoverflow();
}

void cmp()
{
    value = getvalue();
    result = (uint16_t)a - value;

    if (a >= (uint8_t)(value & 0x00FF)) setcarry();
    else clearcarry();
    if (a == (uint8_t)(value & 0x00FF)) setzero();
    else clearzero();
    signcalc(result);
}

void cpx()
{
    value = getvalue();
    result = (uint16_t)x - value;

    if (x >= (uint8_t)(value & 0x00FF)) setcarry();
    else clearcarry();
    if (x == (uint8_t)(value & 0x00FF)) setzero();
    else clearzero();
    signcalc(result);
}

void cpy()
{
    value = getvalue();
    result = (uint16_t)y - value;

    if (y >= (uint8_t)(value & 0x00FF)) setcarry();
    else clearcarry();
    if (y == (uint8_t)(value & 0x00FF)) setzero();
    else clearzero();
    signcalc(result);
}

void dec()
{
    value = getvalue();
    result = value - 1;

    zerocalc(result);
    signcalc(result);

    putvalue(result);
}

void dex()
{
    x--;

    zerocalc(x);
    signcalc(x);
}

void dey()
{
    y--;

    zerocalc(y);
    signcalc(y);
}

void eor()
{
    value = getvalue();
    result = (uint16_t)a ^ value;

    zerocalc(result);
    signcalc(result);

    saveaccum(result);
}

void inc()
{
    value = getvalue();
    result = value + 1;

    zerocalc(result);
    signcalc(result);

    putvalue(result);
}

void inx()
{
    x++;

    zerocalc(x);
    signcalc(x);
}

void iny()
{
    y++;

    zerocalc(y);
    signcalc(y);
}

void jmp()
{
    pc = ea;
}

void jsr()
{
    push16(pc - 1);
    pc = ea;
}

void lda()
{
    value = getvalue();
    a = (uint8_t)(value & 0x00FF);

    zerocalc(a);
    signcalc(a);
}

void ldx()
{
    value = getvalue();
    x = (uint8_t)(value & 0x00FF);

    zerocalc(x);
    signcalc(x);
}

void ldy()
{
    value = getvalue();
    y = (uint8_t)(value & 0x00FF);

    zerocalc(y);
    signcalc(y);
}

void lsr()
{
    value = getvalue();
    result = value >> 1;

    if (value & 1) setcarry();
    else clearcarry();
    zerocalc(result);
    signcalc(result);

    putvalue(result);
}

void nop()
{
}

void ora()
{
    value = getvalue();
    result = (uint16_t)a | value;

    zerocalc(result);
    signcalc(result);

    saveaccum(result);
}

void pha()
{
    push8(a);
}

void php()
{
    push8(cpustatus | FLAG_BREAK);
}

void pla()
{
    a = pull8();

    zerocalc(a);
    signcalc(a);
}

void plp()
{
    cpustatus = pull8() | FLAG_CONSTANT;
}

void rol()
{
    value = getvalue();
    result = (value << 1) | (cpustatus & FLAG_CARRY);

    carrycalc(result);
    zerocalc(result);
    signcalc(result);

    putvalue(result);
}

void ror()
{
    value = getvalue();
    result = (value >> 1) | ((cpustatus & FLAG_CARRY) << 7);

    if (value & 1) setcarry();
    else clearcarry();
    zerocalc(result);
    signcalc(result);

    putvalue(result);
}

void rti()
{
    cpustatus = pull8();
    value = pull16();
    pc = value;
}

void rts()
{
    value = pull16();
    pc = value + 1;
}

void sbc()
{
    value = getvalue() ^ 0x00FF;
    result = (uint16_t)a + value + (uint16_t)(cpustatus & FLAG_CARRY);

    carrycalc(result);
    zerocalc(result);
    overflowcalc(result, a, value);
    signcalc(result);

#ifndef NES_CPU
    if (cpustatus & FLAG_DECIMAL) {
        clearcarry();

        a -= 0x66;
        if ((a & 0x0F) > 0x09) {
            a += 0x06;
        }
        if ((a & 0xF0) > 0x90) {
            a += 0x60;
            setcarry();
        }

        clockticks6502++;
    }
#endif

    saveaccum(result);
}

void sec()
{
    setcarry();
}

void sed()
{
    setdecimal();
}

void sei()
{
    setinterrupt();
}

void sta()
{
    putvalue(a);
}

void stx()
{
    putvalue(x);
}

void sty()
{
    putvalue(y);
}

void tax()
{
    x = a;

    zerocalc(x);
    signcalc(x);
}

void tay()
{
    y = a;

    zerocalc(y);
    signcalc(y);
}

void tsx()
{
    x = sp;

    zerocalc(x);
    signcalc(x);
}

void txa()
{
    a = x;

    zerocalc(a);
    signcalc(a);
}

void txs()
{
    sp = x;
}

void tya()
{
    a = y;

    zerocalc(a);
    signcalc(a);
}

//undocumented instructions
#ifdef UNDOCUMENTED
void lax()
{
    lda();
    ldx();
}

void sax()
{
    sta();
    stx();
    putvalue(a & x);
}

void dcp()
{
    dec();
    cmp();
}

void isb()
{
    inc();
    sbc();
}

void slo()
{
    asl();
    ora();
}

void rla()
{
    rol();
    op_and();
}

void sre()
{
    lsr();
    eor();
}

void rra()
{
    ror();
    adc();
}
#else
#define lax nop
#define sax nop
#define dcp nop
#define isb nop
#define slo nop
#define rla nop
#define sre nop
#define rra nop
#endif


void nmi6502()
{
    push16(pc);
    push8(cpustatus);
    cpustatus |= FLAG_INTERRUPT;
    pc = (uint16_t)read6502(0xFFFA) | ((uint16_t)read6502(0xFFFB) << 8);
}

void irq6502()
{
    push16(pc);
    push8(cpustatus);
    cpustatus |= FLAG_INTERRUPT;
    pc = (uint16_t)read6502(0xFFFE) | ((uint16_t)read6502(0xFFFF) << 8);
}

#ifdef USE_TIMING
prog_char ticktable[256] PROGMEM = {
    /*        |  0  |  1  |  2  |  3  |  4  |  5  |  6  |  7  |  8  |  9  |  A  |  B  |  C  |  D  |  E  |  F  |     */
    /* 0 */      7,    6,    2,    8,    3,    3,    5,    5,    3,    2,    2,    2,    4,    4,    6,    6,  /* 0 */
    /* 1 */      2,    5,    2,    8,    4,    4,    6,    6,    2,    4,    2,    7,    4,    4,    7,    7,  /* 1 */
    /* 2 */      6,    6,    2,    8,    3,    3,    5,    5,    4,    2,    2,    2,    4,    4,    6,    6,  /* 2 */
    /* 3 */      2,    5,    2,    8,    4,    4,    6,    6,    2,    4,    2,    7,    4,    4,    7,    7,  /* 3 */
    /* 4 */      6,    6,    2,    8,    3,    3,    5,    5,    3,    2,    2,    2,    3,    4,    6,    6,  /* 4 */
    /* 5 */      2,    5,    2,    8,    4,    4,    6,    6,    2,    4,    2,    7,    4,    4,    7,    7,  /* 5 */
    /* 6 */      6,    6,    2,    8,    3,    3,    5,    5,    4,    2,    2,    2,    5,    4,    6,    6,  /* 6 */
    /* 7 */      2,    5,    2,    8,    4,    4,    6,    6,    2,    4,    2,    7,    4,    4,    7,    7,  /* 7 */
    /* 8 */      2,    6,    2,    6,    3,    3,    3,    3,    2,    2,    2,    2,    4,    4,    4,    4,  /* 8 */
    /* 9 */      2,    6,    2,    6,    4,    4,    4,    4,    2,    5,    2,    5,    5,    5,    5,    5,  /* 9 */
    /* A */      2,    6,    2,    6,    3,    3,    3,    3,    2,    2,    2,    2,    4,    4,    4,    4,  /* A */
    /* B */      2,    5,    2,    5,    4,    4,    4,    4,    2,    4,    2,    4,    4,    4,    4,    4,  /* B */
    /* C */      2,    6,    2,    8,    3,    3,    5,    5,    2,    2,    2,    2,    4,    4,    6,    6,  /* C */
    /* D */      2,    5,    2,    8,    4,    4,    6,    6,    2,    4,    2,    7,    4,    4,    7,    7,  /* D */
    /* E */      2,    6,    2,    8,    3,    3,    5,    5,    2,    2,    2,    2,    4,    4,    6,    6,  /* E */
    /* F */      2,    5,    2,    8,    4,    4,    6,    6,    2,    4,    2,    7,    4,    4,    7,    7   /* F */
};
#endif

void exec6502(int32_t tickcount)
{
#ifdef USE_TIMING
    clockgoal6502 += tickcount;

    while (clockgoal6502 > 0) {
#else
    while (tickcount--) {
#endif
        opcode = read6502(pc++);
        cpustatus |= FLAG_CONSTANT;

        useaccum = 0;

        switch (opcode) {
            case 0x0:
                imp();
                brk();
                break;
            case 0x1:
                indx();
                ora();
                break;
            case 0x5:
                zp();
                ora();
                break;
            case 0x6:
                zp();
                asl();
                break;
            case 0x8:
                imp();
                php();
                break;
            case 0x9:
                imm();
                ora();
                break;
            case 0xA:
                acc();
                asl();
                break;
            case 0xD:
                abso();
                ora();
                break;
            case 0xE:
                abso();
                asl();
                break;
            case 0x10:
                rel();
                bpl();
                break;
            case 0x11:
                indy();
                ora();
                break;
            case 0x15:
                zpx();
                ora();
                break;
            case 0x16:
                zpx();
                asl();
                break;
            case 0x18:
                imp();
                clc();
                break;
            case 0x19:
                absy();
                ora();
                break;
            case 0x1D:
                absx();
                ora();
                break;
            case 0x1E:
                absx();
                asl();
                break;
            case 0x20:
                abso();
                jsr();
                break;
            case 0x21:
                indx();
                op_and();
                break;
            case 0x24:
                zp();
                op_bit();
                break;
            case 0x25:
                zp();
                op_and();
                break;
            case 0x26:
                zp();
                rol();
                break;
            case 0x28:
                imp();
                plp();
                break;
            case 0x29:
                imm();
                op_and();
                break;
            case 0x2A:
                acc();
                rol();
                break;
            case 0x2C:
                abso();
                op_bit();
                break;
            case 0x2D:
                abso();
                op_and();
                break;
            case 0x2E:
                abso();
                rol();
                break;
            case 0x30:
                rel();
                bmi();
                break;
            case 0x31:
                indy();
                op_and();
                break;
            case 0x35:
                zpx();
                op_and();
                break;
            case 0x36:
                zpx();
                rol();
                break;
            case 0x38:
                imp();
                sec();
                break;
            case 0x39:
                absy();
                op_and();
                break;
            case 0x3D:
                absx();
                op_and();
                break;
            case 0x3E:
                absx();
                rol();
                break;
            case 0x40:
                imp();
                rti();
                break;
            case 0x41:
                indx();
                eor();
                break;
            case 0x45:
                zp();
                eor();
                break;
            case 0x46:
                zp();
                lsr();
                break;
            case 0x48:
                imp();
                pha();
                break;
            case 0x49:
                imm();
                eor();
                break;
            case 0x4A:
                acc();
                lsr();
                break;
            case 0x4C:
                abso();
                jmp();
                break;
            case 0x4D:
                abso();
                eor();
                break;
            case 0x4E:
                abso();
                lsr();
                break;
            case 0x50:
                rel();
                bvc();
                break;
            case 0x51:
                indy();
                eor();
                break;
            case 0x55:
                zpx();
                eor();
                break;
            case 0x56:
                zpx();
                lsr();
                break;
            case 0x58:
                imp();
                cli();
                break;
            case 0x59:
                absy();
                eor();
                break;
            case 0x5D:
                absx();
                eor();
                break;
            case 0x5E:
                absx();
                lsr();
                break;
            case 0x60:
                imp();
                rts();
                break;
            case 0x61:
                indx();
                adc();
                break;
            case 0x65:
                zp();
                adc();
                break;
            case 0x66:
                zp();
                ror();
                break;
            case 0x68:
                imp();
                pla();
                break;
            case 0x69:
                imm();
                adc();
                break;
            case 0x6A:
                acc();
                ror();
                break;
            case 0x6C:
                ind();
                jmp();
                break;
            case 0x6D:
                abso();
                adc();
                break;
            case 0x6E:
                abso();
                ror();
                break;
            case 0x70:
                rel();
                bvs();
                break;
            case 0x71:
                indy();
                adc();
                break;
            case 0x75:
                zpx();
                adc();
                break;
            case 0x76:
                zpx();
                ror();
                break;
            case 0x78:
                imp();
                sei();
                break;
            case 0x79:
                absy();
                adc();
                break;
            case 0x7D:
                absx();
                adc();
                break;
            case 0x7E:
                absx();
                ror();
                break;
            case 0x81:
                indx();
                sta();
                break;
            case 0x84:
                zp();
                sty();
                break;
            case 0x85:
                zp();
                sta();
                break;
            case 0x86:
                zp();
                stx();
                break;
            case 0x88:
                imp();
                dey();
                break;
            case 0x8A:
                imp();
                txa();
                break;
            case 0x8C:
                abso();
                sty();
                break;
            case 0x8D:
                abso();
                sta();
                break;
            case 0x8E:
                abso();
                stx();
                break;
            case 0x90:
                rel();
                bcc();
                break;
            case 0x91:
                indy();
                sta();
                break;
            case 0x94:
                zpx();
                sty();
                break;
            case 0x95:
                zpx();
                sta();
                break;
            case 0x96:
                zpy();
                stx();
                break;
            case 0x98:
                imp();
                tya();
                break;
            case 0x99:
                absy();
                sta();
                break;
            case 0x9A:
                imp();
                txs();
                break;
            case 0x9D:
                absx();
                sta();
                break;
            case 0xA0:
                imm();
                ldy();
                break;
            case 0xA1:
                indx();
                lda();
                break;
            case 0xA2:
                imm();
                ldx();
                break;
            case 0xA4:
                zp();
                ldy();
                break;
            case 0xA5:
                zp();
                lda();
                break;
            case 0xA6:
                zp();
                ldx();
                break;
            case 0xA8:
                imp();
                tay();
                break;
            case 0xA9:
                imm();
                lda();
                break;
            case 0xAA:
                imp();
                tax();
                break;
            case 0xAC:
                abso();
                ldy();
                break;
            case 0xAD:
                abso();
                lda();
                break;
            case 0xAE:
                abso();
                ldx();
                break;
            case 0xB0:
                rel();
                bcs();
                break;
            case 0xB1:
                indy();
                lda();
                break;
            case 0xB4:
                zpx();
                ldy();
                break;
            case 0xB5:
                zpx();
                lda();
                break;
            case 0xB6:
                zpy();
                ldx();
                break;
            case 0xB8:
                imp();
                clv();
                break;
            case 0xB9:
                absy();
                lda();
                break;
            case 0xBA:
                imp();
                tsx();
                break;
            case 0xBC:
                absx();
                ldy();
                break;
            case 0xBD:
                absx();
                lda();
                break;
            case 0xBE:
                absy();
                ldx();
                break;
            case 0xC0:
                imm();
                cpy();
                break;
            case 0xC1:
                indx();
                cmp();
                break;
            case 0xC4:
                zp();
                cpy();
                break;
            case 0xC5:
                zp();
                cmp();
                break;
            case 0xC6:
                zp();
                dec();
                break;
            case 0xC8:
                imp();
                iny();
                break;
            case 0xC9:
                imm();
                cmp();
                break;
            case 0xCA:
                imp();
                dex();
                break;
            case 0xCC:
                abso();
                cpy();
                break;
            case 0xCD:
                abso();
                cmp();
                break;
            case 0xCE:
                abso();
                dec();
                break;
            case 0xD0:
                rel();
                bne();
                break;
            case 0xD1:
                indy();
                cmp();
                break;
            case 0xD5:
                zpx();
                cmp();
                break;
            case 0xD6:
                zpx();
                dec();
                break;
            case 0xD8:
                imp();
                cld();
                break;
            case 0xD9:
                absy();
                cmp();
                break;
            case 0xDD:
                absx();
                cmp();
                break;
            case 0xDE:
                absx();
                dec();
                break;
            case 0xE0:
                imm();
                cpx();
                break;
            case 0xE1:
                indx();
                sbc();
                break;
            case 0xE4:
                zp();
                cpx();
                break;
            case 0xE5:
                zp();
                sbc();
                break;
            case 0xE6:
                zp();
                inc();
                break;
            case 0xE8:
                imp();
                inx();
                break;
            case 0xE9:
                imm();
                sbc();
                break;
            case 0xEB:
                imm();
                sbc();
                break;
            case 0xEC:
                abso();
                cpx();
                break;
            case 0xED:
                abso();
                sbc();
                break;
            case 0xEE:
                abso();
                inc();
                break;
            case 0xF0:
                rel();
                beq();
                break;
            case 0xF1:
                indy();
                sbc();
                break;
            case 0xF5:
                zpx();
                sbc();
                break;
            case 0xF6:
                zpx();
                inc();
                break;
            case 0xF8:
                imp();
                sed();
                break;
            case 0xF9:
                absy();
                sbc();
                break;
            case 0xFD:
                absx();
                sbc();
                break;
            case 0xFE:
                absx();
                inc();
                break;
        }
#ifdef USE_TIMING
        clockgoal6502 -= (int32_t)pgm_read_byte_near(ticktable + opcode);
#endif
        instructions++;
    }
}

uint16_t getpc()
{
    return(pc);
}

uint8_t getop()
{
    return(opcode);
}
