/**
 * @section LICENSE
 *
 * This source code is "zlib" license. 
 * Of course, this is not a part of the zlib software. 
 * The license is zlib license. 
 *
 * Copyright (C) 2010 tosihisa
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * @section DESCRIPTION
 *
 * Apple Remote Decoder
 * http://en.wikipedia.org/wiki/Apple_Remote
 *
 */
#ifndef APPLEREMOTE_H
#define APPLEREMOTE_H

class AppleRemote {
public:
    /**
     * Constructor.
     *
     * Because "_ir mbed pin" is processed by the interruption(use InterruptIn class),
     * the thing set excluding p19 and p20 .
     *
     * @param _ir mbed pin to use for Apple Remote.
     */
    AppleRemote( PinName _ir );

    /**
     * Reading is started. 
     *
     * The interruption of "_ir mbed pin" is effectively done.  
     */
    void start();

    /**
     * Reading is stopped. 
     *
     * The interrupt of "_ir mbed pin" is stopped. 
     */
    void stop();

    /**
     * Reading is reset. 
     *
     * This resets reading processing it now. 
     */
    void reset();

    /**
     * Determine if there is a key-code available to read
     *
     * @return 1 if there is a key-code available to read, else 0
     */
    int readable();

    /**
     * Read a key-code
     *
     * key-code does buffering. The size of the buffer is 15. 
     * The 16 bits of key-code consist of a one byte remote ID , a one byte command.
     *
     * @return The key-code read from the Apple Remote.
     */
    int getc();

    /**
     * Setting of repeat
     *
     * The repeat code of Apple Remote is not used by default. 
     * Please call this function when you use the repeat code.  
     * (The same key code returns while the key is being pushed.)
     *
     * @param enable 1 if use the repeat code. else 0
     */
    void repeat(int enable);

    enum KEY_CODE {
        MENU_1ST   = 0x02,
        PLAY_1ST   = 0x04,
        RIGHT_1ST  = 0x07,
        LEFT_1ST   = 0x08,
        UP_1ST     = 0x0B,
        DOWN_1ST   = 0x0D,
        /* --- */
        CENTER_2ND = 0x5C,
        MENU_2ND   = 0x03,
        PLAY_2ND   = 0x5F,
        RIGHT_2ND  = 0x06,
        LEFT_2ND   = 0x09,
        UP_2ND     = 0x0A,
        DOWN_2ND   = 0x0C,
    };

private:
    static const int ID = 0x87EE;
    static const int bufsize = 0x10;

    enum IR_SCAN_STAT {
        WAIT_READER,
        CHK_READER_1ST,
        CHK_READER_2ND,
        CHK_CODE_1ST,
        CHK_CODE_2ND,
        CHK_STOP_1ST,
        CHK_STOP_2ND,
    };
    InterruptIn IR;
    int nowStart;
    Timer IR_Timer;
    volatile enum IR_SCAN_STAT cjobst;
    volatile int bit_index;
    volatile unsigned long IR_code;
    unsigned long pre_IR_code;
    volatile int pre_timer;
    volatile unsigned short buf[bufsize];
    volatile unsigned char widx;
    volatile unsigned char ridx;
    int _repeatEnable;
    void IR_fall();
    void IR_rise();
};

#endif /* APPLEREMOTE_H */
