/*

Copyright (c) 2012-2014 RedBearLab

Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
and associated documentation files (the "Software"), to deal in the Software without restriction, 
including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, 
and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, 
subject to the following conditions:
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE 
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include "mbed.h"
#include "ble/BLE.h"
#include "ADXL345_I2C/ADXL345_I2C.h"

#define TXRX_BUF_LEN                     20
#define LOCAL_NAME                       "4WD"
#define NOTIFY_INTERVAL                  200
#define MEASURE_INTERVAL                 10

ADXL345_I2C accelerometer(P0_10, P0_8);
BLE             ble;
DigitalOut led(P0_19);

static const uint8_t uart_base_uuid[] = {0xc8, 0xd6, 0xea, 0x62, 0x40, 0x76, 0x45, 0x61, 0xbf, 0xc8, 0x90, 0xaf, 0x4e, 0xd7, 0x8f, 0x4a};
static const uint8_t uart_tx_uuid[]   = {0xd5, 0x31, 0x16, 0xfc, 0x4b, 0x96, 0x44, 0x46, 0xb8, 0x9f, 0xd9, 0x60, 0x00, 0x1c, 0xea, 0x91};
static const uint8_t uart_rx_uuid[]   = {0xd9, 0x46, 0x28, 0x6e, 0xb6, 0xae, 0x4e, 0x75, 0x8c, 0x8d, 0xcb, 0x18, 0x43, 0x4a, 0x13, 0x98};
static uint8_t uart_base_uuid_rev[]   = {0x4a, 0x8f, 0xd7, 0x4e, 0xaf, 0x90, 0xc8, 0xbf, 0x61, 0x45, 0x76, 0x40, 0x62, 0xea, 0xd6, 0xc8};

uint8_t txPayload[TXRX_BUF_LEN] = {0,};
uint8_t rxPayload[TXRX_BUF_LEN] = {0,};

GattCharacteristic  txCharacteristic (uart_tx_uuid, txPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE);
GattCharacteristic  rxCharacteristic (uart_rx_uuid, rxPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);
GattCharacteristic *uartChars[] = {&txCharacteristic, &rxCharacteristic};
GattService         uartService(uart_base_uuid, uartChars, sizeof(uartChars) / sizeof(GattCharacteristic *));

struct Vector3 {
    float x;
    float y;
    float z;
    Vector3():x(0),y(0),z(0){}
    void clear(){x=y=z=0;}
};

struct NotifyData {
    float x;
    float y;
    float z;
    float speed;
};

NotifyData notifyData;

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t* disconnectionCallback) {
    ble.startAdvertising();
}

void WrittenHandler(const GattWriteCallbackParams *Handler) {
}

void notifyValues() {
    uint8_t buf[20];
    memcpy(buf, &notifyData, sizeof(NotifyData));
    ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), buf, sizeof(NotifyData)); 
}

Vector3 rawValue;
int rawCount = 0;

void measure() {
    int readings[3] = {0, 0, 0};
    accelerometer.getOutput(readings);
    // accelerometerの値は16bitの2の補数表現なので、自力で負数に変換する
    for (int i = 0; i < 3; i++) {
        if ((readings[i] & 0x8000) != 0) {
            readings[i] = -((~(readings[i] & 0x7FFF)) & 0x7FFF);
        }
    }
    
    rawValue.x += readings[0];
    rawValue.y += readings[1];
    rawValue.z += readings[2];
    rawCount++;
    
    if (rawCount >= NOTIFY_INTERVAL/MEASURE_INTERVAL) {
        notifyData.x = rawValue.x / 255 / rawCount; // G
        notifyData.y = rawValue.y / 255 / rawCount; // G
        notifyData.z = rawValue.z / 255 / rawCount; // G
        notifyData.speed += (rawValue.x / 255.0 * 9.8 * 3.6 / 2 / MEASURE_INTERVAL) - notifyData.speed; // km/h

        rawValue.clear();
        rawCount = 0;
    }
}

int main(void) {
    
    memset(&notifyData, 0, sizeof(NotifyData));
    
    Ticker ticker;
    ticker.attach_us(notifyValues, NOTIFY_INTERVAL * 1000);
    
    Ticker ticker2;
    ticker2.attach_us(measure, MEASURE_INTERVAL * 1000);

    // Go into standby mode to configure the device.
    accelerometer.setPowerControl(0x00);
    // Full resolution, +/-16g, 4mg/LSB.
    accelerometer.setDataFormatControl(0x0B);     
    // 100Hz data rate.(10msec)
    accelerometer.setDataRate(ADXL345_100HZ);
    // Measurement mode.
    accelerometer.setPowerControl(0x08);
    
    ble.init();
    ble.onDisconnection(disconnectionCallback);
    ble.onDataWritten(WrittenHandler);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                    (const uint8_t *)LOCAL_NAME, sizeof(LOCAL_NAME) - 1);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                    (const uint8_t *)uart_base_uuid_rev, sizeof(uart_base_uuid));
    ble.setDeviceName(LOCAL_NAME);
    ble.setAdvertisingInterval(1000/0.625);
    ble.addService(uartService);
    ble.startAdvertising();
    led = 0;

    while(1) {
        ble.waitForEvent(); 
    }
}
