/* mbed library for the mbed Lab Board  128*32 pixel LCD
 * use C12832 controller
 * Copyright (c) 2012 Peter Drescher - DC2PD
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

// 13.10.12    initial design
// 25.10.12    add autorefresh of screen
// 25.10.12    add standart font 

// optional defines :
// #define debug_lcd  1

#include "gLCD.h"
#include "mbed.h"
#include "stdio.h"


#define BPP    1       // Bits per pixel

gLCD::gLCD(PinName cs, 
           PinName res, 
           PinName rs, 
           PinName sdo, PinName sdi, PinName scl,
           PinName bcl,
           const char* name)
    : _spi(sdo,sdi,scl),_reset(res),_A0(rs),_CS(cs),backLight(bcl),GraphicsDisplay(name)
{
    orientation = 1;
    draw_mode = NORMAL;
    char_x = 0;
    lcd_reset();
}

int gLCD::width()
{
    if (orientation == 0 || orientation == 2) return 64;
    else return 128;
}

int gLCD::height()
{
    if (orientation == 0 || orientation == 2) return 128;
    else return 64;
}
float bclValue = 1;

void gLCD::bclIncrease()
{
    bclValue+=0.1;
    if(bclValue > 1) bclValue = 1;
    backLight = bclValue;
}
void gLCD::bclDecrease()
{
    bclValue-=0.1;
    if(bclValue < 0) bclValue = 0;
    backLight = bclValue;
}
     

void gLCD::set_orientation(unsigned int o)
{
    orientation = o;
    switch (o) {
        case (0):
            wr_cmd(0xA0);
            wr_cmd(0xC0);
            break;
        case (1):
            wr_cmd(0xA0);
            wr_cmd(0xC8);
            break;
        case (2):
            wr_cmd(0xA1);
            wr_cmd(0xC8);
            break;
        case (3):
            wr_cmd(0xA1);
            wr_cmd(0xC0);
            break;
    }
}



void gLCD::invert(unsigned int o)
{
    if(o == 0) wr_cmd(0xA6);
    else wr_cmd(0xA7);
}


void gLCD::set_contrast(unsigned int o)
{
    contrast = o;
    wr_cmd(0x81);      //  set volume
    wr_cmd(o & 0x3F);
}

unsigned int gLCD::get_contrast(void)
{
    return(contrast);
}


// write command to lcd controller

void gLCD::wr_cmd(unsigned char cmd)
{
    _A0 = 0;
    _CS = 0;
#if defined TARGET_LPC1768     // fast without mbed lib
    LPC_SSP1->DR = cmd;
    do {
    } while ((LPC_SSP1->SR & 0x10) == 0x10); // wait for SPI1 idle
#else
    _spi.write(cmd);
#endif
    _A0 = 1;
    wait(0.000040f);
    _A0 = 0;
    _CS = 1;
}

// write data to lcd controller

void gLCD::wr_dat(unsigned char dat)
{
    _A0 = 1;
    _CS = 0;
#if defined TARGET_LPC1768     // fast without mbed lib
    LPC_SSP1->DR = dat;
    do {
    } while ((LPC_SSP1->SR & 0x10) == 0x10); // wait for SPI1 idle
#else
    _spi.write(dat);
#endif
    _A0 = 0;
    wait(0.000040f);
    _A0 = 1;
    _CS = 1;
}

// reset and init the lcd controller

void gLCD::lcd_reset()
{

    _spi.format(8,0);                 // 8 bit spi mode 0
    _spi.frequency(20000000);          // 19,2 Mhz SPI clock
    _A0 = 0;
    _CS = 1;
    _reset = 0;                        // display reset
    wait(0.015);                      //Give display 15ms to reset
    _reset = 1;                       // end reset
    wait_ms(5);
    backLight.period_ms(1);
    backLight.pulsewidth_ms(1);
    backLight = 1;
    /* Start Initial Sequence ----------------------------------------------------*/

 /*   wr_cmd(0xAE);   //  display off
    wr_cmd(0xA2);   //  bias voltage

    

    wr_cmd(0x22);   //  voltage resistor ratio
    wr_cmd(0x2F);   //  power on
    //wr_cmd(0xA4);   //  LCD display ram
    wr_cmd(0x40);   // start line = 0
    wr_cmd(0xAF);     // display ON

    wr_cmd(0x81);   //  set contrast
    wr_cmd(0x17);   //  set contrast

    wr_cmd(0xA6);     // display normal
*/
    wr_cmd(0xA0);       //SEG1->SEG132 x direction
    wr_cmd(0xC8);       //COM1->COM64 y direction
    wr_cmd(0xA3);       // 1/9 bias
    wr_cmd(0x2C);       //Voltage converter on
    wait(0.001);
    wr_cmd(0x2E);       //Voltage regulator on
    wait(0.001);
    wr_cmd(0x2F);       //Voltage follower on
    wr_cmd(0x26);       //voltage reg ratio = 5.29
    wr_cmd(0x81);       //Enter voltage resistor set mode
    wr_cmd(0x8);        //Set volt res (contrast)
    wr_cmd(0x40);       //initial display line = 0
    wr_cmd(0xB0);       //initial page address = 0
    wr_cmd(0x00);   
    wr_cmd(0x10);       //Sets first col
    wr_cmd(0xA6);       //not inverse display
    wr_cmd(0xAD);       //Static indicator off
    wr_cmd(0x00);       //no blinking
    wr_cmd(0xAF);       //display on
#if defined TARGET_LPC1768          //setup DMA channel 0       
    LPC_SC->PCONP |= (1UL << 29);   // Power up the GPDMA
    LPC_GPDMA->DMACConfig = 1;      // enable DMA controller
    LPC_GPDMA->DMACIntTCClear = 0x1;
    LPC_GPDMA->DMACIntErrClr = 0x1;
    LPC_GPDMACH0->DMACCLLI = 0;
#endif
    // clear and update LCD
    memset(buffer,0x00,512);  // clear display buffer
    copy_to_lcd();
    auto_up = 1;              // switch on auto update
    claim(stdout);           // redirect printf to lcd
    locate(0,0);
    set_font((unsigned char*)Small_7);  // standard font 
}

// set one pixel in buffer

void gLCD::pixel(int x, int y, int color)
{
    // first check parameter
    if(x > 128 || y > 64 || x < 0 || y < 0) return;

    if(draw_mode == NORMAL) {
        if(color == 0)
            buffer[x + ((y/8) * 128)] &= ~(1 << (y%8));  // erase pixel
        else
            buffer[x + ((y/8) * 128)] |= (1 << (y%8));   // set pixel
    } else { // XOR mode
        if(color == 1)
            buffer[x + ((y/8) * 128)] ^= (1 << (y%8));   // xor pixel
    }
}

// update lcd

void gLCD::copy_to_lcd(void)
{
#ifndef TARGET_LPC1768 
    int i;
#endif
    //page 0
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB0);      // set page address  0
    _A0 = 1;
#if defined TARGET_LPC1768     
    _CS = 0;
    // start 128 byte DMA transfer to SPI1
    LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP1->DR; // we send to SSP1
    LPC_SSP1->DMACR = 0x2;  // Enable SSP1 for DMA.
    LPC_GPDMA->DMACIntTCClear = 0x1;
    LPC_GPDMA->DMACIntErrClr = 0x1;
    LPC_GPDMACH0->DMACCSrcAddr = (uint32_t) (buffer);
    LPC_GPDMACH0->DMACCControl = 128 | (1UL << 31) |  DMA_CHANNEL_SRC_INC ; // 8 bit transfer , address increment, interrupt
    LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | DMA_DEST_SSP1_TX;
    LPC_GPDMA->DMACSoftSReq = 0x1;
    do {
    } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running
    do {
    } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI1 not idle
    _CS = 1;
#else  // no DMA
    for(i=0;i<128;i++){
        wr_dat(buffer[i]);
        }
#endif 

    // page 1
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB1);      // set page address  1
    _A0 = 1;
#if defined TARGET_LPC1768         
    _CS = 0;
    // start 128 byte DMA transfer to SPI1
    LPC_GPDMA->DMACIntTCClear = 0x1;
    LPC_GPDMA->DMACIntErrClr = 0x1;
    LPC_GPDMACH0->DMACCSrcAddr = (uint32_t) (buffer + 128);
    LPC_GPDMACH0->DMACCControl = 128 | (1UL << 31) |  DMA_CHANNEL_SRC_INC ; // 8 bit transfer , address increment, interrupt
    LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | DMA_DEST_SSP1_TX;
    LPC_GPDMA->DMACSoftSReq = 0x1;
    do {
    } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running
    do {
    } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI1 not idle
    _CS = 1;
#else // no DMA
    for(i=128;i<256;i++){
        wr_dat(buffer[i]);
        }
#endif     

    //page 2
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB2);      // set page address  2
    _A0 = 1;
#if defined TARGET_LPC1768      
    _CS = 0;
    // start 128 byte DMA transfer to SPI1
    LPC_GPDMA->DMACIntTCClear = 0x1;
    LPC_GPDMA->DMACIntErrClr = 0x1;
    LPC_GPDMACH0->DMACCSrcAddr = (uint32_t) (buffer + 256);
    LPC_GPDMACH0->DMACCControl = 128 | (1UL << 31) |  DMA_CHANNEL_SRC_INC ; // 8 bit transfer , address increment, interrupt
    LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | DMA_DEST_SSP1_TX ;
    LPC_GPDMA->DMACSoftSReq = 0x1;
    do {
    } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running
    do {
    } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI1 not idle
    _CS = 1;
#else // no DMA
    for(i=256;i<384;i++){
        wr_dat(buffer[i]);
        }
#endif         

    //page 3
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB3);      // set page address  3
    _A0 = 1;
    
     _CS = 0;
#if defined TARGET_LPC1768     
    // start 128 byte DMA transfer to SPI1
    LPC_GPDMA->DMACIntTCClear = 0x1;
    LPC_GPDMA->DMACIntErrClr = 0x1;
    LPC_GPDMACH0->DMACCSrcAddr = (uint32_t) (buffer + 384);
    LPC_GPDMACH0->DMACCControl = 128  | (1UL << 31) |  DMA_CHANNEL_SRC_INC ; // 8 bit transfer , address increment, interrupt
    LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | DMA_DEST_SSP1_TX;
    LPC_GPDMA->DMACSoftSReq = 0x1;
    do {
    } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running
    do {
    } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI1 not idle
    _CS = 1;
#else // no DMA
    for(i=384;i<512;i++){
        wr_dat(buffer[i]);
        }
#endif

   //page 4
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB4);      // set page address  4
    _A0 = 1;
    
     _CS = 0;
#if defined TARGET_LPC1768     
    // start 128 byte DMA transfer to SPI1
    LPC_GPDMA->DMACIntTCClear = 0x1;
    LPC_GPDMA->DMACIntErrClr = 0x1;
    LPC_GPDMACH0->DMACCSrcAddr = (uint32_t) (buffer + 512);
    LPC_GPDMACH0->DMACCControl = 128  | (1UL << 31) |  DMA_CHANNEL_SRC_INC ; // 8 bit transfer , address increment, interrupt
    LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | DMA_DEST_SSP1_TX;
    LPC_GPDMA->DMACSoftSReq = 0x1;
    do {
    } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running
    do {
    } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI1 not idle
    _CS = 1;
#else // no DMA
    for(i=512;i<640;i++){
        wr_dat(buffer[i]);
        }
#endif  

 //page 5
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB5);      // set page address  5
    _A0 = 1;
    
     _CS = 0;
#if defined TARGET_LPC1768     
    // start 128 byte DMA transfer to SPI1
    LPC_GPDMA->DMACIntTCClear = 0x1;
    LPC_GPDMA->DMACIntErrClr = 0x1;
    LPC_GPDMACH0->DMACCSrcAddr = (uint32_t) (buffer + 640);
    LPC_GPDMACH0->DMACCControl = 128  | (1UL << 31) |  DMA_CHANNEL_SRC_INC ; // 8 bit transfer , address increment, interrupt
    LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | DMA_DEST_SSP1_TX;
    LPC_GPDMA->DMACSoftSReq = 0x1;
    do {
    } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running
    do {
    } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI1 not idle
    _CS = 1;
#else // no DMA
    for(i=640;i<768;i++){
        wr_dat(buffer[i]);
        }
#endif  

 //page 6
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB6);      // set page address  6
    _A0 = 1;
    
     _CS = 0;
#if defined TARGET_LPC1768     
    // start 128 byte DMA transfer to SPI1
    LPC_GPDMA->DMACIntTCClear = 0x1;
    LPC_GPDMA->DMACIntErrClr = 0x1;
    LPC_GPDMACH0->DMACCSrcAddr = (uint32_t) (buffer + 768);
    LPC_GPDMACH0->DMACCControl = 128  | (1UL << 31) |  DMA_CHANNEL_SRC_INC ; // 8 bit transfer , address increment, interrupt
    LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | DMA_DEST_SSP1_TX;
    LPC_GPDMA->DMACSoftSReq = 0x1;
    do {
    } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running
    do {
    } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI1 not idle
    _CS = 1;
#else // no DMA
    for(i=768;i<896;i++){
        wr_dat(buffer[i]);
        }
#endif  

 //page 7
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB7);      // set page address  7
    _A0 = 1;
    
     _CS = 0;
#if defined TARGET_LPC1768     
    // start 128 byte DMA transfer to SPI1
    LPC_GPDMA->DMACIntTCClear = 0x1;
    LPC_GPDMA->DMACIntErrClr = 0x1;
    LPC_GPDMACH0->DMACCSrcAddr = (uint32_t) (buffer + 896);
    LPC_GPDMACH0->DMACCControl = 128  | (1UL << 31) |  DMA_CHANNEL_SRC_INC ; // 8 bit transfer , address increment, interrupt
    LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | DMA_DEST_SSP1_TX;
    LPC_GPDMA->DMACSoftSReq = 0x1;
    do {
    } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running
    do {
    } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI1 not idle
    _CS = 1;
#else // no DMA
    for(i=896;i<1024;i++){
        wr_dat(buffer[i]);
        }
#endif   
}

void gLCD::cls(void)
{
    memset(buffer,0x00,1024);  // clear display buffer
    copy_to_lcd();
}


void gLCD::line(int x0, int y0, int x1, int y1, int color)
{
    int   dx = 0, dy = 0;
    int   dx_sym = 0, dy_sym = 0;
    int   dx_x2 = 0, dy_x2 = 0;
    int   di = 0;

    dx = x1-x0;
    dy = y1-y0;

    //  if (dx == 0) {        /* vertical line */
    //      if (y1 > y0) vline(x0,y0,y1,color);
    //      else vline(x0,y1,y0,color);
    //      return;
    //  }

    if (dx > 0) {
        dx_sym = 1;
    } else {
        dx_sym = -1;
    }
    //  if (dy == 0) {        /* horizontal line */
    //      if (x1 > x0) hline(x0,x1,y0,color);
    //      else  hline(x1,x0,y0,color);
    //      return;
    //  }

    if (dy > 0) {
        dy_sym = 1;
    } else {
        dy_sym = -1;
    }

    dx = dx_sym*dx;
    dy = dy_sym*dy;

    dx_x2 = dx*2;
    dy_x2 = dy*2;

    if (dx >= dy) {
        di = dy_x2 - dx;
        while (x0 != x1) {

            pixel(x0, y0, color);
            x0 += dx_sym;
            if (di<0) {
                di += dy_x2;
            } else {
                di += dy_x2 - dx_x2;
                y0 += dy_sym;
            }
        }
        pixel(x0, y0, color);
    } else {
        di = dx_x2 - dy;
        while (y0 != y1) {
            pixel(x0, y0, color);
            y0 += dy_sym;
            if (di < 0) {
                di += dx_x2;
            } else {
                di += dx_x2 - dy_x2;
                x0 += dx_sym;
            }
        }
        pixel(x0, y0, color);
    }
    if(auto_up) copy_to_lcd();
}

void gLCD::rect(int x0, int y0, int x1, int y1, int color)
{

    if (x1 > x0) line(x0,y0,x1,y0,color);
    else  line(x1,y0,x0,y0,color);

    if (y1 > y0) line(x0,y0,x0,y1,color);
    else line(x0,y1,x0,y0,color);

    if (x1 > x0) line(x0,y1,x1,y1,color);
    else  line(x1,y1,x0,y1,color);

    if (y1 > y0) line(x1,y0,x1,y1,color);
    else line(x1,y1,x1,y0,color);

    if(auto_up) copy_to_lcd();
}

void gLCD::fillrect(int x0, int y0, int x1, int y1, int color)
{
    int l,c,i;
    if(x0 > x1) {
        i = x0;
        x0 = x1;
        x1 = i;
    }

    if(y0 > y1) {
        i = y0;
        y0 = y1;
        y1 = i;
    }

    for(l = x0; l<= x1; l ++) {
        for(c = y0; c<= y1; c++) {
            pixel(l,c,color);
        }
    }
    if(auto_up) copy_to_lcd();
}



void gLCD::circle(int x0, int y0, int r, int color)
{

    int draw_x0, draw_y0;
    int draw_x1, draw_y1;
    int draw_x2, draw_y2;
    int draw_x3, draw_y3;
    int draw_x4, draw_y4;
    int draw_x5, draw_y5;
    int draw_x6, draw_y6;
    int draw_x7, draw_y7;
    int xx, yy;
    int di;
    //WindowMax();
    if (r == 0) {       /* no radius */
        return;
    }

    draw_x0 = draw_x1 = x0;
    draw_y0 = draw_y1 = y0 + r;
    if (draw_y0 < height()) {
        pixel(draw_x0, draw_y0, color);     /* 90 degree */
    }

    draw_x2 = draw_x3 = x0;
    draw_y2 = draw_y3 = y0 - r;
    if (draw_y2 >= 0) {
        pixel(draw_x2, draw_y2, color);    /* 270 degree */
    }

    draw_x4 = draw_x6 = x0 + r;
    draw_y4 = draw_y6 = y0;
    if (draw_x4 < width()) {
        pixel(draw_x4, draw_y4, color);     /* 0 degree */
    }

    draw_x5 = draw_x7 = x0 - r;
    draw_y5 = draw_y7 = y0;
    if (draw_x5>=0) {
        pixel(draw_x5, draw_y5, color);     /* 180 degree */
    }

    if (r == 1) {
        return;
    }

    di = 3 - 2*r;
    xx = 0;
    yy = r;
    while (xx < yy) {

        if (di < 0) {
            di += 4*xx + 6;
        } else {
            di += 4*(xx - yy) + 10;
            yy--;
            draw_y0--;
            draw_y1--;
            draw_y2++;
            draw_y3++;
            draw_x4--;
            draw_x5++;
            draw_x6--;
            draw_x7++;
        }
        xx++;
        draw_x0++;
        draw_x1--;
        draw_x2++;
        draw_x3--;
        draw_y4++;
        draw_y5++;
        draw_y6--;
        draw_y7--;

        if ( (draw_x0 <= width()) && (draw_y0>=0) ) {
            pixel(draw_x0, draw_y0, color);
        }

        if ( (draw_x1 >= 0) && (draw_y1 >= 0) ) {
            pixel(draw_x1, draw_y1, color);
        }

        if ( (draw_x2 <= width()) && (draw_y2 <= height()) ) {
            pixel(draw_x2, draw_y2, color);
        }

        if ( (draw_x3 >=0 ) && (draw_y3 <= height()) ) {
            pixel(draw_x3, draw_y3, color);
        }

        if ( (draw_x4 <= width()) && (draw_y4 >= 0) ) {
            pixel(draw_x4, draw_y4, color);
        }

        if ( (draw_x5 >= 0) && (draw_y5 >= 0) ) {
            pixel(draw_x5, draw_y5, color);
        }
        if ( (draw_x6 <=width()) && (draw_y6 <= height()) ) {
            pixel(draw_x6, draw_y6, color);
        }
        if ( (draw_x7 >= 0) && (draw_y7 <= height()) ) {
            pixel(draw_x7, draw_y7, color);
        }
    }
    if(auto_up) copy_to_lcd();
}

void gLCD::fillcircle(int x, int y, int r, int color)
{
    int i,up;
    up = auto_up;
    auto_up = 0;   // off
    for (i = 0; i <= r; i++)
        circle(x,y,i,color);
    auto_up = up;
    if(auto_up) copy_to_lcd();  
}

void gLCD::setmode(int mode)
{
    draw_mode = mode;
}

void gLCD::locate(int x, int y)
{
    char_x = x;
    char_y = y;
    oldchar_x = x;
}



int gLCD::columns()
{
    return width() / font[1];
}



int gLCD::rows()
{
    return height() / font[2];
}



int gLCD::_putc(int value)
{
    if (value == '\n') {    // new line
        char_x = oldchar_x;
        char_y = char_y + font[2];
        if (char_y >= height() - font[2]) {
            char_y = 0;
        }
    } else {
        character(char_x, char_y, value);
        if(auto_up) copy_to_lcd();
    }
    return value;
}

void gLCD::character(int x, int y, int c)
{
    unsigned int hor,vert,offset,bpl,j,i,b;
    unsigned char* zeichen;
    unsigned char z,w;

    if ((c < 31) || (c > 127)) return;   // test char range

    // read font parameter from start of array
    offset = font[0];                    // bytes / char
    hor = font[1];                       // get hor size of font
    vert = font[2];                      // get vert size of font
    bpl = font[3];                       // bytes per line

    if (char_x + hor > width()) {
        char_x = 0;
        char_y = char_y + vert;
        if (char_y >= height() - font[2]) {
            char_y = 0;
        }
    }

    zeichen = &font[((c -32) * offset) + 4]; // start of char bitmap
    w = zeichen[0];                          // width of actual char
    // construct the char into the buffer
    for (j=0; j<vert; j++) {  //  vert line
        for (i=0; i<hor; i++) {   //  horz line
            z =  zeichen[bpl * i + ((j & 0xF8) >> 3)+1];
            b = 1 << (j & 0x07);
            if (( z & b ) == 0x00) {
                pixel(x+i,y+j,0);
            } else {
                pixel(x+i,y+j,1);
            }

        }
    }

    char_x += w;
}


void gLCD::set_font(unsigned char* f)
{
    font = f;
}

void gLCD::set_auto_up(unsigned int up)
{
    if(up )
    {
     auto_up = 1;
     }else
     {
     auto_up = 0;
     }
}

unsigned int gLCD::get_auto_up(void){
    return (auto_up);
}




