#include "gLCD.h"
#include "mbed.h"

gLCD::gLCD(PinName rst, PinName rs, PinName e, PinName d0, PinName d1, 
           PinName d2, PinName d3, PinName d4, PinName d5, PinName d6, PinName d7, PinName led) 
     :   _rst(rst), _rs(rs), _e(e), _d(d0, d1, d2, d3, d4, d5, d6, d7), _led(led){

    _rst = 0;           // Put display driver into Reset mode
    _e  = 1;
    _rs = 0;            // Put display controller in command mode

    wait(0.015);        // Wait 15ms to ensure powered up

    _rst = 1;           // Switch reset off
    
    
    /* ADC Select
       Horizontal direction
       0xA0 = SEG1->SEG132 (left to right)
       0xA1 = SEG132->SEG1 (right to left) */
    writeCommand(0xA0);  // 0xA0 = SEG1->SEG132 (left to right)
    
    /* SHL Select
       Vertical direction
       0xC0 = COM1->COM64 (top to bottom)
       0xC8 = COM64->COM1 (bottom to top) */
    writeCommand(0xC0);  // 0xC0 = COM1->COM64
    
    /* LCD bias select
       Screen in use: Displaytech 64128E
       Display duty ratio: 1/65
       Display bias: 1/9
       0xA2 = 1/5 or 1/6 or 1/7 bias
       0xA3 = 1/6 or 1/8 or 1/9 bias 
       Note: at 1/6 bias use 0xA2 for 1/49 duty ratio 
       or 0xA3 for 1/33 duty ratio */
    writeCommand(0xA3);  // 0xA3 = 1/9 bias
    
    /*Power control
      Displaytech screen uses built-in power circuits
      Switch on in order as specified in data sheet
      wait 1ms between each command
      1st - Voltage converter ON = 0x2C
      2nd - Voltage regulator ON = 0x2E
      3rd - Voltage follower  ON = 0x2F */
    writeCommand(0x2C);  //0x2C = Voltage converter ON
    wait(0.001);
    writeCommand(0x2E);  //0x2E = Voltage regulator ON
    wait(0.001);
    writeCommand(0x2F);  //0x2F = Voltage follower ON
    
   /*Regulator resistor select
     Sets the internal resistance ratio used in the internal voltage regulator
     Refer to datasheet p.42
     This works as a corse contrast control
     0x20 = 1.9
     0x21 = 2.19
     0x22 = 2.55
     0x23 = 3.02
     0x24 = 3.61
     0x25 = 4.35
     0x26 = 5.29
     0x27 = 6.48 */     
   writeCommand(0x26);
   
   /*Set reference voltage register
     Used as a fine contrast control
     0x81 = Enter voltage register set mode
     0x00 to 0x3F = 0 to 63 */
   writeCommand(0x81);  //0x81 = Enter voltage register set mode
   writeCommand(0x46);  //0x30 = Set ref voltage to 30
   
   /*Initial display line
     Specify DDRAM line for COM1
     0x40 + display line */
   writeCommand(0x40);  //Set initial line to 0
   
   /*Set page address
     Sets the initial page address to write to
     0xB0 + page address 0 to 8 */
   writeCommand(0xB0); //Initial page set to 0
   
   /*Set column address
     Sets the initial column to write to
     for LSB (b3-b0) 0x00 + first nibble
     for MSB (b7-b4) 0x10 + second nibble 
     0x00 to 0x83 = column 0 to 131 */
   writeCommand(0x00);  //Sets LSB to 0
   writeCommand(0x10);  //Sets MSB to 0  - column is now set to 0
   
   /*Reverse display
     Selects either a normal display or a reverse display
     0xA6 = normal
     0xA7 = reverse */    
   writeCommand(0xA6);  //Sets display to normal
   
   /*Set static indicator
     Sets up a static indicator on the display
     See datasheet p.42 
     This is a 2 instruction cycle
     0xAC = static indicator ON
     0xAD = static indicator OFF 
     Next instruction to set indicator type:
     0x00 = OFF
     0x01 = ON - 1 second blinking
     0x02 = ON - 0.5 second blinking
     0x03 = ON - always ON */
   writeCommand(0xAD);  //Static indicator OFF
   writeCommand(0x00);  //OFF - 0.5 second blinking
   
   /*Display ON/OFF
     Switched the display to on or off
     0xAE = Display OFF
     0xAF = Display ON */
   writeCommand(0xAF);
     
   
   
   
    
    
}



void gLCD::writeCommand(int command)
    {
    _rs = 0;
    wait(0.000040f);
    writeByte(command);
    }

void gLCD::writeData(int data) 
    {
    _rs = 1;
    writeByte(data);
    }

void gLCD::writeByte(int value)
    {
    _led = 1;
    _e = 1;
    wait(0.000040f);    //Allow time for display
    _d = value;
    wait(0.000040f);    //Allow time for display
    _e = 0;             //Load value to display - display driver registers on falling edge of e
    _led = 0;
                  
    }

void gLCD::locate(int page, int column)
    {
    _column = column;
    _page = page;
    writeCommand(0xB0 + page);
    writeCommand(0x00 + (column & 0x0f));
    writeCommand(0x10 + ((column >> 4) & 0x0f));
    }

void gLCD::locateChar(int page, int column)
    {
    column = ((column*6)+1);
    _column = column;
    _page = page;
    writeCommand(0xB0 + page);
    writeCommand(0x00 + (column & 0x0f));
    writeCommand(0x10 + ((column >> 4) & 0x0f));
    }    
    
void gLCD::clearScreen() 
    {
    for(int a = 0; a < 8; a++)
        {
        locate(a,0);
        for(int b = 0; b <128; b++)
            {
            writeData(0x00);
            }
        }
    locate(0,0);
    }

void gLCD::writeChar5x8(int p, int col, int c)
    {
    const int *ptrFont5x8 = font5x8;
    locate(p,col);
    ptrFont5x8+= ((c-32)*6);
    
    for(int a=0; a < 6; a++)
        {
        
        writeData(*ptrFont5x8);
        
        ptrFont5x8++;
        }
       
    }
    


int gLCD::_putc(int value) 
    {
   
    
    
    writeChar5x8(_page, _column, value);
    _column+=6;
        
    
    return value;
}

int gLCD::_getc() {
    return -1;
}


void gLCD::writeBitmap(const int *ptrBitmap)
    {
  
     
        
     for(int a=0; a < 8; a++)
        {
        locate(a,0);
        
        for(int b=0; b < 128; b++)
            {
            writeData(*ptrBitmap);
            ptrBitmap+=8;
            
            
            }
        ptrBitmap-=1023;   
        }

     }   