/*Use #define LSM9DS1_ES before you include this file if you have an engineering sample (older EVBs will have them), to find out if you have one, check your accelerometer output. 
If it is half of what you expected, and you still are on the correct planet, you got an engineering sample
*/


#ifndef LSM9DS1_H
#define LSM9DS1_H

/**
 * Includes
 */
#include "mbed.h"


/**
 * Defines
 */
#ifndef LSM9DS1_ADDRESS
    #define LSM9DS1_ADDRESS        0x6B // address pin low (GND), default for InvenSense evaluation board
#endif

#ifdef LSM9DS1_ES
        #define DOUBLE_ACCELERO
#endif  

/**
 * Registers
 */

 
#define LSM9DS1_ACCEL_XOUT_L_REG   0x28
#define LSM9DS1_ACCEL_YOUT_L_REG   0x2A
#define LSM9DS1_ACCEL_ZOUT_L_REG   0x2C

#define LSM9DS1_GYRO_XOUT_L_REG    0x18
#define LSM9DS1_GYRO_YOUT_L_REG    0x1A
#define LSM9DS1_GYRO_ZOUT_L_REG    0x1C

#define LSM9DS1_WHO_AM_I_REG       0x0F

#define LSM9DS1_CTRL_REG1_G        0x10
#define LSM9DS1_ORIENT_CFG_G       0x13
#define LSM9DS1_CTRL_REG4          0x1E
#define LSM9DS1_CTRL_REG6_XL       0x20
#define LSM9DS1_CTRL_REG9          0x23
 
 /**
  * Definitions
  */

#define LSM9DS1_ACCELERO_RANGE_2G   0
#define LSM9DS1_ACCELERO_RANGE_16G  1
#define LSM9DS1_ACCELERO_RANGE_4G   2
#define LSM9DS1_ACCELERO_RANGE_8G   3

#define LSM9DS1_GYRO_RANGE_245      0
#define LSM9DS1_GYRO_RANGE_500      1
#define LSM9DS1_GYRO_RANGE_2000     3

#define LSM9DS1_SLP_BIT             6


/** LSM9DS1 IMU library.
  *
  * Example:
  * @code
  * Later, maybe
  * @endcode
  */
class LSM9DS1 {
    public:
     /**
     * Constructor.
     *
     * Sleep mode of LSM9DS1 is immediatly disabled
     *
     * @param sda - mbed pin to use for the SDA I2C line.
     * @param scl - mbed pin to use for the SCL I2C line.
     */
     LSM9DS1(PinName sda, PinName scl);
     

     /**
     * Tests the I2C connection by reading the WHO_AM_I register. 
     *
     * @return True for a working connection, false for an error
     */     
     bool testConnection( void );
     
     void initCTRL_REG1_G( void );
     void initORIENT_CFG_G( void );
     void initCTRL_REG4( void );
     void initCTRL_REG6_XL( void );
     
     /**
     * Sets the Accelero full-scale range
     *
     * Macros: LSM9DS1_ACCELERO_RANGE_2G - LSM9DS1_ACCELERO_RANGE_4G - LSM9DS1_ACCELERO_RANGE_8G - LSM9DS1_ACCELERO_RANGE_16G
     *
     * @param range - The two bits that set the full-scale range (use the predefined macros)
     */
     void setAcceleroRange(char range);
     
     /**
     * Reads the accelero x-axis.
     *
     * @return 16-bit signed integer x-axis accelero data
     */   
     int getAcceleroRawX( void );
     
     /**
     * Reads the accelero y-axis.
     *
     * @return 16-bit signed integer y-axis accelero data
     */   
     int getAcceleroRawY( void );
     
     /**
     * Reads the accelero z-axis.
     *
     * @return 16-bit signed integer z-axis accelero data
     */   
     int getAcceleroRawZ( void );
     
     /**
     * Reads all accelero data.
     *
     * @param data - pointer to signed integer array with length three: data[0] = X, data[1] = Y, data[2] = Z
     */   
     void getAcceleroRaw( int *data );
     
     /**
     * Sets the Gyro full-scale range
     *
     * Macros: LSM9DS1_GYRO_RANGE_250 - LSM9DS1_GYRO_RANGE_500 - LSM9DS1_GYRO_RANGE_1000 - LSM9DS1_GYRO_RANGE_2000
     *
     * @param range - The two bits that set the full-scale range (use the predefined macros)
     */
     void setGyroRange(char range);

     /**
     * Reads the gyro x-axis.
     *
     * @return 16-bit signed integer x-axis gyro data
     */   
     int getGyroRawX( void );
     
     /**
     * Reads the gyro y-axis.
     *
     * @return 16-bit signed integer y-axis gyro data
     */   
     int getGyroRawY( void );
     
     /**
     * Reads the gyro z-axis.
     *
     * @return 16-bit signed integer z-axis gyro data
     */   
     int getGyroRawZ( void );
     
     /**
     * Reads all gyro data.
     *
     * @param data - pointer to signed integer array with length three: data[0] = X, data[1] = Y, data[2] = Z
     */   
     void getGyroRaw( int *data );
     
     /**
     * Sets the sleep mode of the MPU6050 
     *
     * @param state - true for sleeping, false for wake up
     */     
     void setSleepMode( bool state );
     
     /**
     * Writes data to the device, could be private, but public is handy so you can transmit directly to the MPU. 
     *
     * @param adress - register address to write to
     * @param data - data to write
     */
     void write( char address, char data);
     
     /**
     * Read data from the device, could be private, but public is handy so you can transmit directly to the MPU. 
     *
     * @param adress - register address to write to
     * @return - data from the register specified by RA
     */
     char read( char adress);
     
     /**
     * Read multtiple regigsters from the device, more efficient than using multiple normal reads. 
     *
     * @param adress - register address to write to
     * @param length - number of bytes to read
     * @param data - pointer where the data needs to be written to 
     */
     void read( char adress, char *data, int length);
     

     
        
     private:

     I2C connection;
     char currentAcceleroRange;
     char currentGyroRange;
     

};



#endif
