#include "mbed.h"
#include "TextLCD.h"
#include "PinDetect.h"
#include "Menu.h"
#include "Comms.h"
#include "PCComms.h"
#include "Gears.h"
#include "LEDS.h"
#include "bigchar.h"
//Test comment
//Refresh interval for LCD
#define LCD_REFRESH_TIME 150000

//Refresh interval for rev LEDs
#define REV_REFRESH_TIME 50000

//Warning lights
DigitalOut warn[] = { (p20), (p19), (p27), (p18) };

//mBED LEDs
DigitalOut debug[] = { (LED1), (LED2), (LED3) };

//Heartbeat LED
DigitalOut heartbeat(LED4);

//Rev LEDS
PwmOut leds[] = { (p24), (p25), (p26), (p23), (p22), (p21) };

//LCD
TextLCD lcd(p5, p6, p7, p8, p9, p10, p11);

//Tickers for refreshing
Ticker lcdRefreshTicker;
Ticker revRefreshTicker;

//Used to animate LEDs for testing
Ticker increment;

//Main car state structure
State car;

//Classes for various parts of the firmware
Menu dashMenu(&lcd, p13, p14, p15); //*LCD, OK, Left, Right
PCComms pc(&car);
Gears gearButtons(p17, p16, p12, &car.gear, &pc); //Up, Neutral, Down, *Current Gear
LEDS revs(leds);

//Refresh the rev LEDs and warning LEDs
void revRefresh()
{
    revs.refresh(car.rpm);
    
    for(int i = 0; i < 4; i++)
    {
        warn[i] = car.warnings & (1 << i);
    }
}

//Refresh the LCD
void lcdRefresh()
{
    //If menu off screen, display HUD
    if(dashMenu.display == false)
    {
        lcd.locate(0, 0);
        lcd.printf("R:%-11.0d", car.rpm);
        lcd.locate(0, 1);
        lcd.printf("S:%-3d T:%-5.0f", car.speed, car.coolant_temp);
        
        write_bigchar(&lcd, 13, car.gear);
    }
    //Otherwise show menu
    else
    {
        dashMenu.refresh();
    }    
    
    //Blink heartbeat
    heartbeat = !heartbeat;
}


//Used to animate LEDs for testing
/*void doIncrement()
{
    if(car.rpm < LIMIT && car.gear > 0)
        car.rpm++;
}*/

//Startup animation
void selfTest()
{
    lcd.printf("    FBR 2012");
    
    //Light up LEDs
    for(int i = 0; i < LEDS::NUM_LEDS; i++)
    {
        leds[i] = true;
        if(i < 4)
            warn[i] = true;
        wait(0.04);
    }
    
    //Turn off LEDs
    for(int i = LEDS::NUM_LEDS - 1; i >= 0; i--)
    {
        leds[i] = false;
        if(i < 4)
            warn[i] = false;
        wait(0.04);
    }
    
    lcd.cls();
}

int main()
{   
    //Initialise state
    car.rpm = 5000;
    car.gear = 0;
    car.speed = 150;
    car.coolant_temp = 21.5;    
    car.throttle_pos = 1;
    car.manifold_pres = 2;
    car.air_temp = 3;
    car.lambda = 5;
    car.accel_x = 7;
    car.accel_y = 8;
    car.oil_temp = 10;
    car.oil_pres = 11;
    car.warnings = 12;

    //Set up menu
    dashMenu.addItem<float>("Coolant Temp  ", "%12.1f\xDF\x43", &car.coolant_temp); // \xDF\x43 -> &#65533;C . Need code for C as otherwise it gets taken as hex digit.
    dashMenu.addItem<unsigned char>("Air Temp      ", "%12d\xDF\x43", &car.air_temp);
    dashMenu.addItem<unsigned char>("Throttle Pos  ", "%13d\xDF", &car.throttle_pos);
    dashMenu.addItem<unsigned char>("Manifold Pres ", "%10d psi", &car.manifold_pres);
    dashMenu.addItem<unsigned char>("Lambda        ", "%14d", &car.lambda);    
    dashMenu.addItem<unsigned char>("Oil Temp      ", "%12d\xDF\x43", &car.oil_temp);
    dashMenu.addItem<unsigned char>("Oil Pressure  ", "%10d psi", &car.oil_pres);
    
    //Set up characters on LCS
    setup_bigchar(&lcd);
    
    //Do bootup animation
    selfTest();
    
    //Start refresh tickers
    lcdRefreshTicker.attach_us(&lcdRefresh, LCD_REFRESH_TIME);
    revRefreshTicker.attach_us(&revRefresh, REV_REFRESH_TIME);
    //increment.attach(&doIncrement, 0.0005);
    
    //Infinite loop - program is interrupt driven
    while(true)
    {
        __WFI();
    }
}