#ifndef FBRDASH_MENU_H
#define FBRDASH_MENU_H

#include "TextLCD.h"
#include "PinDetect.h"
#include <string>

class MenuEntryBase
{
    public:
        bool editable;
        
        string label;
        string format;
        
        virtual void increment() = 0;
        virtual void decrement() = 0;
        virtual string getFormatted() = 0;
};

template <typename T>
class MenuEntry : public MenuEntryBase
{
    public:
        T* value;
        T step;
        
        virtual void increment()
        {
            *value += step;
        }
        
        virtual void decrement()
        {
            *value -= step;
        }
        
        virtual string getFormatted()
        {
            char buffer[16];
            sprintf(buffer, format.c_str(), *value);
            return buffer;
        }
};

class Menu
{
    public:
        bool display;
        bool edit;
        
        Menu(TextLCD* _screen, PinName ok, PinName left, PinName right);
        
        template <typename T>
        void addItem(string label, string format, T* value);
        
        template <typename T>
        void addEditableItem(string label, string format, T* value, T step);
        
        void refresh();
        
        void enter();
        void done();
        void left();
        void right();
        
    private:
        MenuEntryBase* entries[20];
        int entryCount;
        
        PinDetect* btnOK;
        PinDetect* btnLeft;
        PinDetect* btnRight;
        
        TextLCD* screen;
        
        int position;
        bool ignoreNextEnter;
        
        Timeout leftHeldTimeout;
        Timeout rightHeldTimeout;
        
        void leftHeld();
        void cancelLeftHeld();
        void rightHeld();
        void cancelRightHeld();
    
    static const float holdRepeatTime = 0.15; //should be multiple of screen refresh rate or wierdness will ensue
    static const char leftArrow = 0x7F;
    static const char rightArrow = 0x7E;
};

template <typename T>
void Menu::addItem(string label, string format, T* value)
{
    MenuEntry<T>* newEntry = new MenuEntry<T>();
    
    newEntry->value = value;
    newEntry->editable = false;
    
    newEntry->label = label;
    newEntry->format = format;
    
    entries[entryCount] = newEntry;
    entryCount++;
}

template <typename T>
void Menu::addEditableItem(string label, string format, T* value, T step)
{
    MenuEntry<T> newEntry;
    
    newEntry.value = value;
    newEntry.editable = true;
    newEntry.step = step;
    
    newEntry.label = label;
    newEntry.format = format;
    
    entries[entryCount] = newEntry;
    entryCount++;
}

#endif