#include "mbed.h"
#include "TextLCD.h"
#include "PinDetect.h"
#include "Menu.h"
#include "Comms.h"
#include "PCComms.h"
#include "Gears.h"
#include "LEDS.h"
#include "bigchar.h"
//Test comment 2
//Refresh interval for LCD
#define LCD_REFRESH_TIME 150000

//Refresh interval for rev LEDs
#define REV_REFRESH_TIME 50000

//Warning lights
DigitalOut warn[] = { (p20), (p19), (p27), (p18) };

//mBED LEDs
DigitalOut debug[] = { (LED1), (LED2), (LED3) };

//Heartbeat LED
DigitalOut heartbeat(LED4);

//Rev LEDS
PwmOut leds[] = { (p24), (p25), (p26), (p23), (p22), (p21) };

//LCD
TextLCD lcd(p5, p6, p7, p8, p9, p10, p11);

//Tickers for refreshing
Ticker lcdRefreshTicker;
Ticker revRefreshTicker;

//Used to animate LEDs for testing
Ticker increment;

//Main car state structure
State car;

//Initialise CAN
CAN can(p30, p29);

//Classes for various parts of the firmware
Menu dashMenu(&lcd, p16, p17, p12); //*LCD, OK, Left, Right
PCComms pc(&car);
Gears gearButtons(p14, p13, p15, &car.gear, &pc); //Up, Neutral, Down, *Current Gear
LEDS revs(leds);

//Refresh the rev LEDs and warning LEDs
void revRefresh()
{
    CANMessage msg;
    revs.refresh(car.rpm);
    
    if(car.voltage<12){
        warn[0]=1;
    }else{
        warn[0]=0;
    }
    if(car.coolant_temp>110){
        warn[1]=1;
    }else{
        warn[1]=0;
    }
    if(car.rpm==0 and car.gear!=0){
        warn[2]=1;
    }else{
        warn[2]=0;
    }
    if(can.read(msg)) {
        if(msg.id==100 and msg.len==8){
            car.rpm = msg.data[0] + (msg.data[1] << 8);
            car.throttle_pos = msg.data[2];
            car.manifold_pres = msg.data[3];
            car.air_temp = msg.data[4];
            car.coolant_temp = msg.data[5];
            car.lambda = msg.data[6];
        }
        else if(msg.id==200 and msg.len==8){
            car.speed = msg.data[0];
            car.accel_x = msg.data[1];
            car.accel_y = msg.data[2];
            car.gear = msg.data[3];
            car.oil_temp = msg.data[4];
            car.voltage = msg.data[5]/16.0;
        } 
    }

}

//Refresh the LCD
void lcdRefresh()
{
    //If menu off screen, display HUD
    if(dashMenu.display == false)
    {
        lcd.locate(0, 0);
        lcd.printf("%3.0fC%5.1dRPM", car.coolant_temp, car.rpm);
        lcd.locate(0, 1);
        lcd.printf("%2dMPH %3.1fV", car.speed, car.voltage);
        
        write_bigchar(&lcd, 13, car.gear);
    }
    //Otherwise show menu
    else
    {
        dashMenu.refresh();
    }    
    
    //Blink heartbeat
    heartbeat = !heartbeat;
}


//Used to animate LEDs for testing
/*void doIncrement()
{
    if(car.rpm < LIMIT && car.gear > 0)
        car.rpm++;
}*/

//Startup animation
void selfTest()
{
    lcd.printf("    FBR 2012");
    lcd.locate(0,1);
    lcd.printf(" Ready to drive");
    int offset = 0;
    int type=1;
    int blk=7;
    int from=1;
    int to=0;
    int id = offset*262144+type*32768+from*2048+to*128+blk*8;
    CANMessage msg;
    msg = CANMessage(id,0,8,CANData,CANExtended);
    lcd.cls();
    lcd.printf("%x",id);
    can.reset();
    if(can.write(msg)){
    wait(0.01);
    lcd.printf(" %d errors",can.tderror());
    can.reset();
    while(can.read(msg)==false){
    wait(0.1);
    }
    wait(0.5);
    lcd.printf(" %d errors",can.tderror());
    }
    //Light up LEDs
    for(int i = 0; i < LEDS::NUM_LEDS; i++)
    {
        leds[i] = true;
        if(i < 4)
            warn[i] = true;
        wait(0.2);
    }
    
    //Turn off LEDs
    for(int i = LEDS::NUM_LEDS - 1; i >= 0; i--)
    {
        leds[i] = false;
        if(i < 4)
            warn[i] = false;
        wait(0.2);
    }
    
    lcd.cls();
}

int main()
{   
    //Initialise CAN
    can.frequency(500000);
    //Initialise state
    car.rpm = 0;
    car.gear = 2;
    car.speed = 0;
    car.coolant_temp = 0;    
    car.throttle_pos = 0;
    car.manifold_pres = 0;
    car.air_temp = 0;
    car.lambda = 0;
    car.accel_x = 0;
    car.accel_y = 0;
    car.oil_temp = 0;
    car.voltage = 0;

    //Set up menu
    dashMenu.addItem<float>("Coolant Temp  ", "%12.1f\xDF\x43", &car.coolant_temp); // \xDF\x43 -> &#65533;C . Need code for C as otherwise it gets taken as hex digit.
    dashMenu.addItem<unsigned char>("Air Temp      ", "%12d\xDF\x43", &car.air_temp);
    dashMenu.addItem<unsigned char>("Throttle Pos  ", "%13d\xDF", &car.throttle_pos);
    dashMenu.addItem<unsigned char>("Manifold Pres ", "%10d psi", &car.manifold_pres);
    dashMenu.addItem<unsigned char>("Lambda        ", "%14d", &car.lambda);    
    dashMenu.addItem<unsigned char>("Oil Temp      ", "%12d\xDF\x43", &car.oil_temp);
    
    //Set up characters on LCS
    setup_bigchar(&lcd);
    
    //Do bootup animation
    selfTest();
    
    //Start refresh tickers
    lcdRefreshTicker.attach_us(&lcdRefresh, LCD_REFRESH_TIME);
    revRefreshTicker.attach_us(&revRefresh, REV_REFRESH_TIME);
    //increment.attach(&doIncrement, 0.0005);
    
    //Infinite loop - program is interrupt driven
    while(true)
    {
        __WFI();
    }
}