/* HTTPFile.cpp */
#include "HTTPFile.h"

#include <cstring>

#define OK 0
#define NG -1

using std::memcpy;
using std::strncpy;
using std::strlen;

HTTPFile::HTTPFile(const char* path) :
    m_fp(NULL),
    m_path(path),
    m_len(0),
    m_chunked(false)
{
}

HTTPFile::~HTTPFile()
{
    closeFile();
}

void HTTPFile::clear()
{
    closeFile();
    //Force reopening
}

/*virtual*/ void HTTPFile::readReset()
{
    if(m_fp) 
        fseek(m_fp, 0, SEEK_SET);
}

/*virtual*/ int HTTPFile::read(char* buf, size_t len, size_t* pReadLen)
{
    if(!openFile("r")) //File does not exist, or I/O error...
        return NG;
    
    *pReadLen = fread(buf, 1, len, m_fp);
    if( feof(m_fp) )
    {
        //File read completely, we can close it
        closeFile();
    }
    return OK;
}

/*virtual*/ void HTTPFile::writeReset()
{
    if(m_fp) 
        fseek(m_fp, 0, SEEK_SET);
}

/*virtual*/ int HTTPFile::write(const char* buf, size_t len)
{
    if(!openFile("w")) //File does not exist, or I/O error...
        return NG;
    
    len = fwrite(buf, 1, len, m_fp);
    //DBG("Written %d bytes in %d\n", len, m_fp);
    if( (!m_chunked && (ftell(m_fp) >= m_len)) || (m_chunked && !len) )
    {
        //File received completely, we can close it
        closeFile();
    }
    return len;
}

/*virtual*/ int HTTPFile::getDataType(char* type, size_t maxTypeLen) //Internet media type for Content-Type header
{
    strncpy(type, "text/plain", maxTypeLen-1);
    type[maxTypeLen-1] = '\0';
    return OK;
}

/*virtual*/ void HTTPFile::setDataType(const char* type) //Internet media type from Content-Type header
{
    //Do not really care here
}

/*virtual*/ bool HTTPFile::getIsChunked() //For Transfer-Encoding header
{
    return false;
}

/*virtual*/ void HTTPFile::setIsChunked(bool chunked) //For Transfer-Encoding header
{
    m_chunked = chunked;
}

/*virtual*/ size_t HTTPFile::getDataLen() //For Content-Length header
{
    return m_len;
}

/*virtual*/ void HTTPFile::setDataLen(size_t len) //For Content-Length header, or if the transfer is chunked, next chunk length
{
    if(!m_chunked)
        m_len = len; //Useful so that we can close file when last byte is written
}

bool HTTPFile::openFile(const char* mode) //true on success, false otherwise
{
    if(m_fp) 
        return true;
    
    m_fp = fopen(m_path.c_str(), mode);
    if(m_fp && mode[0]=='r')
    {
        //Seek EOF to get length
        fseek(m_fp, 0, SEEK_END);
        m_len = ftell(m_fp);
        fseek(m_fp, 0, SEEK_SET); //Goto SOF
    }
    
    //DBG("fd = %d\n", m_fp);
    
    if(!m_fp) 
        return false;
    
    return true;
}

void HTTPFile::closeFile()
{
    if(m_fp)
        fclose(m_fp);
}
