/**********************************************************************************
 * This program monitors the power on state, kill state and battery voltage on
 * Woodstream mouse trap using the Multitech mDot and UDK2 development system.
 * The power on state is monitored on the PA_4(UDK2 Pin D10)and the kill state is
 * monitored on the PA_5 (UDK2 Pin D13) Digital Input Pins.  The battery voltage
 * is monitored on the PB_1 (UDK2 Pin A0) Analog Input.  The status of these pins
 * are transferred from the mDot LoRa to the Conduit LoRa gateway.  The status is
 * also shown on the USB Debug Terminal Window.
 *********************************************************************************/

#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include <string>
#include <vector>
#include <algorithm>
#include "PinDetect.h"

#define MIN_CHANGE_BATTERY_VOLTAGE 0
#define LED_TIME 1
#define BATTERY_TIME 2.0
#define KILL_STATUS_TIME 3.0
#define POWER_ON_TIME 4.0

mDot* dot;
Ticker ledTimer;
Ticker batteryTimer;
Ticker killStatusTimer;
Ticker powerOnTimer;
     
AnalogIn batteryVoltage(PB_1);
DigitalIn powerOn(PA_4);
DigitalIn killStatus(PA_5);
DigitalOut transmitLED(PA_0);

// Configuration variables
static std::string config_network_name = "LORA_NET";
static std::string config_network_pass = "BLUE_HEN1";
static uint8_t config_frequency_sub_band = 7;

//Global Variables
bool readyToSendBatteryVoltage;
bool readyToSendKillStatus;
bool readyToSendPowerOn;
float lastBatteryVoltage;
uint8_t lastKillStatus;
uint8_t lastPowerOn;

//Function prototypes
void ledWrite();
void periodicSendTock();
void batteryRead();
void killStatusRead();
void powerOnRead();
void printError(mDot* dot, int32_t returnCode);
void printVersion();
bool setFrequencySubBand(uint8_t subBand);
bool setNetworkName(const std::string name);
bool setNetworkPassphrase(const std::string passphrase);
bool setTxDataRate(const int dataRate);
bool setPower(uint8_t power);
bool setAck(uint8_t retries);
bool joinNetwork();
bool send(const std::string text);
char latestData[100];


int main()
{
    bool configFail = false;
    readyToSendBatteryVoltage = false;
    readyToSendKillStatus = false;
    readyToSendPowerOn = false;
    lastBatteryVoltage = 0.0;
    lastKillStatus = killStatus;
    lastPowerOn = powerOn;
      
    //Start LED startup sequence
    ledTimer.attach(&ledWrite, LED_TIME);

    printf("\r\n\r\n");
    printf("=====================================\r\n");
    printf("WoodStream LoRa Mousetrap Demo \r\n");
    printf("=====================================\r\n");
    printVersion();

    // get the mDot handle
    dot = mDot::getInstance();  
    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);

    //*******************************************
    // configuration
    //*******************************************
    // reset to default config so we know what state we're in
    dot->resetNetworkSession();
    dot->resetConfig();

    // set up the mDot with our network information: frequency sub band, network name, and network password
    // these can all be saved in NVM so they don't need to be set every time - see mDot::saveConfig()    
    configFail = !setFrequencySubBand(config_frequency_sub_band);
    configFail |= !setNetworkName(config_network_name);
    configFail |= !setNetworkPassphrase(config_network_pass);
    configFail |= !setTxDataRate(mDot::SF_8);
    configFail |= !setPower(20);                                    // Reduce latency for 868 units
    configFail |= !setAck(0);                                       // Disable ack for less latency
    
    printf("Configration %s\r\n",configFail?"Failed":"Passed");
     
    // save this configuration to the mDot's NVM
    printf("saving config\r\n");
    if (! dot->saveConfig())
    {
        logError("failed to save configuration");
    }
    //*******************************************
    // end of configuration
    //*******************************************
  
    // keep trying to join the network
    while (!joinNetwork()) 
    { 
        wait(200); 
        dot->resetNetworkSession(); 
    }

    // Stop LED startup sequence & configure them for operation
    transmitLED = 1;
    // start all paramter timers to sample analog and digital inputs
    batteryTimer.attach(batteryRead, BATTERY_TIME);
    killStatusTimer.attach(killStatusRead, KILL_STATUS_TIME);
    powerOnTimer.attach(powerOnRead, POWER_ON_TIME);

    while (1) 
    {
        // is there anything to send
        if(readyToSendBatteryVoltage)
        {       
             sprintf(latestData,"Voltage %2.2f",(double)lastBatteryVoltage);
            printf("%s\r\n",latestData);
            send(latestData);
            readyToSendBatteryVoltage = false;
        }
        if(readyToSendKillStatus)
        {       
            sprintf(latestData,"KillStatus %d",lastKillStatus);
           printf("%s\r\n",latestData);
           send(latestData);
            readyToSendKillStatus = false;
        }
        if(readyToSendPowerOn)
        {       
            sprintf(latestData,"PowerOn %d",lastPowerOn);
            printf("%s\r\n",latestData);
            send(latestData);
            readyToSendPowerOn = false;
        }
     }
}

void ledWrite()
{
    transmitLED = !transmitLED;
}

void batteryRead() 
{
    lastBatteryVoltage = batteryVoltage * (float)3.3;
    readyToSendBatteryVoltage = true;
}

void killStatusRead() 
{   
    lastKillStatus = killStatus;
    readyToSendKillStatus = true;
}

void powerOnRead() 
{   
    lastPowerOn = powerOn;
    readyToSendPowerOn = true;
 }

void printVersion()
{
    printf("%s Built on: %s %s\r\n\r\n", dot->getId().c_str(),__DATE__,__TIME__);
}

bool setFrequencySubBand(uint8_t subBand)
{
    int32_t returnCode;
    
    printf("Setting frequency sub band to '%d'...\r\n", subBand);
    if ((returnCode = dot->setFrequencySubBand(subBand)) != mDot::MDOT_OK) {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setNetworkName(const std::string name)
{
    int32_t returnCode;
    
    printf("Setting network name to '%s'...\r\n", name.c_str());
    if ((returnCode = dot->setNetworkName(name)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setNetworkPassphrase(const std::string passphrase)
{
    int32_t returnCode;
    
    printf("Setting passphrase to '%s'...\r\n", passphrase.c_str());
    if ((returnCode = dot->setNetworkPassphrase(passphrase)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setTxDataRate(const int dataRate)
{
   int32_t returnCode;
   
    printf("Setting TX Spreading factor to %d...\r\n",dataRate);
    if ((returnCode = dot->setTxDataRate(dataRate)) != mDot::MDOT_OK) 
    {
        logError("Failed To Set Tx Datarate %d:%s", returnCode, mDot::getReturnCodeString(returnCode).c_str());
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setPower(uint8_t power)
{
    int32_t returnCode;
    printf("Setting tx power to '%d'...\r\n", power);
    if ((returnCode = dot->setTxPower(power)) != mDot::MDOT_OK) 
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool joinNetwork()
{
    int32_t returnCode;
    printf("\r\nJoining network...\r\n");
    if ((returnCode = dot->joinNetworkOnce()) != mDot::MDOT_OK) 
    {
        printError(dot, returnCode);
        return false;
    }
    printf("Network Joined!\r\n");
    return true;
}

bool setAck(uint8_t retries)
{
    int32_t returnCode;
    printf("Setting ack to '%d'...\r\n", retries);
    if ((returnCode = dot->setAck(retries)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool send(const std::string text)
{
    int32_t returnCode;
    uint32_t timeTillSend = dot->getNextTxMs();
    if (timeTillSend != 0) {
        printf("waiting %lu ms to send\r\n", timeTillSend);
        return false;
    }

    printf("Sending data...  ");
    std::vector<uint8_t> data(text.begin(), text.end());
    if ((returnCode = dot->send(data, 1)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    printf("Data sent!\r\n");
    return true;
}

void printError(mDot* dot, int32_t returnCode)
{
    std::string error = mDot::getReturnCodeString(returnCode) + " - " + dot->getLastError();
    printf("%s\r\n", error.c_str());
 }
