/* ========================================
 * Filename: common.h
 *
 * Description: common defintions and structures
 *
 * Copyright J-Factor Embedded Technologies 2015
 * Copyright TJM Embedded Software 2015
 * All Rights Reserved
 * UNPUBLISHED, LICENSED SOFTWARE.
 *
 * CONFIDENTIAL AND PROPRIETARY INFORMATION
 * WHICH IS THE PROPERTY OF J-Factor Embedded Technologies.
 *
 * ========================================
*/
/** \file
 * \brief contains constants and prototypes needed by other files.
 */
#ifndef COMMON_H
#define COMMON_H
 
#include <time.h>
#include "mbed.h"
 
/* Project Defines */
#define FALSE  0
#define TRUE   1
#define TRANSMIT_BUFFER_SIZE  80    //!< maximum number of characters to write to console
#define LINE_SIZE  80               //!< maximum number of characters to read from console
#define NUM_LINES 10                //!< number of command lines to save       

/* version history */
/*
Version Author  Date        Description
0.1     TJM     01/24/15    Initial Draft
0.2     TJM     01/24/166   first with CLI interface
*/
 
#define VERSION_MAJOR 0
#define VERSION_MINOR 2
#define VERSION_DATE "01/26/16"

typedef unsigned char uint8;
typedef signed char int8;
typedef unsigned short uint16;
typedef signed short int16;
typedef unsigned long uint32;
typedef signed long int32;
typedef unsigned int uint;
  
/* parameter type */
#define BINARY  0
#define DIGITAL 1
#define REGISTER 2
#define ANALOG  3

/* command defintions */
#define HELP        0
#define VERSION     1
#define READ        2
#define READ_REG    3
#define WRITE       4
#define WRITE_REG   5
#define INFO        6
#define FIND        7
#define RUN         8
#define STOP        9
#define DATE        10
#define HVSTATE     11

/* mode */
#define READONLY    0
#define WRITEONLY   1
#define READWRITE   2

#define MAX_REGISTER_DATA 128

/* parameter names */
enum 
{
     spi,
     i2c,
} ;    
     
typedef   uint8 (*Read)(void);                                          //!< read function template to return 8 bit unsigned int
typedef   uint16 (*ReadRegLength)(uint16 regAddress, uint8 *regData, uint16 length);  //!< read function template to return 8 bit unsigned int
typedef   void (*Write8)(uint8 value);      //!< write function template to write 8 bit unsigned value
typedef   void (*WriteReg8)(uint16 regAddress, uint8 *regData);      //!< write function template to write 8 bit unsigned value
typedef   void (*Write16)(uint16 value);    //!< write function template to write 16 bit unsigned value
typedef   void (*WriteAnalog)(float fltValue);    //!< write function template to write analog value

/* paramater structure */
struct parameterInfo                        //!< paramter information structure
{
    char *name;                             //!< paramter name
    char *port;                             //!< port location (if applicable) on device
    char *connector;                        //!< connector location (if applicable) on board
    char *description;                      //!< brief parameter function and use
    uint8 type;                             //!< binary, digital, or analog
    char *uints;                            //!< parameter units (if applicable)
    uint8 mode;                             //!< read only, write only, read write
    uint16 initialInt;                      //!< initial value if binary or digital
    uint16 minInt;                          //!< minimum value if binary or digital
    uint16 maxInt;                          //!< maximum value if binary or digital
    float initialFloat;                     //!< initial value if analog
    float minFloat;                         //!< minimum value if analog
    float maxFloat;                         //!< maximum value if analog
    Read ptrRead;                           //!< read function needed for this parameter  (null if not needed)
    ReadRegLength ptrReadRegLength;         //!< read function needed for this parameter  (null if not needed)
    Write8 ptrWrite8;                       //!< 8 bit write function needed for this paramater (null if not needed) 
    WriteReg8 ptrWriteReg8;                 //!< 8 bit write function needed for this paramater (null if not needed) 
    Write16 ptrWrite16;                     //!< 16 bit write function needed for this paramater (null if not needed) 
    WriteAnalog ptrWriteAnalog;             //!< analog write function needed for this paramater (null if not needed) 
};


/* function prototypes */
void printVersion();
uint8 getLine(); 
void executeCmd();
int8 getCmd();
void setDate(char *str);
void date(void);
int8 getParameter(char *param);
int8 findParameter(char *string);
uint8 readParam(uint8 paramNum);
void writeParam(uint8 paramNum,uint8 *value);
float strtofloat (char *str);
uint16 power(uint8 num, uint8 pow);
void displayParameter(uint8 num);
uint8 tolower(uint8 c);
uint16 readSPIReg(uint16 regAddress, uint8 *regData, uint16 length);
void writeSPIReg(uint16 regAddress, uint8 *regData);
void SPIRead(uint16 addr, uint8 *data, int length);
void SPIWrite(uint16 addr, uint8 *data, int length);
uint16 readI2CReg(uint16 regAddress, uint8 *regData, uint16 length);
void writeI2CReg(uint16 regAddress, uint8 *regData);
void I2CRead(uint16 addr, uint8 *data, int length);
void I2CWrite(uint16 addr, uint8 *data, int length);

/* externals */
extern Serial pc;
extern char* c_time_string;
extern char time_string[];
extern char TransmitBuffer[TRANSMIT_BUFFER_SIZE];
extern time_t epoch;
extern struct parameterInfo parameters[];
extern uint16 bytesRead;

#endif
/* [] END OF FILE */
