/* ========================================
 * Filename: commandLine.c
 *
 * Description: functions and operations needed for  command line operation
 *
 * Copyright J-Factor Embedded Technologies 2015
 * Copyright TJM Embedded Software 2015
 * All Rights Reserved
 * UNPUBLISHED, LICENSED SOFTWARE.
 *
 * CONFIDENTIAL AND PROPRIETARY INFORMATION
 * WHICH IS THE PROPERTY OF J-Factor Embedded Technologies.
 *
 * ========================================
*/
//#include <project.h>
#include "mbed.h"
#include "stdio.h"
#include "common.h"

/** \file
 * \brief command line - contains all functions related to the command line.
 */

/* externals */

/* Globals */
char line[LINE_SIZE];                       //!< character array read 
char lastLine[NUM_LINES][LINE_SIZE];        //!< array of previous command lines
int8 indexNextLineIn = 0;                   //!< index of next position to save current command line
int8 indexNextLineOut = 0;                  //!< index of next position to read out saved comman dlines
int8 linesInBuffer = 0;                     //!< number of lines in  command line buffer
uint8 enableCLI = 1;
uint8 escapeSeen = 0;                       //!< escape character seen
int8 charCount;                            //!< number of characters in the current command line
int8 cursorPosition;                       //!< current cursor position in the current command line
char TransmitBuffer[TRANSMIT_BUFFER_SIZE];  //!< charaters to be sent to the console
uint8 ptrCmd = 0;
uint8 binaryParameter;                      //!< result of any 8 bit integer read operation
uint16 digitalParameter;                    //!< result of any 16 bit integer read operation
float analogParameter;                      //!< result of any floating point read operation
char *validCmds[] = {"help","version","read","readReg","write","writeReg","info","find","run","stop","date","hvstate"};    //!< array of valid commands - case insensitive    
uint16 regAddress;
uint16 regLength;
uint8 regData[MAX_REGISTER_DATA];

/** 
 * \fn getLine
 *
 * \brief Reads characters from UART and retuns a line when new line seen.
 * if enableCLI == 1 - process input characters<p>
 * if enableCLI == 0 - return 0<p>
 *
 * will call UART_1_CetChar() to get the next character.<p>
 * if ch == 0 - no character available<p>
 * if ch != 0 - add ch to line array<p>
 * features: <p>
 * up arrow will go through last NUM_LINES commands. <p>
 * backspace will remove last typed character<p>
 *
 * \return uint8 - 0 - no character ready or line not finished, 1 - line complete (CR seen)<p>
 *
 */
uint8 getLine()
{
     /* Variable to store UART received character */
    uint8 ch,i;
    
    /* check if CLI enabled */
    if(enableCLI == 0)
        return 0;
    
    /* Non-blocking call to get the latest data recieved  */
    if(pc.readable())
        ch = pc.getc();
    else
        ch = '\0';
        
    /* Set flags based on UART command */
    switch(ch)
    {
        case 0:
            /* No new data was recieved */
            return 0;
         
        case 0x1B:                      /* up arrow seen */
            escapeSeen = 1;
            break;
        
        case '\n':                      /* new line seen */
        case '\r':
            pc.printf("\r\n");
            line[charCount] = '\0';
            charCount = 0;
            cursorPosition = charCount;
            escapeSeen = 0;
            if(strlen(line) > 0)
            {
                strcpy(&lastLine[indexNextLineIn][0],line); /* save current command line in  buffer */
                if(linesInBuffer < NUM_LINES)
                    linesInBuffer++;
                indexNextLineIn++;                          /* point to next free space in command line buffer */
                if(indexNextLineIn == NUM_LINES - 1)        /* wrap around if past end */
                    indexNextLineIn = 0;
                indexNextLineOut = indexNextLineIn;
            }
            return 1;
            
        default:
//            sprintf(TransmitBuffer,"char: %02X %c  charCount: %d\r\n",ch,ch,charCount);
//            UART_1_PutString(TransmitBuffer);
            if( (escapeSeen == 1) && (ch == '[') )
            {
                escapeSeen++;
                break;
            }
            else if( (escapeSeen == 2) && (ch == 'A') )      /* look for up arrow escape sequence */
            {
                if(linesInBuffer == 0)
                    break;
                indexNextLineOut--;                     /* point to last line in */
                if(indexNextLineOut < 0)                /* check if wrap around */
                {
                    if(linesInBuffer < NUM_LINES)
                        indexNextLineOut = linesInBuffer -1;
                    else
                        indexNextLineOut = NUM_LINES - 1;
                }
                strcpy(line,&lastLine[indexNextLineOut][0]); /* copy command line from buffer to command line */
                charCount = strlen(line);
                cursorPosition = charCount;
                escapeSeen = 0;
                pc.printf("\r");
                for (i=0;i<sizeof(line);i++)                 /* clear line of last command */
                    pc.printf(" ");
                sprintf(TransmitBuffer,"\r%s cmd>%s",time_string,line);    /* restore with selected command */
                pc.printf(TransmitBuffer);
                break;
            }
            else if( (escapeSeen == 2) && (ch == 'D') )      /* look for left arrow escape sequence */
            {
                if( cursorPosition > 0 )
                {
                    pc.printf("\b");
                    cursorPosition--;
                }
                escapeSeen = 0;
                break;
            }
            else if( (escapeSeen == 2) && (ch == 'C') )      /* look for right arrow escape sequence */
            {
                if( cursorPosition < charCount )
                {
                    sprintf(TransmitBuffer,"%c",line[cursorPosition]);
                    pc.printf(TransmitBuffer);
                    cursorPosition++;
                }
                escapeSeen = 0;
                break;
            }
            else if( (ch == 0x7F) || (ch == '\b') )  /* backspace seen */
            {
                if( (charCount > 0) && (charCount == cursorPosition) )
                {
                    pc.printf("\b \b");
                    charCount--;
                    cursorPosition = charCount;
                    line[charCount] = '\0';
                }
            }
            else
            {
                if(charCount < (LINE_SIZE - 1))
                {
                    sprintf(TransmitBuffer,"%c",ch);
                    pc.printf(TransmitBuffer);
                    if(charCount != cursorPosition)         /* move characters at cursor up to fit in new character */
                    {
                        for(i = charCount ; i > cursorPosition; i--)
                            line[i] = line[i-1];
                        line[cursorPosition] = ch;
                        line[++charCount] = '\0';
                        for (i=cursorPosition + 1;i<charCount;i++) 
                        {
                            sprintf(TransmitBuffer,"%c",line[i]);
                            pc.printf(TransmitBuffer);
                        }
                        cursorPosition = charCount;
                    }
                    else
                    {
                        line[charCount++] = ch;
                        cursorPosition = charCount;
                    }
                 }
            }
            break;    
    }
    return 0;
}


/** 
 * \fn executeCmd
 *
 * \brief executes the command in line array.
 *
 * calls getCmd to parse command and then switches to proper command handler.
 * command list kept short for simplicity.<p>
 * read, write, and info command will read and verify parametsr on  command line.
 * \return None<p>
 *
 */
void executeCmd()
{
    int8 cmd;
    uint8 type,i;
    int32 convertFloat1,convertFloat2;
    float floatValue;
    int value,num, pos;
    char str[40] = "", str2[40] = "";
    uint8 statusRegister;
    
    cmd = getCmd();
//    sprintf(TransmitBuffer,"cmd: %d ptrCmd: %d\r\n",cmd,ptrCmd);
//    UART_1_PutString(TransmitBuffer);
    
    /* execute command */
    switch (cmd)
    {
        case HELP:
            pc.printf("Valid Commands (case insensitive)\r\n");
            pc.printf("find [string] - find all parameters or all parameters that contain the string\r\n");
            pc.printf("help - display list of valid commands\r\n");
            pc.printf("info [parameter]- display info on all or parameter\r\n");
            pc.printf("read parameter - read value of parameter\r\n");
            pc.printf("readReg parameter register length - read N values from parameter registers\r\n");
            pc.printf("version - display version information\r\n");
            pc.printf("write parameter register value - write value to parameter register\r\n");
            pc.printf("writeReg parameter value - write value to parameter\r\n");
            pc.printf("run - start control running\r\n");
            pc.printf("stop - stop control running\r\n");
            pc.printf("date [date string]- display/set current time and date\r\n");
            pc.printf("hvstate - current HV status\r\n");
            break;
           
        case DATE:
            num = sscanf(&line[ptrCmd],"%s",str);       /* optional parameter on the command line - date string */
            if(num > 0)
            {
                setDate(&line[ptrCmd]);
            }
            date();
            sprintf(TransmitBuffer,"current time is %s\r\n",c_time_string);
            pc.printf(TransmitBuffer);
            break;
             
        case VERSION:
            printVersion();
//            sprintf(TransmitBuffer,"PSK15-5 CSE: V%d.%02d %s\r\n",VERSION_MAJOR,VERSION_MINOR,VERSION_DATE);
//            UART_1_PutString(TransmitBuffer);
//            sprintf(TransmitBuffer,"Built on: %s %s\r\n",__DATE__,__TIME__);
//            UART_1_PutString(TransmitBuffer);
            break;
          
        case READ:
            num = sscanf(&line[ptrCmd],"%s",str);       /* expecting one parameter on the command line - parameter name */
            if(num == 1)
            {
                num = getParameter(str);                /* get paramter position in parameters array from paramter name */
                if (num >= 0)
                {
                    type = readParam(num);              /* read paramter value and return parameter type */
                    if(type == BINARY)
                    {
                        sprintf(TransmitBuffer,"%s: %d %s\r\n",parameters[num].name,binaryParameter,parameters[num].uints);
                    }
                    else if(type == DIGITAL)
                    {
                        sprintf(TransmitBuffer,"%s: %d %s\r\n",parameters[num].name,digitalParameter,parameters[num].uints);
                    }
                    else if(type == ANALOG)
                    {
                        convertFloat1 = analogParameter * 100.0;
                        sprintf(TransmitBuffer,"%s: %ld.%02ld %s\r\n",parameters[num].name,convertFloat1/100,convertFloat1%100,parameters[num].uints);
                    }
                     pc.printf(TransmitBuffer);
                }
                else
                {
                    sprintf(TransmitBuffer,"Invalid parameter\r\nType help for list of valid parameters\r\n");
                    pc.printf(TransmitBuffer);
               }
            }
            else
            {
                sprintf(TransmitBuffer,"Missing parameter\r\nformat: read parameter\r\n");
                pc.printf(TransmitBuffer);
           }
            break;
         
        case READ_REG:
            num = sscanf(&line[ptrCmd],"%s %X %X",str, &regAddress, &regLength); 
            if(num < 0)
            {
                sprintf(TransmitBuffer,"Missing parameter\r\nformat: read parameter [Reg Address (hex)] [Length (hex)]\r\n");
                pc.printf(TransmitBuffer);
                return;
            }
            pos = getParameter(str);                /* get paramter position in parameters array from paramter name */
            if(pos == -1)
            {
                sprintf(TransmitBuffer,"Invalid parameter\r\nType help for list of valid parameters\r\n");
                pc.printf(TransmitBuffer);
                return;
           }
            switch(num)
            {
                 case 1:
                    regAddress = 0;
                    regLength = 1;
                    break;
               
                case 2:
                     regLength = 1;
                    break;
               
                case 3:
                     break;
           
            }
            type = readParam(pos);              /* read paramter value and return parameter type */
            for(i=0; i< bytesRead; i++)
            {
                if( i == 0 )
                    pc.printf("%04X ",regAddress);
                else if( (i % 0x10) == 0 )
                    pc.printf("\r\n%04X ",regAddress + i);
                pc.printf("%02X ",regData[i]);
            }
            pc.printf("\r\n");
            break;
 #if 0       
        case WRITE:
            floatValue = 0.0;
            num = sscanf(&line[ptrCmd],"%s %s",str,str2);       /* expecting two parameters on the command line - parameter name and parameter value */
            if(num == 2)
            {
                num = getParameter(str);                        /* get paramter position in parameters array from paramter name */
                if (num >= 0)
                {
                    type = readParam(num);                      /* read paramter value and return parameter type */
                    if(type == ANALOG)
                    {
                        floatValue = strtofloat(str2);          /* paramters name and floating point number */
                    }
                    else
                    {
                        sscanf(&line[ptrCmd],"%s %d",str,&value);   /* parameters name and integer value */
                    }
                    if(type == BINARY)
                    {
                        if(value > parameters[num].maxInt)
                        {
                           sprintf(TransmitBuffer,"%d exceeds maximum value %d\r\n",value,parameters[num].maxInt);
                        }
                        else
                        {
                            writeParam(num,value,floatValue);
                            type = readParam(num);
                            sprintf(TransmitBuffer,"%s: %d %s\r\n",parameters[num].name,binaryParameter,parameters[num].uints);
                        }
                    }
                    else if(type == DIGITAL)
                    {
                        if(value > parameters[num].maxInt)
                        {
                           sprintf(TransmitBuffer,"%d %s exceeds maximum value %d %s\r\n",value,parameters[num].uints,parameters[num].maxInt,parameters[num].uints);
                        }
                        else if(value < parameters[num].minInt)
                        {
                           sprintf(TransmitBuffer,"%d %s less than minimum value %d %s\r\n",value,parameters[num].uints,parameters[num].minInt,parameters[num].uints);
                        }
                        else
                        {
                            writeParam(num,value,floatValue);
                            type = readParam(num);
                            sprintf(TransmitBuffer,"%s: %d %s\r\n",parameters[num].name,digitalParameter,parameters[num].uints);
                        }
                    }
                    else if(type == ANALOG)
                    {
                        if(floatValue > parameters[num].maxFloat)
                        {
                            convertFloat1 = floatValue * 100;
                            convertFloat2 = parameters[num].maxFloat * 100;
                            sprintf(TransmitBuffer,"%ld.%02ld %s exceeds maximum value %ld.%02ld %s\r\n",convertFloat1/100,convertFloat1%100,parameters[num].uints,convertFloat2/100,convertFloat2%100,parameters[num].uints);
                        }
                        else if(floatValue < parameters[num].minFloat)
                        {
                            convertFloat1 = floatValue * 100;
                            convertFloat2 = parameters[num].minFloat * 100;
                            sprintf(TransmitBuffer,"%ld.%02ld %s is less than value %ld.%02ld %s\r\n",convertFloat1/100,convertFloat1%100,parameters[num].uints,convertFloat2/100,convertFloat2%100,parameters[num].uints);
                        }
                        else
                        {
                            writeParam(num,value,floatValue);
                            type = readParam(num);
                            convertFloat1 = analogParameter * 100;
                            sprintf(TransmitBuffer,"%s: %ld.%02ld %s\r\n",parameters[num].name,convertFloat1/100,convertFloat1%100,parameters[num].uints);
                        }
                    }
                     pc.printf(TransmitBuffer);
                }
                else
                {
                    pc.printf("Invalid parameter\r\nType help for list of valid parameters\r\n");
                }
            }
            else
            {
                pc.printf("Missing parameter(s)\r\nformat: write parameter value\r\n");
            }
            break;
 #endif       
        case WRITE_REG:
             num = sscanf(&line[ptrCmd],"%s %X %X",str, &regAddress, &regData); 
            if(num != 3)
            {
                pc.printf("Missing parameter\r\nformat: write parameter Reg Address (hex) Value (hex)\r\n");
                return;
            }
            pos = getParameter(str);                /* get paramter position in parameters array from paramter name */
             if(pos == -1)
            {
                sprintf(TransmitBuffer,"Invalid parameter\r\nType help for list of valid parameters\r\n");
                pc.printf(TransmitBuffer);
                return;
           }
           writeParam(pos,regData);              /* write paramter value and return parameter type */
            regLength = 1;
            readParam(pos);
            for(i=0; i< bytesRead; i++)
            {
                if( i == 0 )
                    pc.printf("%04X ",regAddress);
                else if( (i % 0x10) == 0 )
                    pc.printf("\r\n%04X ",regAddress + i);
                pc.printf("%02X ",regData[i]);
            }
            pc.printf("\r\n");
            break;
#if 0
           floatValue = 0.0;
            num = sscanf(&line[ptrCmd],"%s %s",str,str2);       /* expecting two parameters on the command line - parameter name and parameter value */
            if(num == 2)
            {
                num = getParameter(str);                        /* get paramter position in parameters array from paramter name */
                if (num >= 0)
                {
                    type = readParam(num);                      /* read paramter value and return parameter type */
                    if(type == ANALOG)
                    {
                        floatValue = strtofloat(str2);          /* paramters name and floating point number */
                    }
                    else
                    {
                        sscanf(&line[ptrCmd],"%s %d",str,&value);   /* parameters name and integer value */
                    }
                    if(type == BINARY)
                    {
                        if(value > parameters[num].maxInt)
                        {
                           sprintf(TransmitBuffer,"%d exceeds maximum value %d\r\n",value,parameters[num].maxInt);
                        }
                        else
                        {
                            writeParam(num,value,floatValue);
                            type = readParam(num);
                            sprintf(TransmitBuffer,"%s: %d %s\r\n",parameters[num].name,binaryParameter,parameters[num].uints);
                        }
                    }
                    else if(type == DIGITAL)
                    {
                        if(value > parameters[num].maxInt)
                        {
                           sprintf(TransmitBuffer,"%d %s exceeds maximum value %d %s\r\n",value,parameters[num].uints,parameters[num].maxInt,parameters[num].uints);
                        }
                        else if(value < parameters[num].minInt)
                        {
                           sprintf(TransmitBuffer,"%d %s less than minimum value %d %s\r\n",value,parameters[num].uints,parameters[num].minInt,parameters[num].uints);
                        }
                        else
                        {
                            writeParam(num,value,floatValue);
                            type = readParam(num);
                            sprintf(TransmitBuffer,"%s: %d %s\r\n",parameters[num].name,digitalParameter,parameters[num].uints);
                        }
                    }
                    else if(type == ANALOG)
                    {
                        if(floatValue > parameters[num].maxFloat)
                        {
                            convertFloat1 = floatValue * 100;
                            convertFloat2 = parameters[num].maxFloat * 100;
                            sprintf(TransmitBuffer,"%ld.%02ld %s exceeds maximum value %ld.%02ld %s\r\n",convertFloat1/100,convertFloat1%100,parameters[num].uints,convertFloat2/100,convertFloat2%100,parameters[num].uints);
                        }
                        else if(floatValue < parameters[num].minFloat)
                        {
                            convertFloat1 = floatValue * 100;
                            convertFloat2 = parameters[num].minFloat * 100;
                            sprintf(TransmitBuffer,"%ld.%02ld %s is less than value %ld.%02ld %s\r\n",convertFloat1/100,convertFloat1%100,parameters[num].uints,convertFloat2/100,convertFloat2%100,parameters[num].uints);
                        }
                        else
                        {
                            writeParam(num,value,floatValue);
                            type = readParam(num);
                            convertFloat1 = analogParameter * 100;
                            sprintf(TransmitBuffer,"%s: %ld.%02ld %s\r\n",parameters[num].name,convertFloat1/100,convertFloat1%100,parameters[num].uints);
                        }
                    }
                     pc.printf(TransmitBuffer);
                }
                else
                {
                    pc.printf("Invalid parameter\r\nType help for list of valid parameters\r\n");
                }
            }
            else
            {
                pc.printf("Missing parameter(s)\r\nformat: write parameter value\r\n");
            }
            break;
#endif
#if 0       
        case INFO:
            num = sscanf(&line[ptrCmd],"%s",str);       /* expecting one parameter on the command line - parameter name */
            if(num == 1)
            {
                num = getParameter(str);                /* get paramter position in parameters array from paramter name */
                if (num >= 0)
                {
         //           type = readParam(num);
                    displayParameter(num);
                }
                else
                {
                    UART_1_PutString("Invalid parameter\r\nType help for list of valid parameters\r\n");
                }
            }
            else
            {
//                for(i=0;strcmpi(parameters[i].name,"end") != 0;i++)
                for(i=0;strcasecmp(parameters[i].name,"end") != 0;i++)
                    displayParameter(i);
            }
            break;
       
        case FIND:
            num = sscanf(&line[ptrCmd],"%s",str);       /* expecting one parameter on the command line - parameter name */
            if(num == 1)
            {
                num = findParameter(str);                /* get paramter position in parameters array from paramter name */
                if (num < 0)
                {
                    sprintf(TransmitBuffer,"String %s not found in any parameter name\r\n",str);
                    UART_1_PutString(TransmitBuffer);
                }
            }
            else
            {
                UART_1_PutString("\r\nValid Paramters\r\n");
    
                /* print parameter list */
//                for(i=0;strcmpi(parameters[i].name,"end") != 0;i++)
                for(i=0;strcasecmp(parameters[i].name,"end") != 0;i++)
                {
                    sprintf(TransmitBuffer,"%s\r\n",parameters[i].name);
                    UART_1_PutString(TransmitBuffer);   
                }
            }
            break;
            
        case RUN:
                if(runStatus == TRUE)
                {
                    sprintf(TransmitBuffer,"%s Control already Running\r\n",time_string);
                    UART_1_PutString(TransmitBuffer);  
                }
                else
                {
                    runStatus = TRUE;
                    lampTestStatus = LAMPTESTNOTRUN;
                    sprintf(TransmitBuffer,"%s Control Running\r\n",time_string);
                    UART_1_PutString(TransmitBuffer);   
                }
                break;
            
        case STOP:
                if(runStatus == FALSE)
                {
                    sprintf(TransmitBuffer,"%s Control already Stopped\r\n",time_string);
                    UART_1_PutString(TransmitBuffer);   
                }
                else
                {
                    runStatus = FALSE;
                    sprintf(TransmitBuffer,"%s Control Stopped\r\n",time_string);
                    UART_1_PutString(TransmitBuffer);   
                    phaseUV = (PHASE_AB_UV_Read() << 0);
                    phaseUV |= (PHASE_AC_UV_Read() << 1);
                    phaseUV |= (PHASE_BC_UV_Read() << 2);
                    statusRegister = STATUS_REGISTER1_ReadMask();
                    statusRegister |= (PHASEABUVIN | PHASEBCUVIN | PHASEACUVIN);
                    STATUS_REGISTER1_INT_ClearPending();
                    STATUS_REGISTER1_WriteMask(statusRegister);
                    HVState = HVSTATEOFF;
                    eStopSeen = FALSE;
                    outputOCSeen = FALSE;
                    HVStateMsg = 0;                 /* no messages printed yet */
                    flashCtrlSys = 0;
                    tempFailed = FALSE;
                    pendantTestStatus = 0;
                    pendantPresent = FALSE;
                    waitForPowerCycle = FALSE;
                    PEND_PRESENT_NOT_INT_Stop();
                    SOFT_START_RELAY_CTRL_Write(0);
                    PWM_DC_Stop();
                    statusRegister = STATUS_REGISTER1_ReadMask();
                    statusRegister &= ~(BRIDGEOCFLAGIN | OUTPUTOCFLAGIN | UTILOCFLAGIN  );
                    STATUS_REGISTER1_WriteMask(statusRegister);
                    ENCODER_COUNTER_Stop();
                    initializeAllParameters();
writeCurrent0(0.0);                               /* test prurposes only */
                }
                break;
                
        case HVSTATE:
            sprintf(TransmitBuffer,"%s HV State: %02X\r\n",time_string,HVState);
            UART_1_PutString(TransmitBuffer);
            printHVState();
            break;
            #endif
           
        default:
            if(strlen(line) !=0)
            {
                sprintf(TransmitBuffer,"Unknown cmd: %s\r\nType help for list of valid commands\r\n",line);
                pc.printf(TransmitBuffer);
//                linesInBuffer--;
//                indexNextLineIn--;
//                if(linesInBuffer == NUM_LINES)
//                    indexNextLineIn = NUM_LINES -1;
            }
            break;
     }
    for(i=0;i < sizeof(line);i++)
        line[i] = '\0';
}

/** 
 * \fn getCmd
 *
 * \brief parses command line to find position of command in validCmds array.
 *
 * copies line array into cmd array until a space or end of line then
 * searches cmd array for a match in validCmds array.
 * \return int8 - position of command in command array if found. -1 if not found<p>
 *
 */
int8 getCmd()
{
    uint i;
    char cmd[40];
    
    for(i=0;i<strlen(line);i++)
    {
        if(line[i] == ' ')
        {
            break;
        }
        cmd[i] = line[i];
    }
    cmd[i] = '\0';
    ptrCmd = strlen(cmd) + 1;
//    sprintf(TransmitBuffer,"cmd: %d %s\r\n",strlen(cmd),cmd);
//    UART_1_PutString(TransmitBuffer);
    
    for(i=0;i< sizeof(validCmds)/sizeof(validCmds[0]);i++)
    {
//        if(strcmpi(validCmds[i],cmd) == 0)
        if(strcasecmp(validCmds[i],cmd) == 0)
        {
            return i;
        }
    }
    return -1;
}

/** 
 * \fn int8 getParameter(char *param)
 *
 * \brief searches parameters array to find position of command in parameters array.
 *
 * searches the entire parameters array for a name match with param.
 *
 * \param[in] param - pointer to an array containing the parameter name.
 *
 * \return int8 - position of command in command array if found. -1 if not found<p>
 *
 */
int8 getParameter(char *param)
{
    uint i;
    
 //   sprintf(TransmitBuffer,"param: %d %s\r\n",strlen(param),param);
 //   pc.printf(TransmitBuffer);
    
    for(i=0;strcasecmp(parameters[i].name,"end") != 0;i++)
    {
        if(strcasecmp(parameters[i].name,param) == 0)
        {
            return i;
        }
    }
    return -1;
}

/** 
 * \fn int8 findParameter(char *string)
 *
 * \brief finds all parameters with string in the name.
 *
 * searches the entire parameters array for all names 
 * that contain string.
 *
 * \param[in] string - pointer to a string.
 *
 * \return int8 - 0 string found in some names. -1 if string not found in any name<p>
 *
 */
int8 findParameter(char *string)
{
    uint i,j;
    int8 result = -1;
    char name[40];
    char string1[40];
    
//    sprintf(TransmitBuffer,"param: %d %s\r\n",strlen(param),param);
//    UART_1_PutString(TransmitBuffer);
   
    strcpy(string1,string);
    for(i=0;i < strlen(string1);i++)
        string1[i] = tolower((unsigned char)string1[i]);
        
//    for(i=0;strcmpi(parameters[i].name,"end") != 0;i++)
    for(i=0;strcasecmp(parameters[i].name,"end") != 0;i++)
    {
        strcpy(name,parameters[i].name);        /* copy parameter name and convert to lower case */
        for(j=0;j < strlen(name);j++)
            name[j] = tolower((unsigned char)name[j]);
        if(strstr(name,string1) != NULL)
        {
            sprintf(TransmitBuffer,"%s\r\n",parameters[i].name);
            pc.printf(TransmitBuffer);
            result = 0;
        }
    }
    return result;
}

/** 
 * \fn uint8 readParam(uint8 paramNum)
 *
 * \brief reads in parameter at position paramNum.
 *
 * This will only be called if there was a valid parameter match found.
 * will read correct parameter based on parameter type. This will call the
 * correct read function specified in the paramter array. A parameter will be
 * read into one of binaryParameter, digitalParameter, or analogParameter based on type.
 *
 * \param[in] paramNum - pointer to a character string containing  parameter name
 * \return uint8 - type of parameter<p>
 *
 */
uint8 readParam(uint8 paramNum)
{
    uint8 type =  parameters[paramNum].type;
    
    if(type == BINARY)
    {
        if(parameters[paramNum].ptrRead == NULL)
        {
            sprintf(TransmitBuffer,"%s has a NULL ptrRead parameter\r\n",parameters[paramNum].name);
            pc.printf(TransmitBuffer);
            binaryParameter = 0;
        }
        else
            binaryParameter = parameters[paramNum].ptrRead();
        return type;
    }
    if(type == ANALOG)
    {
        if(parameters[paramNum].ptrRead == NULL)
        {
            sprintf(TransmitBuffer,"%s has a NULL ptrRead parameter\r\n",parameters[paramNum].name);
            pc.printf(TransmitBuffer);
            analogParameter = 0.0;
        }
        else
            parameters[paramNum].ptrRead();
        return type;
    }
    if(type == DIGITAL)
    {
        if(parameters[paramNum].ptrRead == NULL)
        {
            sprintf(TransmitBuffer,"%s has a NULL ptrRead parameter\r\n",parameters[paramNum].name);
            pc.printf(TransmitBuffer);
            digitalParameter = 0;
        }
        else
            parameters[paramNum].ptrRead();
        return type;
    }
    if(type == REGISTER)
    {
        bytesRead = 0;
        if(parameters[paramNum].ptrReadRegLength == NULL)
        {
            sprintf(TransmitBuffer,"%s has a NULL ptrReadRegLength parameter\r\n",parameters[paramNum].name);
            pc.printf(TransmitBuffer);
            digitalParameter = 0;
        }
        else
            bytesRead = parameters[paramNum].ptrReadRegLength(regAddress, regData, regLength);
        return type;
    }
    return type;
}

/** 
 * \fn void writeParam(uint8 paramNum,uint16 value, float floatValue)
 *
 * \brief reads in parameter at position paramNum.
 *
 * This will only be called if there was a valid parameter match found.
 * will write value to the correct parameter based on parameter type. This will call the
 * correct write function specified in the paramter array. value will not be written
 * to read only paramters.
 *
 * \param[in] paramNum - pointer to a character string containing the parameter name
 * \param[in] value - 16 bit value to write to a digital parameter.
 * \param[in] floatValue - analog value to write to an anlog parameter.
 * \return uint8 - type of parameter<p>
 *
 */
void writeParam(uint8 paramNum,uint8 *value)
{   
    uint8 type =  parameters[paramNum].type;
     if(parameters[paramNum].mode == READONLY)
    {
            sprintf(TransmitBuffer,"%s is a read only parameter\r\n",parameters[paramNum].name);
            pc.printf(TransmitBuffer);
            return;
    }
   #if 0    
    if(type == BINARY)
    {
        if(parameters[paramNum].ptrWrite8 == NULL)
        {
            sprintf(TransmitBuffer,"%s has a NULL pteWrite8 parameter\r\n",parameters[paramNum].name);
            pc.printf(TransmitBuffer);
        }
        else
            parameters[paramNum].ptrWrite8(value);
    }
    else if(type == DIGITAL)
    {
        if(parameters[paramNum].ptrWrite16 == NULL)
        {
            sprintf(TransmitBuffer,"%s has a NULL pteWrite16 parameter\r\n",parameters[paramNum].name);
            pc.printf(TransmitBuffer);
        }
        else
            parameters[paramNum].ptrWrite16(value);
    }
    #endif
    else if(type == REGISTER)
    {
        if(parameters[paramNum].ptrWriteReg8 == NULL)
        {
            sprintf(TransmitBuffer,"%s has a NULL ptrWriteReg8 parameter\r\n",parameters[paramNum].name);
            pc.printf(TransmitBuffer);
        }
        else
            parameters[paramNum].ptrWriteReg8(regAddress, value);
    }
    #if 0
    else if(type == ANALOG)
    {
        if(parameters[paramNum].ptrWriteAnalog == NULL)
        {
            sprintf(TransmitBuffer,"%s has a NULL pteWriteAnalog parameter\r\n",parameters[paramNum].name);
            pc.printf(TransmitBuffer);
        }
        else
            parameters[paramNum].ptrWriteAnalog(floatValue);
    }
    #endif
        return;
}

/** 
 * \fn void displayParameter(uint8 num)
 *
 * \brief displays information about a parameter
 *
 * This routine all of the information in the parameters array
 * for the selected parameter.
 *
 * \param[in] num - 8 bit integer location in the parameters array 
  * \return uint16 - 16 bit num to the power pow<p>
 *
 */
void displayParameter(uint8 num)
{
    char *strType;
    char *strMode;
    uint16 convertFloat1,convertFloat2,convertFloat3;
    
    switch (parameters[num].type)
    {
        case BINARY:
            strType = "Binary";
            break;
          
        case DIGITAL:
            strType = "16 Bit";
            break;
        
        case ANALOG:
            strType = "Analog";
            break;
        
        default:
            strType = "Unknown";
            break;
    }
    
    switch (parameters[num].mode)
    {
        case READONLY:
            strMode = "Read Only";
            break;
        
        case READWRITE:
            strMode = "Read/Write";
            break;
        
        case WRITEONLY:
            strMode = "Write Only";
            break;
        
        default:
            strMode = "Unknown";
            break;
    }
    
    sprintf(TransmitBuffer,"Name:\t'%s' Port: '%s' Connector: '%s' Units: '%s'\r\n",parameters[num].name,parameters[num].port,parameters[num].connector,parameters[num].uints);
    pc.printf(TransmitBuffer);   
    sprintf(TransmitBuffer,"\tDescription: '%s'\r\n",parameters[num].description);
    pc.printf(TransmitBuffer); 
    
    if(parameters[num].type != ANALOG)
    {
        sprintf(TransmitBuffer,"\tType: '%s' mode: '%s' Initial: %d Min: %d Max: %d\r\n",strType,strMode,parameters[num].initialInt,parameters[num].minInt,parameters[num].maxInt);
        pc.printf(TransmitBuffer);   
    }
    else
    {
        convertFloat1 = parameters[num].initialFloat * 100;
        convertFloat2 = parameters[num].minFloat * 100;
        convertFloat3 = parameters[num].maxFloat * 100;
        sprintf(TransmitBuffer,"\tType: '%s' mode: '%s' Initial: %d.%02d Min: %d.%02d Max: %d.%02d\r\n",strType,strMode,convertFloat1/100,convertFloat1%100,convertFloat2/100,convertFloat2%100,convertFloat3/100,convertFloat3%100);
        pc.printf(TransmitBuffer);   
    }
}

uint16 readSPIReg(uint16 regAddress, uint8 *regData, uint16 length)
{
    
//    sprintf(TransmitBuffer,"read SPI %02X %04X\r\n",regAddress, length);
//    pc.printf(TransmitBuffer); 
    if(length > MAX_REGISTER_DATA)
    {
        sprintf(TransmitBuffer,"Length 0x%X > 0x%X",length,MAX_REGISTER_DATA);
        pc.printf(TransmitBuffer);   
      return 0;
    }  
    SPIRead(regAddress,regData,length);
    return(length);
}

void writeSPIReg(uint16 regAddress, uint8 *regData)
{
    sprintf(TransmitBuffer,"write SPI %02X %02X\r\n",regAddress, regData[0]);
    pc.printf(TransmitBuffer); 
    SPIWrite(regAddress, regData,1);
    return;
}

uint16 readI2CReg(uint16 regAddress, uint8 *regData, uint16 length)
{
    
//    sprintf(TransmitBuffer,"read I2C %02X %04X\r\n",regAddress, length);
//    pc.printf(TransmitBuffer); 
    if(length > MAX_REGISTER_DATA)
    {
        sprintf(TransmitBuffer,"Length 0x%X > 0x%X",length,MAX_REGISTER_DATA);
        pc.printf(TransmitBuffer);   
      return 0;
    }  
    I2CRead(regAddress,regData,length);
    return(length);
}


void writeI2CReg(uint16 regAddress, uint8 *regData)
{
    sprintf(TransmitBuffer,"write I2C %02X %02X\r\n",regAddress, regData[0]);
    pc.printf(TransmitBuffer); 
    I2CWrite(regAddress, regData,1);
    return;
}

/* [] END OF FILE */
