/*
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C) 2014 Semtech

Description: -

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainers: Miguel Luis, Gregory Cristian and Nicolas Huguenin
*/
#include "sx1272-hal.h"
#include "debug.h"

const RadioRegisters_t SX1272MB1xAS::RadioRegsInit[] = RADIO_INIT_REGISTERS_VALUE;

SX1272MB1xAS::SX1272MB1xAS( RadioEvents_t *events,
                            PinName mosi, PinName miso, PinName sclk, PinName nss, PinName reset,
                            PinName dio0, PinName dio1, PinName dio2, PinName dio3, PinName dio4, PinName dio5,
                            PinName antSwitch )
                            : SX1272( events, mosi, miso, sclk, nss, reset, dio0, dio1, dio2, dio3, dio4, dio5 ),
                            antSwitch( antSwitch ),
                        #if( defined ( TARGET_NUCLEO_L152RE ) )
                            fake( D8 ) 
                        #else
                            fake( A3 )
                        #endif
{
    debug("SX1272MB1xAS constructor\r\n");
    this->RadioEvents = events;

    Reset( );
    
    RxChainCalibration( );
    
    IoInit( );
    
    SetOpMode( RF_OPMODE_SLEEP );
    
    IoIrqInit( dioIrq );
    
    RadioRegistersInit( );

    SetModem( MODEM_FSK );


    this->settings.State = RF_IDLE ;
}

SX1272MB1xAS::SX1272MB1xAS( RadioEvents_t *events ) 
                        #if defined ( TARGET_NUCLEO_L152RE )
                        :   SX1272( events, D11, D12, D13, D10, A0, D2, D3, D4, D5, A3, D9 ), // For NUCLEO L152RE dio4 is on port A3
                            antSwitch( A4 ),
                            fake( D8 )
                        #elif defined( TARGET_LPC11U6X )
                        :   SX1272( events, D11, D12, D13, D10, A0, D2, D3, D4, D5, D8, D9 ),
                            antSwitch( P0_23 ), 
                            fake( A3 )
                        #else
                        :   SX1272( events, PTD6, PTD7, PTD5, PTD4, PTA2, PTC2, PTB1, PTC3, D5, PTC4, D9 ),
                            antSwitch( PTC6 ), //changed from A4 to PTC6 by jlc
                            fake( PTA2 )
                        #endif
{
   debug("SX1272MB1xAS constructor1\r\n");
     this->RadioEvents = events;

    Reset( );
    
    boardConnected = UNKNOWN;
    
    DetectBoardType( );
    
    RxChainCalibration( );
    
    IoInit( );
    
    SetOpMode( RF_OPMODE_SLEEP );
    IoIrqInit( dioIrq );
    
    RadioRegistersInit( );


//    SetModem( MODEM_LORA );

    this->settings.State = RF_IDLE ;
}

//-------------------------------------------------------------------------
//                      Board relative functions
//-------------------------------------------------------------------------
uint8_t SX1272MB1xAS::DetectBoardType( void )
{
    if( boardConnected == UNKNOWN )
    {
        antSwitch.input( );
        wait_ms( 1 );
        if( antSwitch == 1 )
        {
            boardConnected = SX1272RF1;
        }
        else
        {
            boardConnected = SX1272RF1;
        }
        antSwitch.output( );
        wait_ms( 1 );
    }
    debug("DetectBoardType exit %d\r\n",boardConnected);
    return ( boardConnected );
}

void SX1272MB1xAS::IoInit( void )
{
    debug("IoInit enter\r\n");
    AntSwInit( );
    SpiInit( );
}

void SX1272MB1xAS::RadioRegistersInit( )
{
   uint8_t i = 0;
    for( i = 0; i < sizeof( RadioRegsInit ) / sizeof( RadioRegisters_t ); i++ )
    {
     debug("RadioRegistersInit %d %02X %02X\r\n",RadioRegsInit[i].Modem ,RadioRegsInit[i].Addr, RadioRegsInit[i].Value  );
        SetModem( RadioRegsInit[i].Modem );
        Write( RadioRegsInit[i].Addr, RadioRegsInit[i].Value );
    }    
}

void SX1272MB1xAS::SpiInit( void )
{
      debug("SpiInit enter\r\n");
    nss = 1;    
    spi.format( 8,0 );   
    uint32_t frequencyToSet = 1000000;
 //  uint32_t frequencyToSet = 8000000;
//    #if( defined ( TARGET_NUCLEO_L152RE ) ||  defined ( TARGET_LPC11U6X ) )
        spi.frequency( frequencyToSet );
//    #elif( defined ( TARGET_KL25Z ) ) //busclock frequency is halved -> double the spi frequency to compensate
//        spi.frequency( frequencyToSet * 2 );
//    #else
//        #warning "Check the board's SPI frequency"
//    #endif
    wait(0.1); 
}

void SX1272MB1xAS::IoIrqInit( DioIrqHandler *irqHandlers )
{
      debug("IoIrqInit enter\r\n");
#if( defined ( TARGET_NUCLEO_L152RE ) ||  defined ( TARGET_LPC11U6X ) )
    dio0.mode(PullDown);
    dio1.mode(PullDown);   
    dio2.mode(PullDown);
    dio3.mode(PullDown); 
    dio4.mode(PullDown); 
#endif
    dio0.rise( this, static_cast< TriggerMB1xAS > ( irqHandlers[0] ) );
    dio1.rise( this, static_cast< TriggerMB1xAS > ( irqHandlers[1] ) );
    dio2.rise( this, static_cast< TriggerMB1xAS > ( irqHandlers[2] ) );
    dio3.rise( this, static_cast< TriggerMB1xAS > ( irqHandlers[3] ) );
    dio4.rise( this, static_cast< TriggerMB1xAS > ( irqHandlers[4] ) );
}

void SX1272MB1xAS::IoDeInit( void )
{
       debug("IoDeInit\r\n");
   //nothing
}

uint8_t SX1272MB1xAS::GetPaSelect( uint32_t channel )
{
      debug("GetPaSelect\r\n");
    if( channel > RF_MID_BAND_THRESH )
    {
        if( boardConnected == SX1276MB1LAS )
        {
            return RF_PACONFIG_PASELECT_PABOOST;
        }
        else
        {
            return RF_PACONFIG_PASELECT_RFO;
        }
    }
    else
    {
        return RF_PACONFIG_PASELECT_RFO;
    }
}

void SX1272MB1xAS::SetAntSwLowPower( bool status )
{
      debug("SetAntSwLowPower enter %d\r\n",status);
    if( isRadioActive != status )
    {
        isRadioActive = status;
    
        if( status == false )
        {
            AntSwInit( );
        }
        else
        {
            AntSwDeInit( );
        }
    }
}

void SX1272MB1xAS::AntSwInit( void )
{
       debug("AntSwInit enter\r\n");
   antSwitch = 0;
}

void SX1272MB1xAS::AntSwDeInit( void )
{
       debug("AntSwDeInit enter\r\n");
   antSwitch = 0;
}

void SX1272MB1xAS::SetAntSw( uint8_t rxTx )
{
    debug("SetAntSw enter %d\r\n",rxTx);
   if( this->rxTx == rxTx )
    {
        //no need to go further
        return;
    }

    debug("SetAntSw set %d\r\n",rxTx);
    this->rxTx = rxTx;

    if( rxTx != 0 )
    {
        antSwitch = 1;
    }
    else
    {
        antSwitch = 0;
    }
}

bool SX1272MB1xAS::CheckRfFrequency( uint32_t frequency )
{
      debug("CheckRfFrequency\r\n");
    //TODO: Implement check, currently all frequencies are supported
    return true;
}


void SX1272MB1xAS::Reset( void )
{
    debug("Reset enter\r\n");
    reset.output();
    reset = 0;
    wait_ms( 1 );
    reset.input();
    wait_ms( 6 );
}
    
void SX1272MB1xAS::Write( uint8_t addr, uint8_t data )
{
       debug("Write %02X %02X\r\n",addr, data);
    Write( addr, &data, 1 );
}

uint8_t SX1272MB1xAS::Read( uint8_t addr )
{
     uint8_t data;
    Read( addr, &data, 1 );
      debug("Read %02X %02X\r\n",addr ,data);
    return data;
}

void SX1272MB1xAS::Write( uint8_t addr, uint8_t *buffer, uint8_t size )
{
    uint8_t i;

    nss = 0;
    spi.write( addr | 0x80 );
    for( i = 0; i < size; i++ )
    {
        spi.write( buffer[i] );
    }
    nss = 1;
}

void SX1272MB1xAS::Read( uint8_t addr, uint8_t *buffer, uint8_t size )
{
    uint8_t i;

    nss = 0;
    spi.write( addr & 0x7F );
    for( i = 0; i < size; i++ )
    {
        buffer[i] = spi.write( 0 );
    }
    nss = 1;
}

void SX1272MB1xAS::WriteFifo( uint8_t *buffer, uint8_t size )
{
      debug("WriteFifo\r\n");
    Write( 0, buffer, size );
}

void SX1272MB1xAS::ReadFifo( uint8_t *buffer, uint8_t size )
{
      debug("ReadFifo\r\n");
    Read( 0, buffer, size );
}
