// Copyright 2017 Baidu Inc. All Rights Reserved.
// Author: Su Hao (suhao@baidu.com)
//
// Description: Scheduler

#ifndef BAIDU_IOT_TINYDU_IOT_OS_SRC_IOT_BAIDU_CA_SCHEDULER_BAIDU_CA_SCHEDULER_H
#define BAIDU_IOT_TINYDU_IOT_OS_SRC_IOT_BAIDU_CA_SCHEDULER_BAIDU_CA_SCHEDULER_H

#include "rtos.h"
#include "baidu_ca.h"
#include "baidu_ca_handler.h"
#include "baidu_ca_network_socket.h"
#include "baidu_util_list.h"
#include "baidu_ca_object.h"

namespace duer {

class Scheduler : public Handler::OnMessageListener {
public:

    /**
     * @brief Scheduler 的事件回调接口
     */
    class IOnEvent {
    public:
        /**
         * @brief Scheduler 在启动时的回调函数接口
         */
        virtual int on_start() = 0;
        /**
         * @brief Scheduler 在停止时的回调函数接口
         */
        virtual int on_stop() = 0;
        /**
         * @brief Scheduler 在有数据通信时的回调函数接口
         */
        virtual int on_action(const char* action) = 0;

    };
    /**
     * @brief 获取 Scheduler 的唯一单例引用
     */
    static Scheduler& instance();
    /**
     * @brief 注册 Scheduler 的事件回调接口
     * @param[in] listener 事件回调接口
     * @return 0
     */
    int set_on_event_listener(IOnEvent* listener);
    /**
     * @brief 添加控制点
     * @param[in] list_res 待注册的控制点数组
     * @param[in] list_res_size 控制点数组长度
     * @return 0
     */
    int add_controll_points(const bca_res_t list_res[], bca_size_t list_res_size);
    /**
     * @brief 启动 Scheduler
     * @return 0
     */
    int start();
    /**
     * @brief 停止 Scheduler
     * @return 0
     */
    int stop();
    /**
     * @brief 上报数据 json格式
     * @param[in] data 具体的数据内容
     * @return 0
     */
    int report(const Object& data);
    /**
     * @brief 上报数据 json格式
     * @param[in] data 数据内容
     * @param[in] size 数据长度
     * @return 0
     */
    int report(const void* data, size_t size);
    /**
     * @brief 回复对端的请求
     * @param[in] req 请求
     * @param[in] msg_code 回复的code
     * @param[in] payload 回复的内容
     * @return 0
     */
    int response(const bca_msg_t* req, int msg_code, const char* payload);
    /**
     * @brief 上报语音数据
     * @param[in] data 语音数据
     * @param[in] size 语音数据长度
     * @param[in] eof  是否最后一个数据包
     * @return 0
     */
    int send_content(const void* data, size_t size, bool eof);
    /**
     * @brief 清除待处理传输数据
     * @return 0
     */
    int clear_content();
    /**
     * @brief handler回调接口
     * @param[in] message 需要处理的消息
     * @return true
     */
    virtual bool on_message(Message& message);

private:

    Scheduler();

    virtual ~Scheduler();

    void data_available(const bca_addr_t* addr);

    int start_handle();

    int stop_handle();

    int report_handle(Message& message);

    int data_available_handle(Message& message);

    void keep_alive_handle();

    void timer_callback();

    int report_list_clear();

    int on_start(int status);

    int on_stop();

    int on_action(const char* action);

    static bca_status_t play_uri(bca_context ctx, bca_msg_t* msg, bca_addr_t* addr);

    static bca_status_t update_package(bca_context ctx,
                                       bca_msg_t* msg,
                                       bca_addr_t* addr);

    static bca_status_t output_result(bca_context ctx,
                                      bca_msg_t* msg,
                                      bca_addr_t* addr);

    IOnEvent*           _on_event_listener;

    Handler             _event_handler;

    bca_handler         _handler;

    int                 _state;

    rtos::Mutex         _mutex;

    rtos::RtosTimer     _timer;

    unsigned int        _start_timeout;

    duer::SocketAdapter::Callback  _on_read_callback;

    bu_list_t           _send_list;

    mbed::Timer         _stop_timer;

    unsigned int        _stop_timeout;

    int                 _topic_id;
    bool                _enable_ota;

    static Scheduler    _s_scheduler;
};

} // namespace duer

#endif // BAIDU_IOT_TINYDU_IOT_OS_SRC_IOT_BAIDU_CA_SCHEDULER_BAIDU_CA_SCHEDULER_H
