/*The following program is intended for reading the MAX31723 Temperature Sensor
via SPI connections. The program was evaluated using a MAX32600MBED board, with
temperature values being displayed via USB connection to a serial monitor. Both Celsius
and Fahrenheit are supported and displayed. 

By default, the sensor is placed into Sleep-Mode, and temperature samples are only
collected after a One-Shot Configuration is sent to the Configuration Register.
There is a pause for the maximum sampling time before the new temperature values are
read. The Configuration register is set to the highest precision mode, requiring 4 bits
stored in the LSB Register. */

#include "mbed.h"              

Serial pc(USBTX, USBRX);
SPI spi(P2_1,P2_2,P2_0);        //Mosi, Miso, sck
DigitalOut cs(P2_5);        //Define enable pin

/*Define Constant address variables for reading and writing to
temperature sensor registers. All values can be found on MAX31723
data sheet and are labled below */

const uint8_t Garbage = 0xff;
const uint8_t Configuration_Write_Address = 0x80;
const uint8_t Configuration_Read_Address = 0x00;
const uint8_t Temperature_Read_Address_LSB = 0x01;
const uint8_t Temperature_Read_Address_MSB = 0x02;
const uint8_t Configuration_Sleep = 0x80;
const uint8_t Configuration_OneShot = 0x10;

int data, data2, config;
char bdata[7];

int main(){
    
    pc.printf("=================  STARTING  =================\r\n");    
    spi.format(8,3);        //Read 8 bit burst, mode 3
    spi.frequency(10000);       //Set frequency to 10000Hz 
    cs=0;       //Set enable to 0
    
    wait_ms(500);
    
    while(1){

        /*Send Oneshot signal with highest percision bit setting */

        cs=1;
        spi.write(Configuration_Write_Address);
        spi.write(0x17);      
        cs=0;
        
        wait_ms(200);           //Wait maximum time for temperature capture
        
        /*Read temperature register holding LSB*/
        
        cs=1;
        spi.write(Temperature_Read_Address_LSB);
        data = spi.write(0xff);
        cs=0;

        /*Converts the decimal number held in "data" into a binary array*/
        
        int count = 7;
        for (int j = 7;j>=0;j--){
                int r = data>>j;
                if(r&1)
                     bdata[count] = 1;
                else
                     bdata[count] = 0;
                count --;
            }
        
        /*Read temperature register holding MSB*/
        
        cs=1;
        spi.write(Temperature_Read_Address_MSB);
        data2 = spi.write(Garbage);
        cs=0;
        
        
        /*Function for calculating temperature based on bit values stored
        in temperature registers, All equations and values found on MAX31723
        data sheet under "Measuring Temperature" section header.  */
        
        double temp = data2;        //Temperature MSB is already converted to Decimal
        int index = 7;
        for (float z=-1;z>=-4;z--){         //Calculate temperature from array holding binary value
            float factor = pow (2,z);       //Calculate Added Temperature for each bit (Per data sheet)
            temp += bdata[index] * factor;      //Combine all bit temperature to aquire total sum
            index--;
        }
        
        /*Calculate temperature in F using equation F(deg) = (5/9) * C(deg) + 32 */
        
        float fahrenheit = (((temp*9)/5) + 32);     
        
        /*Print out Results*/
        
        pc.printf("Temperature is %f degrees Celcius\r\n", temp);
        pc.printf("Temperature is %f degrees Fahrenheit\r\n", fahrenheit);
        pc.printf("Reading Complete\r\n");
        pc.printf("\r\n\n\n\n\n");
        wait(2);
    }
}