#include "mbed.h"
#include "nRF51822n.h"

#define DEVICE_NAME "Smart Button"

#define NEED_CONSOLE_OUTPUT 1 /* Set this if you need debug messages on the console;
                               * it will have an impact on code-size and power consumption. */
 
#if NEED_CONSOLE_OUTPUT
Serial  pc(USBTX, USBRX);
#define DEBUG(...) { pc.printf(__VA_ARGS__); }
#else
#define DEBUG(...) /* nothing */
#endif /* #if NEED_CONSOLE_OUTPUT */

nRF51822n   nrf;                               /* BLE radio driver */
DigitalIn sw1(p2);
DigitalIn sw2(p3);
DigitalIn sw3(p4);
DigitalIn sw4(p5);

enum {
        UUID_SWITCH_STATUS_SERVICE = 0xFFE0,
        UUID_SWITCH_STATUS_CHAR = 0xFFE1
};

/* LEDs for indication: */
DigitalOut  oneSecondLed(LED1);        /* LED1 is toggled every second. */
DigitalOut  advertisingStateLed(LED2); /* LED2 is on when we are advertising, otherwise off. */

/* Switch Status Service */
uint8_t             btns_in = 0;
GattService         switchsService(UUID_SWITCH_STATUS_SERVICE);
GattCharacteristic  switchsStatus( UUID_SWITCH_STATUS_CHAR, 1, 1,
                                 GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY |
                                 GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);

/* Health Thermometer Service */ 
/*
uint8_t             thermTempPayload[5] = { 0, 0, 0, 0, 0 };
GattService         thermService (GattService::UUID_HEALTH_THERMOMETER_SERVICE);
GattCharacteristic  thermTemp (GattCharacteristic::UUID_TEMPERATURE_MEASUREMENT_CHAR,
                               5, 5, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_INDICATE);
*/

/* Battery Level Service */
uint8_t            batt = 100;     /* Battery level */
uint8_t            read_batt = 0; /* Variable to hold battery level reads */
GattService        battService ( GattService::UUID_BATTERY_SERVICE );
GattCharacteristic battLevel   ( GattCharacteristic::UUID_BATTERY_LEVEL_CHAR, 1, 1,
                                 GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY |
                                 GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);


/* Advertising data and parameters */
GapAdvertisingData   advData;
GapAdvertisingData   scanResponse;
GapAdvertisingParams advParams ( GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED );

uint16_t             uuid16_list[] = {UUID_SWITCH_STATUS_SERVICE,
                                      GattService::UUID_BATTERY_SERVICE};

void updateServiceValues(uint8_t n);

/**************************************************************************/
/*!
    @brief  This custom class can be used to override any GapEvents
            that you are interested in handling on an application level.
*/
/**************************************************************************/
class GapEventHandler : public GapEvents
{
    //virtual void onTimeout(void) {}   
     
    virtual void onConnected(void)
    {
        advertisingStateLed = 0;
    }

    /* When a client device disconnects we need to start advertising again. */
    virtual void onDisconnected(void)
    {
        nrf.getGap().startAdvertising(advParams);
        advertisingStateLed = 1;
    }
};

/**************************************************************************/
/*!
    @brief  Program entry point
*/
/**************************************************************************/
int main(void)
{
    DEBUG("main process >>>\n\r");
    
    /* Setup blinky led */
    oneSecondLed=0;
    
    /* Setup an event handler for GAP events i.e. Client/Server connection events. */
    nrf.getGap().setEventHandler(new GapEventHandler());
    
    /* Initialise the nRF51822 */
    nrf.init();

    /* Make sure we get a clean start */
    nrf.reset();    

    /* Add BLE-Only flag and complete service list to the advertising data */
    advData.addFlags(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    advData.addData(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, 
                   (uint8_t*)uuid16_list, sizeof(uuid16_list));
    
    advData.addData(GapAdvertisingData::COMPLETE_LOCAL_NAME,
        (uint8_t*)DEVICE_NAME, 
        strlen(DEVICE_NAME)
        );
    
    advData.addAppearance(GapAdvertisingData::GENERIC_THERMOMETER);
    nrf.getGap().setAdvertisingData(advData, scanResponse);

    /* Switchs Status Service */
    switchsService.addCharacteristic(switchsStatus);
    nrf.getGattServer().addService(switchsService);

    /* Health Thermometer Service */
//    thermService.addCharacteristic(thermTemp);
//    nrf.getGattServer().addService(thermService);
    
    /* Add the Battery Level service */
    battService.addCharacteristic(battLevel);
    nrf.getGattServer().addService(battService);

    /* Start advertising (make sure you've added all your data first) */
    nrf.getGap().startAdvertising(advParams);
//    advertisingStateLed = 1;

    uint8_t n_current = 0;
    int n_sw1,n_sw2,n_sw3,n_sw4;

    while(1)
    {
        n_sw1 = sw1.read();
        n_sw2 = sw2.read();
        n_sw3 = sw3.read();
        n_sw4 = sw4.read();
        
        oneSecondLed = (n_sw1 < 1) ? 1 : 0; 
        
        uint8_t n = 0;
        if(n_sw1 < 1) n += 0x01;
        if(n_sw2 < 1) n += 0x02;
        if(n_sw3 < 1) n += 0x04;
        if(n_sw4 < 1) n += 0x08;
        
        /* Now that we're live, update the battery level & temperature characteristics */
        //updateServiceValues(oneSecondLed.read());
        //DEBUG("adv sw1 %d >>>\n\r",n_sw1);
        if(n_current != n){       
            DEBUG("adv %d %d >>>\n\r",n,n_current);
            updateServiceValues(n);
            n_current = n;            
        }
    }
}

/**************************************************************************/
/*!
    @brief  Ticker callback to switch advertisingStateLed state
*/
/**************************************************************************/
void updateServiceValues(uint8_t n)
{
      
      /* Update switches status */      
      btns_in = n;
      nrf.getGattServer().updateValue(switchsStatus.handle, (uint8_t*)&btns_in, sizeof(btns_in) );
      
      /* Update battery level */
//      nrf.getGattServer().updateValue(battLevel.handle, (uint8_t*)&batt, sizeof(batt));
      /* Decrement the battery level. */
//      batt <=50 ? batt=100 : batt--;;
      
      /* Update the temperature. Note that we need to convert to an ieee11073 format float. */
      //float temperature = healthThemometer.read();
      /*
      float temperature = 2.1*n;
      uint32_t temp_ieee11073 = quick_ieee11073_from_float(temperature);
      memcpy(thermTempPayload+1, &temp_ieee11073, 4);
      nrf.getGattServer().updateValue(thermTemp.handle, thermTempPayload, sizeof(thermTempPayload));
      */
}
