/*
 * mbed library to draw graphs to serial console
 * Copyright (c) 2013 Tobias Knapp
 * Released under the MIT License: http://mbed.org/license/mit
 */

#include "ASCIIGraph.h"

ASCIIGraph::ASCIIGraph(Terminal * terminal, int startX, int startY, int width, int height, int labelXoffset){
    term    = terminal;
    START_X = startX >= 0 ? startX : 0;
    START_Y = startY >= 0 ? startY : 0;
    HEIGHT  = height > 2 ? height : 8;
    WIDTH   = width > 2 ? width : 20;
    LABEL_X_OFFSET = labelXoffset >= 7 ? labelXoffset : 7;
    
    //Set default drawing chars
    xAxisChar  = '-';
    zeroChar   = '+';
    yAxisChar  = '|';
    filledChar = '#';
    plainChar  = ' ';
    
    //Init data and fill with 0
    pData = new float[WIDTH];
    for(int i = 0; i < WIDTH; ++i){
        pData[i] = 0.0;
    }
    
    //Init line
    pLine = new char[WIDTH+1];
    pLine[WIDTH] = '\0'; // end-char
    
    //Set first true
    first = true;
    
    //Clear screen
    //term->cls();
    
    //Set coord sys bases
    BASE_X = START_X + LABEL_X_OFFSET + 1;
    BASE_Y = START_Y + HEIGHT - 1;
    
}

void ASCIIGraph::initGraph(){
    
    //Clear Screen
    term->cls();
    
    //Draw y-axis
    for(int i = 0; i < HEIGHT; ++i){
        term->locate(START_X + LABEL_X_OFFSET ,START_Y+i);
        term->putc(yAxisChar); //166  //179   //186
    }
    
    //Draw (0|0)
    term->locate(START_X + LABEL_X_OFFSET, START_Y + HEIGHT);
    term->putc(zeroChar); //43 //192   //200
    
    //Draw x-axis
    for(int i = 0; i < WIDTH; ++i){
        term->locate(START_X+LABEL_X_OFFSET+1+i, START_Y + HEIGHT);
        term->putc(xAxisChar); //45  //196   //205
    }
    
    //Gimmic: Print Hello if large enough
    if(WIDTH > 48 && HEIGHT > 6){
        int x = (WIDTH - 47)/2;
        int y = (HEIGHT - 7)/2;
        
        int boxX = START_X + LABEL_X_OFFSET + 1 + x;
        int boxY = START_Y + 1 + y;
        term->locate(boxX, boxY++);
        term->printf(" ##   ###   ### ### ###  ###               #    ");
        term->locate(boxX, boxY++);
        term->printf("#  # #     #     #   #  #                  #    ");
        term->locate(boxX, boxY++);
        term->printf("####  ###  #     #   #  #  ## ###  ## ###  ###  ");
        term->locate(boxX, boxY++);
        term->printf("#  #     # #     #   #  #   # #   # # #  # #  #");
        term->locate(boxX, boxY++);
        term->printf("#  # ####   ### ### ###  ###  #   ### ###  #  # ");
        term->locate(boxX, boxY++);
        term->printf("                                      #         ");
        term->locate(boxX, boxY++);
        term->printf("                                      #         ");
        wait(3);
    }
    
}

ASCIIGraph::~ASCIIGraph(){
    delete[] pData;
    delete[] pLine;
}

void ASCIIGraph::pushDataPoint(float dataPoint){
        
    if(!first){
        //Shift data to left
        for(int i = 1; i < WIDTH; ++i)
            pData[i-1] = pData[i];
        
        //Set new dataPoint at the end
        pData[WIDTH-1] = dataPoint;
        
    } else {
        //Fill entire array with first dataPoint if array is empty
        for(int i = 0; i < WIDTH; ++i){
            pData[i] = dataPoint;
        }
        first = false;
    }
}

//Draws Graph 
void ASCIIGraph::drawGraph(){
    
    float min;
    float max;
    float diff;
    float valOfChar;
    
    //Find min and max
    min = pData[0];
    max = pData[0];
    for(int i = 1; i < WIDTH; ++i){
        if(pData[i] < min)
            min = pData[i];
        if(pData[i] > max)
            max = pData[i];
    }
    
    //Calculate diff
    diff = max - min;
    
    /* Calculate y-value of one cell in terminal
     * e.g. diff = 5, HEIGHT in terminal is 10
     *      => valOfChar = 5 / 10 = 0.5
     *      => The y-value of each vertical cell is 0.5
     *      => Two set (filled) cells represent the value 1.0
     */
    valOfChar = (diff / (float)HEIGHT);
    term->locate(BASE_X - 1, BASE_Y + 3);
    term->printf("Min: %f   Max: %f   Diff: %f  Val: %f", min, max, diff, valOfChar);
    
    //Update scale
    term->locate(START_X, START_Y);
    term->printf("%4.1f", max);

    term->locate(START_X, BASE_Y + 1);
    term->printf("%4.1f", min);
    
    /*
    //Draw cell-by-cell (slow)
    for(int i = 0; i < WIDTH; ++i){
        for(int j = 0; j < HEIGHT; ++j){
            term->locate(BASE_X + i, BASE_Y - j);
            if( (j*valOfChar) < (pData[i]-min) )
                term->putc((int)'#');
            else
                term->putc((int)' ');
        }
    }
    */
    
    //Draw line-by-line (fast)
    
    for(int i = 0; i < HEIGHT; ++i){
        for(int j = 0; j < WIDTH; ++j){
            pLine[j] = ((i*valOfChar) < (pData[j]-min)) ? filledChar : plainChar;
        }
        term->locate(BASE_X, BASE_Y - i);
        term->printf(pLine);
    }
    
    //TODO: Create one array holding ALL cells and print them at once!
}

void ASCIIGraph::reset(){
    first = true;
    term->locate(START_X, START_Y);
    term->printf("      ");
    term->locate(START_X, BASE_Y + 1);
    term->printf("      ");
}