/*-- SERVER INCLUDES ---------------------------------------------------------*/
#include "mbed.h"
#include "EthernetInterface.h"
/*-- GLOBAL CONSTANTS --------------------------------------------------------*/
const int PORT = 7;                             //arbitrary port
/*-- IP CONSTANT CONFIGURATION --*/
static const char* SERVER_IP = "192.168.1.101"; //IP of server board

static const char* MASK = "255.255.255.0";      //mask
static const char* GATEWAY = "192.168.1.1";     //gateway
/*-- INITIALISATION ----------------------------------------------------------*/
Serial pc(USBTX, USBRX);        //create PC interface
EthernetInterface eth;          //create ethernet
UDPSocket server;               //create Ethernet server
Endpoint client;                //create endpoint
/* -- DECLARE LEDs for STATUS DISPLAY --*/
DigitalOut red(LED_RED);
DigitalOut green(LED_GREEN);
DigitalOut blue(LED_BLUE);
/*-- DECLARE INTERRUPTS --*/
InterruptIn touch(SW2);
InterruptIn butt3(SW3);
Timer t_level, t_shuttle; //define timers to measure each level and shuttle
/*--VARIABLES-----------------------------------------------------------------*/
int i; //Loop iteration counter
int n;                  //size of received message
char message[6] = {0,01,1,1,'\0'};  //MESSAGE status[1],level[2],blue[1],line[1]
char c_status[1], c_level[2], c_blue[1], c_line[1];  //receival buffers

/*---- Status of program? ----*/
/*------ 0 = Loaded      --*/
/*------ 1 = Start       --*/
/*------ 2 = In Progress --*/
/*------ 3 = Stop        --*/
int status, level; //Keep track of the level of the Beep Test
bool line;  //Has runner crossed the line?

/*-- FUNCTION DECLARATIONS ---------------------------------------------------*/
void init_usb(void);    //initializes pc.printf if required
void init_eth(void);    //initializes Ethernet
void init_load(void);   //initialise program
void send(void);        //Send packets
void receive(void);     //receives packets
int main(void);         //main


/*-- INTERRUPT DEFINITIONS ---------------------------------------------------*/
void ISR1()
{
    //Did the runner cross line in time, before showing RED?
    if (red == 0) line = false; //Crossed line late
    else line = true; // Crossed line in time
}
void ISR2()
{
    //Start/End Shuttle Run Test
    if (status == 0) status = 1;
    else status = 0;
}

/*-- FUNCTION DEFINITIONS ---------------------------------------------------*/
/****INIT_USB BAUD RATE***/
void init_usb(void)
{
    pc.baud(9600);    //baud rate
}

/****OPEN_ETHERNET CONNECTION***/
void init_eth(void)
{
    eth.init(SERVER_IP, MASK, GATEWAY);                       //set up IP
    eth.connect();                                       //connect ethernet
    pc.printf("\nSERVER - Server IP Address is %s\r\n", eth.getIPAddress());    //get server IP address;

    server.bind(PORT);                                       //bind server
}

/**** INITIALISE PROGRAM PARAMETERS***/
void init_load(void)
{
    //Initialise flags and reset timers
    level = 1; //Reset to starting level 1
    line = true;
    //Server LEDs flashes RED at 1Hz Loaded Status
    red = !red;
    green = 1;
    blue = 1;
    wait(0.5);
    //Reset the timers
    t_level.reset();
    t_shuttle.reset(); //20*3600/(8000+500*level) = 144/(level+16) sec
}

/**** SEND MESSAGE ***/
void send(void)
{
    sprintf(message, "%i %02i %i %i %c", status, level, int(blue), line, '\0');
    server.sendTo(client, message, sizeof(message));
    pc.printf("SERVER: Send '%s' to client %s\r\n", message, client.get_address()); //print message sent
}

/**** RECEIVE ***/
void receive(void)
{
    pc.printf("\nSERVER: Waiting for UDP packet...\r\n");                                      //wait for packet
    n = server.receiveFrom(client, message, sizeof(message)); //receive message from client
    message[n] = ' ';                               //Remove \0 from message
    pc.printf("SERVER: Received '%s' from client %s\r\n", message, client.get_address());   //print message from client
    sscanf(message, "%s %s %s %s", &c_status, &c_level, &c_blue, &c_line);
    status = atoi(c_status);
    level = atoi(c_level);
    blue = atoi(c_blue);
    c_blue[0] = (blue == 0) ? '0' : '1';
    line = atoi(c_line);
    pc.printf("SERVER: Converted '%i %i %c %i' from client \r\n", status, level, c_blue[0], line);   //print converted message
}   //end receive()

/************ MAIN ************************************************************/
int main(void)
{
    touch.fall(&ISR1); // attach address ISR to interrupt falling edge
    butt3.rise(&ISR2); // attach address ISR to interrupt rising edge

    init_usb();     //initialize the PC interface
    init_eth();     //initialize the Ethernet connection
    receive();     //Wait for connection from client
    
    while (true) { //Repeat forever
        switch (status) {
            case 0 :
                init_load(); //Display PROGRAM LOADED (SERVER) status
                break;
            case 1 : //Start Beep Test
                send();            //Send Message to client
                t_level.start();   //Start timer to measure each level
                t_shuttle.start(); //Start timer to specify each shuttle
                red = 0;
                status = 2; //Change status to In Progress
                break;
            case 2 : //Wait for message
                receive();
                if (line == false) {
                    if (blue == 1) blue = 0;
                    else {
                        status = 3; //Stop the Beep Test
                        break;
                    }
                } //end if-statement
                line = true; //reset the line flag
                if (t_level.read() >= 60) {
                    level++; //New level
                    t_level.reset();
                    t_level.start();//New level - Restart timer
                    for (i=0; i<4; i++) { //Flash green to indicate new level
                        green = !green;
                        wait_ms(250);
                    }
                } //end if-statement
                red = 1;
                green = 0; //Display Solid green
                wait(72/(level+16)); //half(t_shuttle)
                red = 0;  //Display solid Yellow
                wait(72/(level+16));
                send();               //Send message to client
                green = 1;
                break;
            case 3 : //Stop the Beep Test, delay, then reload
                red = blue = 0;
                send();               //Send message to client                
                //LED FLASHES PURPLE at 1HZ square waves for 2 minutes
                for (i=0; i<180; i++){
                    red = !red;
                    blue = !blue;
                    wait_ms(500);
                }
                status = 0; //Reset the Beep Test
                break;
            default :
                red = 0; green = 1;
                for (i=0; i<60; i++) { //FLASHING red green at 2Hz - status-error
                    red = !red;
                    green = !green;
                    wait_ms(250);
                }
                return(0);
        } //End Switch-statement
    } //End while-loop
}   //End Main()