/*-- CLIENT INCLUDES ---------------------------------------------------------*/
#include "mbed.h"
#include "EthernetInterface.h"
/*-- GLOBAL CONSTANTS --------------------------------------------------------*/
const int PORT = 7;                             //arbitrary port
/*-- IP CONSTANT CONFIGURATION --*/
static const char* SERVER_IP = "192.168.1.101"; //IP of server board
static const char* CLIENT_IP = "192.168.1.102"; //IP of client board
static const char* MASK = "255.255.255.0";      //mask          
static const char* GATEWAY = "192.168.1.1";     //gateway
/*-- INITIALISATION ----------------------------------------------------------*/
Serial pc(USBTX, USBRX);        //create PC interface
EthernetInterface eth;          //create ethernet
UDPSocket sock;                 //creat Ethernet socket
Endpoint server;                //create endpoint
/* -- DECLARE LEDs for STATUS DISPLAY --*/
DigitalOut red(LED_RED);    
DigitalOut green(LED_GREEN);
DigitalOut blue(LED_BLUE);
/*-- DECLARE INTERRUPTS --*/
InterruptIn touch(SW2);
InterruptIn butt3(SW3);
Timer t_level, t_shuttle; //define timers to measure each level and shuttle
/*--VARIABLES------------------------------------------------------------*/
int i; //Loop iteration counter
int n;                  //size of received message
char message[6] = {0,01,1,1,'\0'};  //MESSAGE status[1],level[2],blue[1],line[1]
char c_status[1], c_level[2], c_blue[1], c_line[1];  //receival buffers

/*---- Status of program? ----*/
/*------ 0 = Loaded      --*/ 
/*------ 1 = Start       --*/
/*------ 2 = In Progress --*/
/*------ 3 = Stop        --*/
int status, level;       //Keep track of the level of the Beep Test
bool line;               //Has the runner crossed the line?

/*-- FUNCTION DECLARATIONS ---------------------------------------------------*/
void init_usb(void);    //initializes pc.printf if required
void init_eth(void);    //initializes Ethernet
void init_load(void);   //Initialise program
void send(void);        //Send packets
void receive(void);     //Recv packets
int main(void);         //main
void end_eth(void);     //closes Ethernet socket

/*-- INTERRUPT DEFINITIONS ---------------------------------------------------*/
void ISR1()
{
    //Did the runner cross line in time, before showing RED?
    if (red == 0) line = false; //Crossed line late
    else line = true; // Crossed line in time
}
void ISR2()
{
    //Reset the program
    status =0;
}

/*-- FUNCTION DEFINITIONS ---------------------------------------------------*/
/****INIT_USB BAUD RATE***/
void init_usb(void)
{
    pc.baud(9600);    //baud rate
}

/****OPEN_ETHERNET CONNECTION***/
void init_eth(void)
{
    eth.init(CLIENT_IP, MASK, GATEWAY);                  //set up IP
    eth.connect();                                       //connect ethernet
    pc.printf("\nCLIENT - Client IP Address is %s\r\n", eth.getIPAddress()); //get client IP address
    sock.init();                                         //initialize socket
    server.set_address(SERVER_IP, PORT);                 //set address of server
}

/**** INITIALISE CLIENT PARAMETERS***/
void init_load(void)
{
    //Initialise flags and reset timers
    level = 1; //Reset to starting level 1
    line = true;
    //Client LED is solid BLUE in Loaded Status
    blue = 0;
    green = 1;
    red = 1;
    //Reset the timers
    t_level.reset();
    t_shuttle.reset(); //20*3600/(8000+500*level) = 144/(level+16) sec
}

/**** SEND MESSAGE ***/
void send(void)
{
    sprintf(message, "%i %02i %i %i %c", status, level, int(blue), line, '\0');   
    sock.sendTo(server, message, sizeof(message));                //send message
    pc.printf("CLIENT: Send '%s' to server %s\r\n", message, SERVER_IP);//print message
}

/**** RECEIVE ***/
void receive(void)
{
    pc.printf("\nCLIENT: Waiting for UDP packet...\r\n");      //wait for packet
    n = sock.receiveFrom(server, message, sizeof(message)); //receive message from server
    message[n] = ' ';                               //Remove \0 from message
    pc.printf("CLIENT: Received '%s' from server %s\r\n", message, SERVER_IP);   //print message from server
    sscanf(message, "%s %s %s %s", &c_status, &c_level, &c_blue, &c_line);
    status = atoi(c_status);
    level = atoi(c_level);
    blue = atoi(c_blue);
    c_blue[0] = (blue == 0) ? '0' : '1';
    line = atoi(c_line);
    pc.printf("CLIENT: Converted '%i %i %c %i' from server \r\n", status, level, c_blue[0], line);   //print converted message
}   //end receive()

/****CLOSE_ETHERNET CONNECTION***/
void end_eth(void){
    sock.close();       //close socket
    eth.disconnect();   //close Ethernet
}

/************ MAIN ************************************************************/
int main(void)
{
   touch.fall(&ISR1); // attach address ISR to interrupt falling edge
   butt3.rise(&ISR2); // attach address ISR to interrupt rising edge
   
    init_usb();     //initialize the PC interface
    init_eth();     //initialize the Ethernet connection

    while(true) { //repeat forever
        switch (status) {
            case 0 :
                init_load(); //Display Program Loaded status -> Flash Blue
                send();    //Connect to server
                receive(); //Wait for START trigger from Server
                break;
            case 1 : //Start the Beep Test
                blue = 1; //Make sure Blue is OFF
                //2 second COUNTDOWN red, purple, blue, GO!
                red = 0;
                wait_ms(500);
                blue = 0;
                wait_ms(500);
                red = 1;
                wait_ms(500);
                blue = 1;
                status = 2;
                line = true;
                for (i=0; i<4; i++) {
                    green = !green;
                    wait_ms(250);
                }
                green = 0;
                wait(72/(level - 1)); //half(t_shuttle - 1)
                red = 0;
                wait(72/(level)); //half(t_shuttle)
                send();
                red = 1;
                break;
            case 2 :  //Wait for message
                receive();
                if (line == false) {
                    if (blue == 1) blue = 0;
                    else {
                        status = 3; //Stop the Beep Test
                        break;
                    }
                } //end if-statement
                line = true; //reset the line flag
                if (t_level.read() >= 60) {
                    level++; //New level
                    t_level.reset();
                    t_level.start();//New level - Restart timer
                    for (i=0; i<4; i++) { //Flash green to indicate new level
                        green = !green;
                        wait_ms(250);
                    }
                } //end if-statement
                red = 1;
                green = 0; //Display Solid green
                wait(144/(level+16)/2); //t_shuttle = 20*3600/(8000+500*level)
                red = 0; //Display solid Yellow
                wait(144/(level+16)/2);
                send();               //Send message to server
                green = 1;
                break;
            case 3 : //Stop the Beep Test, delay, then reload
                red = blue = 0;
                send();               //Send message to client                
                //LED FLASHES PURPLE at 1HZ square waves for 2 minutes
                for (i=0; i<180; i++){
                    red = !red;
                    blue = !blue;
                    wait_ms(500);
                }
                status = 0; //Reset the Beep Test
                break;
            default :
                red = 0; green = 1;
                for (i=0; i<60; i++) { //FLASHING red green at 2Hz - status-error
                    red = !red;
                    green = !green;
                    wait_ms(250);
                }
                return(0);
                } //End switch-statement
    }//End while-loop
    
}   //end main()