/* WiFi Example
 * Copyright (c) 2018 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "TCPSocket.h"
#include "HTS221Sensor.h"
#include "DHT22.h"
#include "TSL2561.h"

#define WIFI_IDW0XX1    2
#define UBIDOTS_TOKEN   "BBFF-xp89MM8kmzzbp6eJ074XjDpqz7qryh"
#define UBIDOTS_DEVICE  "stmwifi"



#if (defined(TARGET_DISCO_L475VG_IOT01A) || defined(TARGET_DISCO_F413ZH))
#include "ISM43362Interface.h"
ISM43362Interface wifi(MBED_CONF_APP_WIFI_SPI_MOSI, MBED_CONF_APP_WIFI_SPI_MISO, MBED_CONF_APP_WIFI_SPI_SCLK, MBED_CONF_APP_WIFI_SPI_NSS, MBED_CONF_APP_WIFI_RESET, MBED_CONF_APP_WIFI_DATAREADY, MBED_CONF_APP_WIFI_WAKEUP, false);

#else // External WiFi modules

#if MBED_CONF_APP_WIFI_SHIELD == WIFI_IDW0XX1
#include "SpwfSAInterface.h"
SpwfSAInterface wifi(MBED_CONF_APP_WIFI_TX, MBED_CONF_APP_WIFI_RX);
#endif // MBED_CONF_APP_WIFI_SHIELD == WIFI_IDW0XX1

#endif

const char *sec2str(nsapi_security_t sec)
{
    switch (sec) {
        case NSAPI_SECURITY_NONE:
            return "None";
        case NSAPI_SECURITY_WEP:
            return "WEP";
        case NSAPI_SECURITY_WPA:
            return "WPA";
        case NSAPI_SECURITY_WPA2:
            return "WPA2";
        case NSAPI_SECURITY_WPA_WPA2:
            return "WPA/WPA2";
        case NSAPI_SECURITY_UNKNOWN:
        default:
            return "Unknown";
    }
}

int scan_demo(WiFiInterface *wifi)
{
    WiFiAccessPoint *ap;

    printf("Scan:\n");

    int count = wifi->scan(NULL,0);
    printf("%d networks available.\n", count);

    /* Limit number of network arbitrary to 15 */
    count = count < 15 ? count : 15;

    ap = new WiFiAccessPoint[count];
    count = wifi->scan(ap, count);
    for (int i = 0; i < count; i++)
    {
        printf("Network: %s secured: %s BSSID: %hhX:%hhX:%hhX:%hhx:%hhx:%hhx RSSI: %hhd Ch: %hhd\n", ap[i].get_ssid(),
               sec2str(ap[i].get_security()), ap[i].get_bssid()[0], ap[i].get_bssid()[1], ap[i].get_bssid()[2],
               ap[i].get_bssid()[3], ap[i].get_bssid()[4], ap[i].get_bssid()[5], ap[i].get_rssi(), ap[i].get_channel());
    }

    delete[] ap;
    return count;
}

void http_demo(NetworkInterface *net)
{
    TCPSocket socket;
    nsapi_error_t response;

    // Send a simple http request
    char sbuffer[256];
    char message[64];
    
    /* Analog ressources */
    AnalogIn adc_temp(ADC_TEMP);    // Internal Temp Sensor to ADC Channel
    AnalogIn adc_vbat(ADC_VBAT);    // VBAT / 3 internal to ADC channel 
    
    // static DevI2C devI2c(PB_11,PB_10);
    // static DevI2C devI2c(D14,D15);
    //static HTS221Sensor sen_hum_temp(&devI2c);
    static DHT22 tempSensor(D2); // Vérifier que le capteur DHT22 est relié au PIN D2
    static TSL2561 LUM(PB_11, PB_10); // Vérifier que le capteur TSL2591 est bien relié aux ports SDA et SCL
    static AnalogIn analog_value(A0); // Vérifier que le capteur d'humidité du sol est bien relié au port analogique A0
    
    //sen_hum_temp.init(NULL);
    //sen_hum_temp.enable();
    
    /* Global variables */
    float temp;
    float t;
    float hum;
    float h;
    float lux;
    float moist_r;
    float moist_v;
    
    while(true){
        // Open a socket on the network interface, and create a TCP connection
        socket.open(net);
        response = socket.connect("things.ubidots.com", 80);
        if(0 != response) {
            printf("Error connecting: %d\n", response);
            socket.close();
            return;
        }
        printf("Connected to the Server\n");
        
        /* Sensor acquisition */
        // ADC sensors
        printf("Temperature and humidity acquisition\n");
        tempSensor.sample();
        printf("\n");
        t = (float) tempSensor.getTemperature();
        temp = t/10;
        h = (float) tempSensor.getHumidity();
        hum = h/10;
        printf("Luminosity acquisition\n");
        lux = (float) LUM.lux();
        printf("Soil moisture acquisition\n");
        moist_r = analog_value.read();
        printf("Soil moisture conversion\n");
        moist_v = moist_r*100;
        
        /* Construct content of HTTP command */
        sprintf(message, "{\"temperature\": %0.2f, \"humidity\": %0.2f, \"soilmoisture\": %f, \"luminosite\": %0.2f}", temp, hum, moist_v, lux); // Vérifier que les API labels sont bien renseignés
        printf("Content Length = %d\r\n", (int)strlen(message));
        
        /* Construct HTTP command to send */
        sprintf(sbuffer, "POST /api/v1.6/devices/%s/?token=%s HTTP/1.1\r\nHost: things.ubidots.com\r\nContent-Type: application/json\r\nContent-Length: %d\r\n\r\n%s", UBIDOTS_DEVICE, UBIDOTS_TOKEN, (int)strlen(message),message); 
        printf("HTTP command %s\r\n", sbuffer);
        wait(2.0);
                     
        /* Send http request to Ubidots */      
        printf("Sending HTTP request to ubidots.com...\n");
        nsapi_size_t size = strlen(sbuffer);
        printf("send %d [%.*s]\r\n", size, strstr(sbuffer, "\r\n") - sbuffer, sbuffer);
        response = 0;
        while(size)
        {
            response = socket.send(sbuffer+response, size);
            if (response < 0) {
                printf("Error sending data: %d\n", response);
                socket.close();
                return;
            } else {
                size -= response;
                // Check if entire message was sent or not
                printf("sent %d [%.*s]\n", response, strstr(sbuffer, "\r\n")-sbuffer, sbuffer);
            }
        }
    
        /* Receive a simple http response and print out the response line */
        char respBuffer[64];
        int rcount = socket.recv(respBuffer, sizeof respBuffer);
        printf("recv %d [%.*s]\r\n", rcount, strstr(respBuffer, "\r\n") - respBuffer, respBuffer);
        wait(10);
        // Close the socket to return its memory and bring down the network interface
        socket.close();
    }
}

int main()
{
    int count = 0;

    printf("WiFi example\n\n");

    /*
    count = scan_demo(&wifi);
    if (count == 0) {
        printf("No WIFI APNs found - can't continue further.\n");
        return -1;
    }*/

    printf("\nConnecting to %s...\n", MBED_CONF_APP_WIFI_SSID);
    int ret = wifi.connect(MBED_CONF_APP_WIFI_SSID, MBED_CONF_APP_WIFI_PASSWORD, NSAPI_SECURITY_WPA_WPA2);
    if (ret != 0) {
        printf("\nConnection error\n");
        return -1;
    }

    printf("Success\n\n");
    printf("MAC: %s\n", wifi.get_mac_address());
    printf("IP: %s\n", wifi.get_ip_address());
    printf("Netmask: %s\n", wifi.get_netmask());
    printf("Gateway: %s\n", wifi.get_gateway());
    printf("RSSI: %d\n\n", wifi.get_rssi());

    http_demo(&wifi);

    wifi.disconnect();

    printf("\nDone\n");
}
