/** SST25VF - drive the Microchip SST25VF Serial Flash using SPI
*
* Assumes spi mode is default (8,0).
*
* You can clock the SST25VF at up to >20MHz, so it supports the mbed's maximum SPI frequency of 12MHz.
*/

#include "mbed.h"

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*    Caution: Do not write to adresses where data != 0xFF
*    Data in selected adress range MUST be 0xFF (See datasheet)
*  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
*/

#ifndef SST25VF_H
#define SST25VF_H

#define SPI_FREQ        12000000    // 12Mhz is mbed max (one eight of periph. clock)

#define waitShort() wait_us(10)
#define waitErase() wait_ms(50)

// command codes for SST25VF
#define WRITE_STATUS    0x01 // called WRSR in datasheet
#define WRITE           0x02 // Byte-Program
#define READ            0x03
#define WRDI            0x04
#define READ_STATUS     0x05 // called RDSR 
#define WREN            0x06
//#define HSREAD          0x0B // High-Speed-Read
#define EWSR            0x50 // Enable-Write-Status-Register
#define CHIP_ERASE      0x60
#define AAIWRITE        0xAD // word based write

class SST25VF {
public:
    /** Create an interface
    *
    *
    * @param spi   An SPI object
    * @param ncs   Not chip select pin - any free Digital pin will do
    */
    SST25VF(SPI& spi, PinName ncs);
    /** read a byte from SRAM
    * @param address    The address to read from
    * @return the character at that address
    */
    char read(long int address);

    /** read multiple bytes from SRAM into a buffer
    * @param address    The SRAM address to read from
    * @param buffer     The buffer to read into (must be big enough!)
    * @param count      The number of bytes to read
    */
    void read(long int address, char * buffer, int count);
    /** write a byte to SRAM
    * @param address    The address SRAM to write to
    * @param byte       The byte to write there
    */
    void write(long int address, char byte);
    /** write multiple bytes to SRAM from a buffer
    * @param address    The SRAM address write to
    * @param buffer     The buffer to write from
    * @param count      The number of bytes to write
    */
    void write(long int address, char * buffer, int count);


    void chipErase();
    char readStatus();

private:
    SPI& _spi;
    DigitalOut _ncs;

    void writeStatus(char status);
    void prepareCommand(char command, long int address);
    void select();
    void deselect();
    void wren();
    void unlock();
    void wrdi();
};

#endif
