/* SST25VF - drive the Microchip SST25VF Serial Flash using SPI
*/

#include "mbed.h"
#include "SST25VF.h"

SST25VF::SST25VF(SPI& spi, PinName ncs) : _spi(spi), _ncs(ncs)  {

    _spi.format(8,0);
    _spi.frequency(SPI_FREQ);
    deselect();
    unlock();
    chipErase();

}

void SST25VF::select() {
    _ncs = 0;
}

void SST25VF::deselect() {
    _ncs = 1;
}

void SST25VF::wren() {
    select();
    _spi.write(WREN);
    deselect();
    waitShort();
}

void SST25VF::wrdi() {
    select();
    _spi.write(WRDI);
    deselect();
}

void SST25VF::chipErase() {
    wren();
    select();
    _spi.write(CHIP_ERASE);
    deselect();
    waitErase();
}

void SST25VF::unlock() {
    select();
    _spi.write(EWSR);
    deselect();

    select();
    _spi.write(WRITE_STATUS);
    _spi.write(0);
    deselect();
    waitShort();
}

void SST25VF::writeStatus(char status) {
    select();
    _spi.write(WRITE_STATUS);
    _spi.write(status);
    deselect();
    waitShort();
}

char SST25VF::readStatus() {
    select();
    _spi.write(READ_STATUS);
    char result = (char) _spi.write(0);
    deselect();
    return result;
}

void SST25VF::prepareCommand(char command, long int address) {
    select();
    _spi.write(command);
    _spi.write((address & 0xFFFFFF) >> 16);
    _spi.write((address & 0xFFFF)>> 8);
    _spi.write(address & 0xFF);
}

// write or read a single byte

void SST25VF::write(long int address, char byte) {
    wren();
    prepareCommand(WRITE, address);
    _spi.write(byte);
    deselect();
    waitShort();
}

char SST25VF::read(long int address) {
    prepareCommand(READ, address);
    int result = _spi.write(0);
    deselect();
    return (char) result;
}

// continuous write and read

void SST25VF::write(long int address, char * buffer, int count) {
    wren();
    // write frist two bytes
    for (int i = 0; i < count-1; i=i+2) {
        if (i==0) {
            prepareCommand(AAIWRITE, address);
        } else {
            select();
            _spi.write(AAIWRITE);
        }
        _spi.write(buffer[i]);
        _spi.write(buffer[i+1]);
        deselect();
        waitShort();
    }
    if ((count % 2) == 1) { // odd number of bytes       
        select();
        _spi.write(AAIWRITE);
        _spi.write(buffer[count-1]);
        _spi.write(0); // odd number of bytes, so write dummy byte to flash
        deselect();
        waitShort();
    }
    wrdi();
    waitShort();
}

void SST25VF::read(long int address, char * buffer, int count) {
    prepareCommand(READ, address);
    // _spi.write(0); // dummy cycle, only necessary if READ is replaced by HSREAD
    for (int i = 0; i < count; i++) {
        buffer[i] = _spi.write(0);
    }
    deselect();
}