/*
 * Copyright (c) 2006-2020 Arm Limited and affiliates.
 * SPDX-License-Identifier: Apache-2.0
 ***********************************
 * Round trip delay meter. This is the "echo" module. 
 * A microcontroller board with an Ethernet interface.
 * An other microcontroller with "Round Trip Delay" will be needed. 
 * ST NUCLEO H743ZI2 used for testing.
 *   
 * Timo Karppinen 24.04.2021
 ***********************************/
#include "mbed.h"
#include "EthernetInterface.h"

#define REMOTE_PORT 5001    // The port numbers the other way round
#define LOCAL_PORT 5000     // than in the "Round Trip Delay"
#define BUFF_SIZE 1024  // 512 can be takes as largest UDP byte number. 
int UDP_SIZE = 128;    // the actual size will be taken from incoming message

//Network interface
EthernetInterface net;

//Threads
    Thread recv_thread;
//    Thread send_thread; 
    
// UDP
uint8_t ip[4] = {192, 168, 1, 10};       // The remote IP address
SocketAddress clientUDP(ip, NSAPI_IPv4); // The remote device
UDPSocket serverUDP;   // UDP server in this device

// Functions
void udpReceive( void );
void udpSend( void );

DigitalIn sw2(PC_13);    // Blue button on H743ZI, button pressed = TRUE
DigitalOut led2(PE_1);   // Yellow LED on H743ZI

int sw2state = 0;
int sw2old = 1;

char in_data[BUFF_SIZE];
int newDatagram = 0;
int newDatagramOld = 0;

char out_data[BUFF_SIZE]; 

Timer armedFor;
    
int main() {

    printf("\nEcho for the Round Trip Delay application (using Ethernet)\r\n");

    //Bring up the network interface
    //eth.set_network(IP_Adress,MASK,GATEWAY);
    net.set_network("192.168.1.12","255.255.252.0","192.168.1.1");
    net.connect();
    
    // Show network address
    SocketAddress netAddress;
    net.get_ip_address(&netAddress);
    printf("\n\n UDPServer IP Address: %s\n", netAddress.get_ip_address() ? netAddress.get_ip_address():"None");
    
    
    // UDP server 
    
    serverUDP.open(&net);
    int err = serverUDP.bind(LOCAL_PORT);
    printf("Port status is: %d\n",err);
    
    recv_thread.start(udpReceive);
    printf("Listening has been started at port number %d\n", LOCAL_PORT);
    
    //send_thread.start(udpSend);
    printf("Sending out \"Echo\" data to port number %d", REMOTE_PORT);
    printf("Will be armed for triggering by pushing Blue button.\n");
    printf("The IP for the \"Delay metering station\" is fixed! \n");
  
    while(1) {  
    sw2state = sw2.read();
    printf( "\nsw2  %d  - ", sw2state); 
    printf("Not listening - Push the blue button!\n");
    
    if((sw2state == 1)&&(sw2state != sw2old)) {
        led2.write(1);
        armedFor.reset();   // reset timer to zero
        armedFor.start();
        snprintf(out_data, BUFF_SIZE, "Echo -server listening for 60 sec" );
        clientUDP.set_port(REMOTE_PORT);
        udpSend();
         
        snprintf(out_data, UDP_SIZE, "Echo" ); // Updates size on pushing sw2.
        
        // Start polling for the incoming "Echo" UDP datagram   
        while ( armedFor.elapsed_time().count() < 60000000 ){
            //snprintf(out_data, UDP_SIZE, "Echo" );
            if((newDatagram == 1)&&(newDatagram != newDatagramOld)){
                //char firstChar;
                //firstChar = in_data[0];
                //if (firstChar == 85){    // ASCII symbol 85 = "U" for the incoming
                    udpSend();      // Sending out first and then printing!
                    //printf( "firstChar: %s\n", &firstChar);
                 //   } 
            }
            newDatagramOld = newDatagram; //Sending the "Echo" once only.  
            newDatagram = 0;
        }
    }
    sw2old = sw2state; // Once only with pushing the button as long as you like.
    led2.write(0);
    armedFor.stop();    // Stop the "armed for" if did not receive the request.
    for (int k =0; k < BUFF_SIZE; k++){
                    in_data[k] = 0;
                    }

    //printf("\nIn sleep mode for one \n");
    ThisThread::sleep_for(1000ms);
    }
}

// The functions
  
void udpReceive()
{
    int bytes;
    while(1) {
        bytes = serverUDP.recvfrom(&clientUDP, &in_data, BUFF_SIZE);
        //ThisThread::sleep_for(100ms); // For testing only! Comment this line!
        UDP_SIZE = bytes;   // updated for the responce 
        newDatagram = 1;    // set this before using time for printing
        ThisThread::sleep_for(100ms);
        printf("\n");
        printf("bytes received: %d\n",bytes);
        printf("string: %s\n",in_data);
        printf("client address: %s\n", clientUDP.get_ip_address());
        printf("\n");
        }
}

void udpSend()
{
        //char out_data[BUFF_SIZE];
        //snprintf(out_data, BUFF_SIZE, "Echo" );
        //clientUDP.set_port(REMOTE_PORT);
        //ThisThread::sleep_for(200ms); // For testing only! Comment this line!
        serverUDP.sendto(clientUDP, out_data, UDP_SIZE);
        printf("Sending out: %s\n", out_data);
        printf("with %d" , UDP_SIZE);
        printf(" data bytes in UDP datagram\n");   
}