/*
 * Copyright (c) 2006-2020 Arm Limited and affiliates.
 * SPDX-License-Identifier: Apache-2.0
 ***********************************
 * Round trip delay meter. Date and time is taken from a NTP server. 
 * A microcontroller board with an Ethernet interface.
 * An other microcontroller with "Round trip echo" will be needed. 
 * ST NUCLEO H743Zi used for testing.
 *   
 * Timo Karppinen 17.1.2021
 ***********************************/
#include "mbed.h"
#include "EthernetInterface.h"
#include "ntp-client/NTPClient.h"

#define REMOTE_PORT 5000
#define LOCAL_PORT 5001
#define BUFF_SIZE 128  // test with 32, 128, 512, etc

//Network interface
EthernetInterface net;

//Threads
    Thread recv_thread;

// UDP
SocketAddress clientUDP;  // Client on remote device
UDPSocket serverUDP;   // UDP server in this device

//NTP server is a time server used for delivering timing information for networks.
//Returns 32 bits for seconds and 32 bits for fraction of seconds. 
//#define ntpAddress "2.pool.ntp.org"
#define ntpAddress "time.mikes.fi"  // The VTT Mikes in Helsinki
#define ntpPort 123
// The address and port number can be replaced with the ones for the local 
// network NTP server. 

// Functions
//time_t getNTP();
void udpReceive( void );
void udpSend( void );

DigitalIn sw2(PC_13);    // Blue button on H743ZI, button pressed = TRUE
DigitalOut led2(PE_1);   // Yellow LED on H743ZI, TRUE = LED on. 
int sw2state = 0;
int sw2old = 1;

char in_data[BUFF_SIZE];
int newDatagram = 0;
int newDatagramOld = 0;
//time_t timeNTP = 0;

Timer stopwatch; // Timer is an operating system class since OS 6.0
Timer armedFor;
    
int main() {
    printf("\nRound trip delay in UDP messaging (using Ethernet)\n");
    
    //Bring up the network interface
    //eth.set_network(IP_Adress,GATEWAY,MASK); // leave out if using DHCP
    net.set_network("192.168.1.10","192.168.1.1","255.255.252.0");
    net.connect();
    
    // Show network address
    SocketAddress netAddress;
    net.get_ip_address(&netAddress);
    printf("\n\n NTPClient - UDPServer IP Address: %s\n", netAddress.get_ip_address() ? netAddress.get_ip_address():"None");
    
   
    // NTP client 
    
    //printf("Message to NTP time server...\n");
    //timeNTP = getNTP();
    //printf("Current time in day month hour.min.sec year is  %s\r\n", ctime(&timeNTP));
    
    // UDP server 
    
    serverUDP.open(&net);
    int err = serverUDP.bind(LOCAL_PORT);
    printf("Port status is: %d\n",err);
    
    recv_thread.start(udpReceive);
    printf("Listening has been started at port number %d\n", LOCAL_PORT);
    printf("The operator for the \"Echo board\" should send a message!\n ");
    printf("The IP will be taken from the incoming message\n");
    printf("Press the blue switch after receiving \"Echo server listening\"\n");
  
    while(1) {  
    sw2state = sw2.read();
    printf( "\nsw2state is  %d\n", sw2state); // Printing showing "I am alive".
    
    if((sw2state == 1)&&(sw2state != sw2old)) {
        led2.write(0);
        armedFor.reset();   // reset timer to zero
        stopwatch.reset();   // reset stopwatch timer to zero
        //timeNTP = getNTP();
        armedFor.start();
        //stopwatch.start();  // moved to udpSend subroutine
        udpSend();
        
    
        // Start polling for the incoming "Echo" UDP datagram   
        while ( armedFor.elapsed_time().count() < 8000000 ){
            if((newDatagram == 1)&&(newDatagram != newDatagramOld)){
                stopwatch.stop(); 
                char firstChar;
                firstChar = in_data[0];
                printf( "firstChar: %s\n", &firstChar);
                for (int k =0; k < BUFF_SIZE; k++){
                    in_data[k] = 0;
                    } 
            }
            newDatagramOld = newDatagram; //Reading the stopwatch once only 
            newDatagram = 0;   
        }
        
        // printing for testing. Replace with writing once to a SD memory card.
        //printf("Measured at ( day month hour.min.sec year ) %s\r\n", ctime(&timeNTP));
        printf("The time taken was %llu microseconds\n", stopwatch.elapsed_time().count());
        
    }
    sw2old = sw2state; // Once only with pushing the button as long as you like.
    led2.write(1);
    armedFor.stop();
    stopwatch.stop();  // Stop the stopwatch if we did not receive the echo.
    
    ThisThread::sleep_for(1000ms);
    }
}

// The functions
  
time_t getNTP() {
    NTPClient ntp(&net);
    ntp.set_server(ntpAddress, ntpPort);
    time_t timestamp = ntp.get_timestamp();
    if (timestamp < 0) {
        printf("An error occurred when getting the time. Code: %u\r\n", timestamp);
    } 
    else {     // the printings for testing only!
        printf("The timestamp seconds from the NTP server in\r\n  32 bit hexadecimal number is %X\r\n", timestamp);
        printf("  decimal number is %u\r\n", timestamp);
        timestamp += (60*60*2);  //  GMT +2  for Finland for the winter time.
        printf("Current time is %s\r\n", ctime(&timestamp));
    } 
    return timestamp; 
}
    
void udpReceive()
{
    int bytes;
    while(1) {
        bytes = serverUDP.recvfrom(&clientUDP, &in_data, BUFF_SIZE);
        newDatagram = 1;    // set this before using time for printing
        ThisThread::sleep_for(400ms); // waiting for the print buffer
        printf("\n");
        printf("bytes received: %d\n",bytes);
        printf("string: %s\n",in_data);
        printf("client address: %s\n", clientUDP.get_ip_address());
        printf("\n");
 
        }
}

void udpSend()
{
        char out_data[BUFF_SIZE];
        
        snprintf(out_data, BUFF_SIZE, "UDP message for getting the Echo" );
        printf("Sending out: %s\n", out_data);
        printf("with %d" , sizeof(out_data));
        printf(" data bytes in UDP datagram\n");
        clientUDP.set_port(REMOTE_PORT);
        stopwatch.start();  // starting the stopwatch after preparations are done
        serverUDP.sendto(clientUDP, out_data, sizeof(out_data)); 
}
