/**
@file sht11.h

@brief Header file containing member functions and variables

*/

#ifndef __SHT11_H__
#define __SHT11_H__

#include "mbed.h"

/** 
@brief Simple library for interfacing with Sensirion SHT11 

@brief Revision 1.0

@author Tim Meese
@date   January 201
 *
 * Example:
 * @code
 
 #include "mbed.h"
 #include "sht11.h"
 
//        clk, dat
 SHT11 sht(p7, p8);
 
 int main() {
    
   // initialise display 
  lcd.init();
  // print a string in top-left corner
  lcd.printString("Hello, World!",0,0);
  // move cursor to 4th row
  lcd.setXYAddress(0,3);
  // print character
  lcd.printChar('X');
    
  while(1);
 }
  
 * @endcode
 */
 
class SHT11
{

private:
    DigitalOut* clk;
    DigitalInOut* data;
    float lastTemperature;
    float lastHumidity;

    void sendTransmissionStart();
    void connectionReset();
    void softReset();
    int writeByte(unsigned char writeData);
    int readByte(unsigned char *pReadData, bool doAck);
    int readStatus(unsigned char *pRetStatus);
    int writeStatus(unsigned char writeValue);
    int measureTemp(unsigned short *pRetTempRaw);
    int measureHumid(unsigned short *pRetHumidRaw);
    float convertTempCelsius(unsigned short rawTempIn);
    float convertTempFahrenheit(unsigned short rawTempIn);
    float convertHumid(unsigned short rawHumidIn, unsigned short rawTempIn);

public:
    /** Create a SHT11 object connected to the specified pins
    *
    * @param clkPin  Pin connected to clk
    * @param dataPin Pin connected to data
    * 
    */
    SHT11(PinName clkPin, PinName dataPin);
    
    /** Initialise SHT11
    *
    */
    void init();
    int getTemperature(float * pRetTemperature);
    int getTempHumid(float * pRetTemperature, float * pRetHumidity);
};

#endif /* __SHT11_H__ */
