/**
@file bmp085.h

@brief Header file containing member functions and variables

*/

#ifndef __BMP085_H__
#define __BMP085_H__

#include "mbed.h"

#define BMP085_I2C_ADDR_READ  (0xEF)
#define BMP085_I2C_ADDR_WRITE (0xEE)

#define CTRL_REG_ADDR (0xF4)
#define DATA_REG_ADDR (0xF6)

#define MEASURE_TEMP_CMD (0x2E)
#define MEASURE_PRES_CMD (0x34)

// MSB addresses, MSB first
#define CAL_DATA_AC1    (0xAA)
#define CAL_DATA_AC2    (0xAC)
#define CAL_DATA_AC3    (0xAE)
#define CAL_DATA_AC4    (0xB0)
#define CAL_DATA_AC5    (0xB2)
#define CAL_DATA_AC6    (0xB4)
#define CAL_DATA_B1     (0xB6)
#define CAL_DATA_B2     (0xB8)
#define CAL_DATA_MB     (0xBA)
#define CAL_DATA_MC     (0xBC)
#define CAL_DATA_MD     (0xBE)


/** 
@brief Simple library for interfacing with Bosch BMP085

@brief Revision 1.0

@author Tim Meese
@date   January 2015
 *
 * Example:
 * @code
 
 #include "mbed.h"
 #include "bmp085.h"
 
  
 * @endcode
 */


class BMP085
{

private:

    // I2C interface
    I2C * i2c;
 
    const float p0 = 101325;     // Pressure at sea level (Pa)
    float altitude;
   
    // Calibration parameters
    short AC1;
    short AC2;
    short AC3;
    unsigned short AC4;
    unsigned short AC5;
    unsigned short AC6;
    short B1;
    short B2;
    short MB;
    short MC;
    short MD;
    
    // uncalibrated temp, pressure
    long UT;    
    long UP;
    
    // shared values 
    long X1;
    long X2;
    long X3;
    long B3;
    unsigned long B4;
    long B5;
    long B6;
    unsigned long B7;
    long T; // temp in 0.1 degrees C
    long p; // pressure in Pascals

    void readCalData();
    void readUncompTemp();
    void readUncompPressure(int mode);

public:
    /** Create a BMP085 object connected to the specified pins
    *
    * @param clkPin  Pin connected to clk
    * @param dataPin Pin connected to data
    * 
    */
    BMP085(PinName clkPin, PinName dataPin);
    
    /** Initialise SHT11
    *
    */
    void init();
    float getTemperature();
    float getPressure(int oss);
    long getUT();
    long getUP();
};

#endif /* __BMP085_H__ */
