/*   
 * Copyright 2011 Timothy Goh
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */




#ifndef HT1632_H
#define HT1632_H
/*
 * Some defines for internal uses
 */
#define RC_MASTER_MODE 0x830 //Set master mode and clock
                             //source from on-chip RC oscillator
#define SYS_DIS 0x800 //Turn off both system oscillator and
                      //LED duty cycle generator
#define SYS_EN 0x802 //Turn on system oscillator
#define LED_OFF 0x804 //Turn off LED duty cycle generator
#define LED_ON 0x806 //Turn on LED duty cycle generator
#define N_MOS_COM8 0x850
#define PWM_16 0x95e //PWM 16/16 duty

/** Class to provide control over SPI for HT1632 dot-matrix drivers
 *  Note that the protocol is psuedo-SPI. Wirtten for SURE electronics
 *  LED board
 *  Reads are NOT supported
 *
 *  @TODO: Refactor to expose initialization routine
 *
 */

class HT1632 {
public:
    /** Creates a new HT1632 object. Requires that it be connected
     *  to an SPI-capable pin, except for d4. Defaults to 1MHz rate.
     *
     *  @param d1 MISO (Not in use, but specify anyway to make SPI module happy)
     *  @param d2 MOSI (Connect to DATA pin)
     *  @param d3 CLK  (Connect to WR pin)
     *  @param d4 CS   (Connect to CS pin)
     */
    HT1632(PinName d1, PinName d2, PinName d3, PinName d4);
    
    /** Primitive for writing HT1632 12-bit command word. Blocking.
     *  Use draw* functions if possible
     *
     *  @param data Right-aligned 12-bit command
     */
    void write12(uint16_t data);
    
    /** Primitive for writing HT1632 14-bit data word with address.
     *  Blocking.
     *  Use draw* functions if possible
     *
     *  @param data Right-aligned 14-bit data
     */
    void write14(uint16_t data);
    
    /** Display is 4-bit address. Draws a nibble at numerical address
     *  starting from 0.
     *
     *  @param address Address (0-0xff)
     *  @param data Bit pattern to write (0-0x0f)
     */
    void drawnibble(uint8_t address, uint8_t data);
    
    
    /** Draws a byte at column address
     *  starting from 0.
     *
     *  @param address Address (0-0x80)
     *  @param data Bit pattern to write (0-0xff)
     */
    void drawbyte(uint8_t address, uint8_t data);
    
    /** Draws from a buffer provided, to sequential addresses
     *  Swaps nibbles as appropriate. Faster than calling
     *  drawbyte repeatedly because cont-mode is used
     *
     *  @param address Start address
     *  @param data Pointer to buffer containing display pattern
     *  @param count Size of buffer
     */
    void drawbuf(uint8_t address, uint8_t *data, uint8_t count);
private:
    SPI _spi;
    DigitalOut _cs;
    enum _numbits {BITS12, BITS14, BLOCK8} _numbits;

};

#endif