/**
 * @author Eric Lieser
 *
 * @section LICENSE
 *
 * Copyright (c) 2010 ARM Limited
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * ST Microelectronics L6470 dSPIN fully integrated microstepping motor driver
 * with motion engine and SPI
 *
 * Datasheet: http://www.st.com/internet/com/TECHNICAL_RESOURCES/TECHNICAL_LITERATURE/DATASHEET/CD00255075.pdf
 *
 *
 */

/**
* Includes
*/
#include "dSPIN.h"

dSPIN::dSPIN(PinName mosi,
             PinName miso,
             PinName sck,
             PinName cs) : spi_(mosi, miso, sck), nCS_(cs) {

    //5MHz, see page 13 of datasheet for max clock.  May need to be reduces if using long wires
    spi_.frequency(5000000);
    spi_.format(8,3);

    nCS_ = 1;

    wait_us(500);

}

void dSPIN::run(int dir, int speed) {
    char val[3];
    val[0] = (speed & 0xff0000UL) >> 16;
    val[1] = (speed & 0x00ff00UL) >>  8;
    val[2] = (speed & 0x0000ffUL)      ;
    nCS_ = 0;
    if (dir == 1)
        spi_.write(0x51);
    else spi_.write(0x50);
    nCS_ = 1;
    for (int i = 0; i < 3; i++) {
        wait_us(1);
        nCS_ = 0;
        spi_.write(val[i]);
        nCS_ = 1;
    }
}

void dSPIN::move(int dir, int steps) {
    char val[3];
    val[0] = (steps & 0xff0000UL) >> 16;
    val[1] = (steps & 0x00ff00UL) >>  8;
    val[2] = (steps & 0x0000ffUL)      ;
    nCS_ = 0;
    if (dir == 1)
        spi_.write(0x41);
    else spi_.write(0x40);
    nCS_ = 1;
    for (int i = 0; i < 3; i++) {
        wait_us(1);
        nCS_ = 0;
        spi_.write(val[i]);
        nCS_ = 1;
    }
}

void dSPIN::soft_stop() {
    nCS_ = 0;
    spi_.write(0xB0);
    nCS_ = 1;
}

void dSPIN::set_param(char parameter, int length, int value) {
    char val[3];
    switch (length) {
        case 1:
            val[0] = (value & 0x0000ffUL)      ;
            break;
        case 2:
            val[0] = (value & 0x00ff00UL) >>  8;
            val[1] = (value & 0x0000ffUL)      ;
            break;
        case 3:
            val[0] = (value & 0xff0000UL) >> 16;
            val[1] = (value & 0x00ff00UL) >>  8;
            val[2] = (value & 0x0000ffUL)      ;
            break;
    }
    nCS_ = 0;
    spi_.write(parameter);
    nCS_ = 1;
    for (int i = 0; i < length; i++) {
        wait_us(1);
        nCS_ = 0;
        spi_.write(val[i]);
        nCS_ = 1;
    }

}

int dSPIN::get_param(char parameter, int length) {
    char output = 0x20 | parameter;
    char buf[3] = {0, 0, 0};
    nCS_ = 0;
    spi_.write(output);
    nCS_ = 1;
    for (int i = 0; i < length; i++) {
        wait_us(1);
        nCS_ = 0;
        buf[i] = spi_.write(0x00);
        nCS_ = 1;
    }
    switch (length) {
        case 1:
            return buf[0];
        case 2:
            return buf[0] << 8 | buf[1];
        case 3:
            return buf[0] << 16 | buf[1] << 8 | buf[2];
    }
    return 0;
}

int dSPIN::get_status() {
    int ret_bytes[2];
    nCS_ = 0;
    spi_.write(0xD0); //write request for status return
    nCS_ = 1;
    wait_us(1);
    nCS_ = 0;
    ret_bytes[0] = spi_.write(0x00);  //read first byte
    nCS_ = 1;
    wait_us(1);
    nCS_ = 0;
    ret_bytes[1] = spi_.write(0x00);  //read second byte
    nCS_ = 1;
    return ret_bytes[0] << 8 | ret_bytes[1];  //return i16 response
}

void dSPIN::reset_device() {
    nCS_ = 0;
    spi_.write(0xC0);
    nCS_ = 1;
}