/**
 * @author Eric Lieser
 *
 * @section LICENSE
 *
 * Copyright (c) 2010 ARM Limited
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * ST Microelectronics L6470 dSPIN fully integrated microstepping motor driver
 * with motion engine and SPI
 *
 * Datasheet: http://www.st.com/internet/com/TECHNICAL_RESOURCES/TECHNICAL_LITERATURE/DATASHEET/CD00255075.pdf
 *
 *
 */

#ifndef dSPIN_H
#define dSPIN_H


/**
 * Includes
 */
#include "mbed.h"

/**
 * Defines
 */
#define ABS_POS         0x01        //length: 22      reset value: 0
#define EL_POS          0x02        //length: 9       reset value: 0
#define MARK            0x03        //length: 22      reset value: 0
#define SPEED           0x04        //length: 20      reset value: 0
#define ACC             0x05        //length: 12      reset value: 08A
#define DEC             0x06        //length: 12      reset value: 08A
#define MAX_SPEED       0x07        //length: 10      reset value: 41
#define MIN_SPEED       0x08        //length: 13      reset value: 0
#define FS_SPD          0x15        //length: 10      reset value: 027
#define KVAL_HOLD       0x09        //length: 8       reset value: 29
#define KVAL_RUN        0x0A        //length: 8       reset value: 29
#define KVAL_ACC        0x0B        //length: 8       reset value: 29
#define KVAL_DEC        0x0C        //length: 8       reset value: 29
#define INT_SPD         0x0D        //length: 14      reset value: 0408
#define ST_SLP          0x0E        //length: 8       reset value: 19
#define FN_SLP_ACC      0x0F        //length: 8       reset value: 29
#define FN_SLP_DEC      0x10        //length: 8       reset value: 29
#define K_THERM         0x11        //length: 4       reset value: 0
#define ADC_OUT         0x12        //length: 5       reset value: 
#define OCD_TH          0x13        //length: 4       reset value: 8
#define STALL_TH        0x14        //length: 7       reset value: 40
#define STEP_MODE       0x16        //length: 8       reset value: 7
#define ALARM_EN        0x17        //length: 8       reset value: FF
#define CONFIG          0x18        //length: 16      reset value: 2E88
#define STATUS          0x19        //length: 16      reset value:

/**
 * ST Micro dSPIN stepper motor controller
 */
class dSPIN {

public:

    /**
     * Constructor.
     *
     * @param mosi mbed pin to use for MOSI line of SPI interface.
     * @param miso mbed pin to use for MISO line of SPI interface.
     * @param sck mbed pin to use for SCK line of SPI interface.
     * @param cs mbed pin to use for not chip select line of SPI interface.
     */
    dSPIN(PinName mosi, PinName miso, PinName sck, PinName cs);
    
    void run(int dir, int speed);
    
    void move(int dir, int steps);
    
    void soft_stop(void);
    
    void set_param(char parameter, int length, int value);
    
    int get_param(char parameter, int length);
    
    int get_status(void);
    
    void reset_device(void);

private:

    SPI        spi_;
    DigitalOut nCS_;

};



#endif /* dSPIN */
