#include "mbed.h"
#include "register_address.h"
#include "common_task.h"
#include "EthernetInterface.h"
#include "TCPServer.h"
#include "TCPSocket.h"


extern Serial pc;

extern DigitalOut led3;
extern DigitalOut led2;
extern DigitalOut led1;

extern SPI spi; // mosi, miso, sclk
extern DigitalOut SSN;

extern DigitalIn button_capture;
extern InterruptIn irq_cis;

extern DigitalOut RSTN;  
extern DigitalOut SLEEPN; 
extern DigitalOut CAPTURE;
extern DigitalOut CONFIG1;           
extern DigitalOut CONFIG2;           

extern uint8_t DCMI_8b_mode;

extern EthernetInterface eth;
extern TCPSocket cs;  //com socket
extern char ServerIP[17];
extern int portnumber;
extern int portnumber_com;

char toprint[65];
char toprinttemp[65];


/**
  * @brief print string to pc and sent it via com socket
  * @param const char* mystring : string to print
  * @retval int number of byte sent via com socket
  */
int myprint(const char* mystring ){ 

        int byte_sent=-1;
        int status=0;
        
        pc.printf("%s\n\r",mystring);
        byte_sent=cs.send(mystring, strlen(mystring));
        return byte_sent;
        
}

/**
  * @brief usage
  * @param nonce
  * @retval void
  */
void usage(void){
    
    myprint("Usage\n\r");
    myprint("----------------------------------------\n\r");
    myprint("H   -> help\n\r");
    myprint("--------------SPI--------------------------\n\r");
    myprint("spi -> test SPI\n\r");
    myprint("G   -> get register configurations\n\r");
    myprint("--------------IMAGER--------------------------\n\r");
    myprint("F   -> take frame 8b\n\r");
    myprint("FQ  -> take frame ion QQVGA mode 8b\n\r");
    myprint("sp  -> suspend frame\n\r");
    myprint("st  -> stop frame\n\r");
    myprint("r   -> resume frame\n\r");
    myprint("s   -> send frame tp teraterm\n\r");
    myprint("e   -> set exposure time SPI\n\r");
    myprint("m   -> set mode via SPI\n\r");
    myprint("AG  -> set analog gain SPI\n\r");
    myprint("A   -> test all analog gain configurations SPI\n\r");
    myprint("CC  -> set cclk value SPI\n\r");
    myprint("C   -> auto calibration of cclk SPI\n\r");
    myprint("IO  -> set IO SR and driving strength SPI\n\r");
    myprint("pclk -> set PCLK mode\n\r"); 
    myprint("pclkd -> set PCLK delay\n\r");   
    myprint("md   -> set motion detection cycle time delay\n\r");   
    myprint("oen      -> (dis)enable oen\n\r"); 
    myprint("pbp      -> (dis)enable pbp\n\r"); 
    myprint("sleepn   -> (dis)enable sleepn\n\r"); 
                
    myprint("------------- ethernet-----------------------\n\r");
    myprint("si  -> set ip server\n\r");
    myprint("gi  -> get ip server\n\r");
    myprint("------------- DCMI-----------------------\n\r");
    myprint("gs  -> get status DCMI\n\r");
    myprint("ge  -> get error DCMI\n\r");
    myprint("gsd -> get status DMA\n\r");
    myprint("ged -> get error DMA\n\r");
    myprint("gl  -> get nb line received on DCMI\n\r");
    myprint("gf  -> get nb frame received on DCMI\n\r");
    myprint("------------- SCVR-----------------------\n\r");
    myprint("vdda  -> set vdda register\n\r");
    myprint("vddd  -> set vddd register\n\r");
    myprint("pm  -> set power mode register\n\r");
    myprint("pma  -> set pwd vddla low medium high register\n\r");
    myprint("pmd  -> set pwd vddld low medium high register\n\r");
    myprint("----------------------------------------\n\r\n\r");

}

/**
  * @brief initialise gpio for chip control
  * @param nonce
  * @retval void
  */
void TASK_INIT_SENSOR(){ 
                      
        // Enable reset
        RSTN=0;       //0 => reset                OK             
        wait_us(5000);             
        
         //DEFAULT SIGNAL OUTPUT
        
        // Output enable
        
        CONFIG2=0;
        CONFIG1=1;
        /* config 
        •   Pad 2’b00: version 1
        •   Pad 2’b01: version 2 (default)
        •   Pad 2’b10: version 3
        •   Pad 2’b11: test mode - scan chain
        */
        
        
        SLEEPN=0;     //1 => active 
        CAPTURE=0;    //0 => no capture  
        SSN=1;        //1 => spi disabled 
        wait_us(5000);
        
        // Disable reset
        RSTN=1;       
        wait_us(5000);
        
        
        
                
}

/**
  * @brief reset CIS001
  * @param nonce
  * @retval void
  */
void TASK_RSTN_SENSOR(){
        RSTN=0;
        wait_ms(2000);  
        RSTN=1;
        wait_us(1000);
            
}

/**
  * @brief TASK_SPI_WRITE : write data in the register located at adr via the SPI
  * @param int addresse
  * @param int data
  * @retval void
  */
void TASK_SPI_WRITE(int adr, int data){
    
    //sprintf(toprint,"writing addr %d data %d\n\r",adr,data);
    //sprintf(toprint,"sending addr %02X data %02X\n\r",adr,data);
    //myprint(toprint);
    
    int packet = (adr<<8 | data);
    SSN=0;
    wait_us(50);
    spi.write(packet);
    
    wait_us(50);
    SSN=1;
    wait_us(100);
    
}

/**
  * @brief TASK_SPI_READ : read data in the register located at adr via the SPI
  * @param int addresse
  * @retval int the data read at address adr
  */
int TASK_SPI_READ(int adr){
    
    //sprintf(toprint,"reading addr %d \n\r",adr);
    //sprintf(toprint,"reading addr %02X \n\r",adr);
    //myprint(toprint);
    
    int rx_buffer=0; 
    int packet = (adr|0x80)<<8;
    
    SSN=0; 
    rx_buffer=spi.write(packet);                           //--ok le spi repond mais je sais pas choper la data.
    wait_us(50);
    SSN=1;
    wait_us(100);
     
    //sprintf(toprint,"read at add %d -> data %d  \n\r",adr,rx_buffer);
    //myprint(toprint);
    return (int) rx_buffer;
    
}

/**
  * @brief TASK_TEST_SPI : write and read back different data in the test reg via the spi
  * @param none
  * @retval int : return 1 if OK, return 0 if KO
  */
int TASK_TEST_SPI(void){
    
    uint8_t a;
    const uint8_t size=4;
    int data[size]={0x55, 0xAA, 0xFF, 0x00};
    int pass=0;
    int i=0;
    
    int length = sizeof(data)/sizeof(data[0]);
    
    for (i=0;i<length;i++){
        TASK_SPI_WRITE((int)test_add,data[i]);
        a=TASK_SPI_READ((int)test_add);
        if (a==data[i])
            pass++;
    }
    
   return (pass==size);
   
}



void TASK_ANALOG_CALIB(void){
    
    uint8_t a,b,c,e;
    int d;
    float average;
    
    e=TASK_SPI_READ(analog_gain_add);     
    
    TASK_SPI_WRITE(mode_add,0x02);       //mode SO+calib
    myprint("mode SO+calib\r\n");
    
    for (int i=0;i<16;i++){
        
        TASK_SPI_WRITE(analog_gain_add,i); 
        
        TASK_SPI_WRITE(capture_add,0xff);       //capture on
        thread_sleep_for(1);
        TASK_SPI_WRITE(capture_add,0x00);       //capture off
        thread_sleep_for(200);
        
 
        a=TASK_SPI_READ(CAL_RAMP_COUNT_0_add);             //read value LSB
        b=TASK_SPI_READ(CAL_RAMP_COUNT_1_add);
        c=TASK_SPI_READ(CAL_RAMP_COUNT_2_add);             //MSB
    
        d=(int) c*256*256+b*256+a;
        average=(float) d/480;
        
        sprintf(toprint,"ramp cal: %i -> %i\taverage %f\r\n",i,d,average);
        myprint(toprint);
        
        wait_us(5000);
    }
    TASK_SPI_WRITE(mode_add,0x03);       //mode DRS+calib
    myprint("\r\nmode DRS+calib\r\n");
    
    for (int i=0;i<16;i++){
        
        TASK_SPI_WRITE(analog_gain_add,i); 
        
        TASK_SPI_WRITE(capture_add,0xff);       //capture on
        wait_us(1000);
        TASK_SPI_WRITE(capture_add,0x00);       //capture off
        wait_us(200000);
        
 
        a=TASK_SPI_READ(CAL_RAMP_COUNT_0_add);             //read value LSB
        b=TASK_SPI_READ(CAL_RAMP_COUNT_1_add);
        c=TASK_SPI_READ(CAL_RAMP_COUNT_2_add);             //MSB
    
        d=(int) c*256*256+b*256+a;
        average=(float) d/480;
        
        
        sprintf(toprint,"ramp cal: %i -> %i\taverage %f\r\n",i,d,average);
        myprint(toprint);
        //wait_ms(50);
    }
    
    TASK_SPI_WRITE(0x03,0x01);       //mode drs 
    TASK_SPI_WRITE(analog_gain_add,e); 
    
}




void TASK_auto_ccal(){
    
    uint8_t a,b;
    uint8_t cclk=10;
    uint8_t upper=255;
    uint8_t lower=0;
    int res;
    int iter=0;
    
    int lb=1020;
    int ub=1024;
    
    myprint("**************\r\n");
    myprint("AUTO CCLK CALIB\r\n");
    myprint("**************\r\n");
    TASK_SPI_WRITE(mode_add,0x05);       //mode cclk calib  
    
    
    
    TASK_SPI_WRITE(cclk_cal_add,cclk);                   //default value of cclk reg
    TASK_SPI_WRITE(capture_add,0xff);                   //capture on
    wait_us(100);
    TASK_SPI_WRITE(capture_add,0x00);                   //capture off
    wait_us(100);
    a=TASK_SPI_READ(CAL_CLK_BUFF_0_add);                //read value LSB
    b=TASK_SPI_READ(CAL_CLK_BUFF_1_add);                //MSB
    res=(int) b*256+a;
    
    while( (res <lb || res > ub) && iter<8){ 
 
        TASK_SPI_WRITE(cclk_cal_add,cclk);       //default value of cclk reg
        a=TASK_SPI_READ(cclk_cal_add);
        //myprint("write 0x%02x at 0x%02x. read 0x%02x at 0x%02x\r\n",cclk,cclk_cal_add,a,cclk_cal_add);
        
        TASK_SPI_WRITE(capture_add,0xff);       //capture on
        wait_us(1000);
        TASK_SPI_WRITE(capture_add,0x00);       //capture off
        wait_us(1000);
        a=TASK_SPI_READ(CAL_CLK_BUFF_0_add);             //read value LSB
        b=TASK_SPI_READ(CAL_CLK_BUFF_1_add);             //MSB
        res=(int) b*256+ a;
        
        //myprint("cclk cal: %i -> cclk %i\r\n",cclk, b*256+a);
        sprintf(toprint,"cclk cal: %i -> cclk %i\r\n",cclk,res);
        myprint(toprint);
        
        //updated stuff
        if(res<lb){         
            //myprint("too low: %i %i %i %i",res,cclk,lower,upper);
            lower=cclk;
            cclk=(lower+upper)/2;
            //myprint(" -> %i  %i %i\r\n\r\n",cclk,lower,upper);
        } else if(res>ub){
            //myprint("too high: %i %i %i %i",res,cclk,lower,upper);
            upper=cclk;
            cclk=(lower+upper)/2;
            //myprint(" -> %i %i %i \r\n\r\n",cclk,lower,upper);
        } else {
            myprint("ok !!\r\n");
        }    
        iter++;
    }
    
    sprintf(toprint,"cclk cal: %i -> cclk %i\r\n",cclk,res);
    myprint(toprint);
    TASK_SPI_WRITE(0x03,0x01);       //mode drs
    
}

