#include "mbed.h"
#include "QEI.h"
#include <ros.h>
#include <std_msgs/String.h>
#include <std_msgs/Float32.h>

#include "millis.h"
 
// ===== ENCODERS
// motor_pulses_per_rev
int pulses_per_rev = (2000); // motor revs

// Gear ratio / (2 * pi) * encoder_counts
float pulses_per_wheel_rad = (35 / 28) * (2000) / (2 * 3.14159);

// Prev encoder value
float wheel0 = 0;
float wheel1 = 0;
float wheel2 = 0;
float wheel3 = 0;

float wheel0_prev = 0;
float wheel1_prev = 0;
float wheel2_prev = 0;
float wheel3_prev = 0;

float kP = 0.05;
float kF = 0.035;

int pub_skip = 2;
int count = 0;

float dt = 0.005;
float prev_t = 0;

// Motor speed set points
float motor0_sp = 0;
float motor1_sp = 0;
float motor2_sp = 0;
float motor3_sp = 0;

// Verify encoder lengths
QEI encP0(A0, A2, NC, pulses_per_rev);
QEI encP1(D6, D12, NC, pulses_per_rev);
QEI encP2(A3, A6, NC, pulses_per_rev);
QEI encP3(D4, D5, NC, pulses_per_rev);
 
// ===== MOTOR outputs
DigitalOut motor2A(D13);  
DigitalOut motor2B(A1);

DigitalOut motor1A(D8);
DigitalOut motor1B(D10);


DigitalOut motor3A(A4); 
DigitalOut motor3B(A5);

DigitalOut motor0A(D2); 
DigitalOut motor0B(D3);

PwmOut motor2_En(D0);
PwmOut motor1_En(D9); 
PwmOut motor3_En(D1); 
PwmOut motor0_En(D11);


void setMotor0Power(float power){
    if( power > 0 ){
        motor0A = 1;
        motor0B = 0;
    }else if( power < 0 ){
        motor0A = 0;
        motor0B = 1;
    } else {
        motor0A = 0;
        motor0B = 0;
    }
    motor0_En = abs(power);
}

void setMotor1Power(float power){
    if( power > 0 ){
        motor1A = 1;
        motor1B = 0;
    }else if( power < 0 ){
        motor1A = 0;
        motor1B = 1;
    } else {
        motor1A = 0;
        motor1B = 0;
    }
    motor1_En = abs(power);
}

void setMotor2Power(float power){
    if( power > 0 ){
        motor2A = 1;
        motor2B = 0;
    }else if(power < 0 ){
        motor2A = 0;
        motor2B = 1;
    } else {
        motor2A = 0;
        motor2B = 0;
    }
    motor2_En = abs(power);
}

void setMotor3Power(float power){
    if( power > 0 ){
        motor3A = 1;
        motor3B = 0;
    }else if( power < 0 ){
        motor3A = 0;
        motor3B = 1;
    } else {
        motor3A = 0;
        motor3B = 0;
    }
    motor3_En = abs(power);
}

// ======= MOTOR CALLBACKS
void motor0_callback(const std_msgs::Float32& msg){
    motor0_sp = msg.data;
}

void motor1_callback(const std_msgs::Float32& msg){
    motor1_sp = msg.data;
}

void motor2_callback(const std_msgs::Float32& msg){
    motor2_sp = msg.data;
}

void motor3_callback(const std_msgs::Float32& msg){
    motor3_sp = msg.data;
}

    

// ===== LED
//DigitalOut myled(LED2);
//DigitalOut led = LED1;

// ===== ROSSERIAL SETUP
ros::NodeHandle  nh;

// Encoder publisher
std_msgs::Float32 enc_pos0;
ros::Publisher encoderP0("hardware/encoders/drive0", &enc_pos0);

std_msgs::Float32 enc_pos1;
ros::Publisher encoderP1("hardware/encoders/drive1", &enc_pos1);

std_msgs::Float32 enc_pos2;
ros::Publisher encoderP2("hardware/encoders/drive2", &enc_pos2);

std_msgs::Float32 enc_pos3;
ros::Publisher encoderP3("hardware/encoders/drive3", &enc_pos3);


// Drive subscribers
ros::Subscriber<std_msgs::Float32> motor0("hardware/motors/drive0", motor0_callback);
ros::Subscriber<std_msgs::Float32> motor1("hardware/motors/drive1", motor1_callback);
ros::Subscriber<std_msgs::Float32> motor2("hardware/motors/drive2", motor2_callback);
ros::Subscriber<std_msgs::Float32> motor3("hardware/motors/drive3", motor3_callback);

// ===== MAIN
int main() {
    
    nh.initNode();
    nh.advertise(encoderP0);
    nh.advertise(encoderP1);
    nh.advertise(encoderP2);
    nh.advertise(encoderP3);
    
    nh.subscribe(motor0);
    nh.subscribe(motor1);
    nh.subscribe(motor2);
    nh.subscribe(motor3);
    
    while (1) {
        
        //dt = (millis() / 1000.0) - prev_t;
        //prev_t = (millis() / 1000.0);
        
        if(count >= pub_skip){
            
            // ENCODER P4
            wheel0 = (float) encP0.getPulses() / pulses_per_wheel_rad;
            setMotor0Power( -kP * ((wheel0 - wheel0_prev)/dt - motor0_sp) + kF * motor0_sp);
            enc_pos0.data = wheel0;
            encoderP0.publish( &enc_pos0 );
            wheel0_prev = wheel0;
            
            // ENCODER P3
            wheel1 = (float) encP1.getPulses() / pulses_per_wheel_rad;
            setMotor1Power( -kP * ((wheel1 - wheel1_prev)/dt - motor1_sp) + kF * motor1_sp);
            enc_pos1.data = wheel1;
            encoderP1.publish( &enc_pos1 );
            wheel1_prev = wheel1;
            
            // ENCODER P2
            wheel2 = (float) encP2.getPulses() / pulses_per_wheel_rad;
            setMotor2Power( -kP * ((wheel2 - wheel2_prev)/dt - motor2_sp) + kF * motor2_sp);
            enc_pos2.data = wheel2;
            encoderP2.publish( &enc_pos2 );
            wheel2_prev = wheel2;
            
            // ENCODER P3
            wheel3 = -(float) encP3.getPulses() / pulses_per_wheel_rad;
            setMotor3Power( -kP * ((wheel3 - wheel3_prev)/dt - motor3_sp) + kF * motor3_sp);
            enc_pos3.data = wheel3;
            encoderP3.publish( &enc_pos3 );
            wheel3_prev = wheel3;
            
            count = 0;
        
        }
        
        
        // Spin the rosserial node handler
        nh.spinOnce();
        
        wait(dt); // wait.....
        count ++;
    }
}
