/*
 * Copyright (c) 2013, The Robot Studio
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Created on: Feb 27, 2013
 *      Author: Cyril Jourdan (cyril.jourdan@therobotstudio.com)
 */
 
#include "../include/eposCmd.h"

CAN cantoepos(p9, p10);
//Serial pc(p28, p27);

Serial pc(USBTX, USBRX);
DigitalOut ledchain[] = {(LED1), (LED2), (LED3), (LED4)}; //used for debugging
char data[8];
//unsigned int count[12] = {0};
ActivatedMode activMode[NUMBER_MAX_EPOS2_PER_SLAVE];

//ADC variables
AnalogIn an1(p15);
AnalogIn an2(p16);
AnalogIn an3(p17);
AnalogIn an4(p18);
AnalogIn an5(p19);
AnalogIn an6(p20);

//sensor variables
int32_t encPosition[NUMBER_MAX_EPOS2_PER_SLAVE] = {0};
int16_t potiPosArray[NUMBER_MAX_EPOS2_PER_SLAVE] = {0};
int16_t medPotiPosition[NUMBER_MAX_EPOS2_PER_SLAVE] = {0};
int16_t avgCurrent[NUMBER_MAX_EPOS2_PER_SLAVE] = {0};
int16_t medForce[NUMBER_MAX_EPOS2_PER_SLAVE] = {0};
int32_t velocity[NUMBER_MAX_EPOS2_PER_SLAVE] = {0};

int8_t boardStatus[NUMBER_MAX_EPOS2_PER_SLAVE] = {0}; //0 = OK, 1 = fault

//Multiplexer address lines
DigitalOut A0(p24);
DigitalOut A1(p22);
DigitalOut A2(p21);

void setMultiplexerChannel(const int8_t nodeID) //for force values
{
    switch (nodeID)
    {
        //Analog 1
        case 2:             //Supra
            A0 = 1;
            A1 = 0;
            A2 = 0;
                break;    
        case 3:              //Infra
            A0 = 0;
            A1 = 1;
            A2 = 0;
                break;
        case 4:              //Tmin  
            A0 = 1;
            A1 = 1;
            A2 = 0;
               break;
        case 6:               //Subscap  
            A0 = 1;
            A1 = 0;
            A2 = 1;
                break;
        case 7:              //Ant Delt  
            A0 = 0;
            A1 = 1;
            A2 = 1;  
                break; 
        case 8:              //Lat Delt  
            A0 = 1;
            A1 = 1;
            A2 = 1;
                break;
        //Analog 3
        case 9:              //Post Delt nb AN3
            A0 = 1;
            A1 = 0;
            A2 = 1;
                break;
        default:
            //pc.printf("invalid node");       
                break;
    }
}

void setNMT(const int8_t nodeID, uint8_t cs)
{
    //Firmware Spec 7.3
    //CANMessage canmsg; //test
    data[0] = cs;
    data[1] = nodeID;
    cantoepos.write(CANMessage(CAN_NMT_ID, data, 2));
   /* 
    //test
    if(cs == CS_RESET_COMMUNICATION)
    {
        while (!(cantoepos.read(canmsg))) //wainting for the boot-up message
        {
            wait_us(1);
        }  
    }
    //fin test
     */   
    wait_ms(PAUSE);      
}

int8_t setObjectSDO(const int8_t nodeID, const int32_t object, int32_t value)
{
    CANMessage canmsg; 
    uint8_t nbByteObject;
    uint8_t nbByte = 0;
    uint32_t timeout = 0;
    
    //nbByteObject = object & 0x000000FF;
    nbByteObject = (uint8_t)object;
    
    switch(nbByteObject)
    {
        case 0x08 : 
            nbByteObject = WRITING_OBJECT_1_BYTE;
            nbByte = 1;
            break;
            
        case 0x10 : 
            nbByteObject = WRITING_OBJECT_2_BYTE;
            nbByte = 2;
            break;
            
        case 0x20 : 
            nbByteObject = WRITING_OBJECT_4_BYTE;
            nbByte = 4;
            break;
        
        default :
            return EPOS2_ERROR;            
    }   
    
    data[0] = nbByteObject;
    data[1] = (uint8_t)(object >> 16); 
    data[2] = (uint8_t)(object >> 24); 
    data[3] = (uint8_t)(object >> 8);
    data[4] = (uint8_t)value;
    data[5] = (uint8_t)(value >> 8);
    data[6] = (uint8_t)(value >> 16);
    data[7] = (uint8_t)(value >> 24);
    cantoepos.write(CANMessage(COB_ID_SDO_CLIENT_TO_SERVER_DEFAULT + nodeID, data, 4 + nbByte));
    while (!(cantoepos.read(canmsg))) 
    {
        wait_us(1); //use a timeout instead
        
        if(timeout >= TIMEOUT)
        {
            return EPOS2_ERROR;
        }
        
        timeout++;
    }
    wait_ms(PAUSE);   
    
    //pc.printf("setSDO[%d] [%02X %02X %02X %02X %02X %02X %02X %02X]\n\r", nodeID, data[7], data[6], data[5], data[4], data[3], data[2], data[1], data[0]);
          
    return EPOS2_OK;
}

int8_t getObjectSDO(const int8_t nodeID, const int32_t object, int32_t value)
{
    //todo
    return EPOS2_OK;
}

int8_t setPDO(const int8_t nodeID, uint16_t pdoIdx, uint8_t subIdx, uint32_t value, uint8_t nbByte)
{
    CANMessage canmsg; 
    uint8_t nbByteObject;
        
    switch(nbByte)
    {
        case 1 : 
            nbByteObject = WRITING_OBJECT_1_BYTE;
            break;
            
        case 2 : 
            nbByteObject = WRITING_OBJECT_2_BYTE;
            break;
            
        case 4 : 
            nbByteObject = WRITING_OBJECT_4_BYTE;
            break;
        
        default :
            return EPOS2_ERROR;            
    }   
    
    data[0] = nbByteObject;
    data[1] = (uint8_t)pdoIdx;
    data[2] = (uint8_t)(pdoIdx >> 8);
    data[3] = subIdx;
    data[4] = (uint8_t)value;
    data[5] = (uint8_t)(value >> 8);
    data[6] = (uint8_t)(value >> 16);
    data[7] = (uint8_t)(value >> 24);
    cantoepos.write(CANMessage(COB_ID_SDO_CLIENT_TO_SERVER_DEFAULT + nodeID, data, 4 + nbByte));
    while (!(cantoepos.read(canmsg))) 
    {
        wait_us(1);
    }
    wait_ms(PAUSE);   
          
    return EPOS2_OK;
}

int8_t setModeOfOperation(const int8_t nodeID, uint8_t mode)
{
    //Switch Mode of Operation  (Firmware Spec 8.2.89)
    
    CANMessage canmsg;
 
    data[0] = WRITING_OBJECT_1_BYTE;
    data[1] = (uint8_t)OBJECT_MODE_OF_OPERATION_INDEX;
    data[2] = (uint8_t)(OBJECT_MODE_OF_OPERATION_INDEX >> 8);
    data[3] = OBJECT_MODE_OF_OPERATION_SUBINDEX;
    data[4] = mode;
    data[5] = 0x00;
    data[6] = 0x00;
    data[7] = 0x00;
    
    cantoepos.write(CANMessage(COB_ID_SDO_CLIENT_TO_SERVER_DEFAULT + nodeID, data, 5));
    while (!(cantoepos.read(canmsg))) 
    {
        wait_us(1);
    }
    
    //set activMode        
    switch(mode)
    {    
        case VALUE_PROFILE_POSITION_MODE : 
            activMode[nodeID-1] = PROFILE_POSITION;            
            break;
            
        case VALUE_POSITION_MODE : 
            activMode[nodeID-1] = POSITION;            
            break;
            
        case VALUE_CURRENT_MODE : 
            activMode[nodeID-1] = CURRENT;            
            break;
            
        case VALUE_VELOCITY_MODE : 
            activMode[nodeID-1] = VELOCITY;            
            break;
        
        default :
            return EPOS2_ERROR;            
    }   
    
    wait_ms(PAUSE);
    
    return EPOS2_OK;
}

void setModeOfOperationPDO(const int8_t nodeID, uint8_t mode)
{
    data[0] = mode;            
    cantoepos.write(CANMessage(COB_ID_RECEIVE_PDO_4_ENABLE + nodeID, data, 1)); //RxPDO 4 for Modes of Operation   
    
    //set activMode        
    switch(mode)
    {    
        case VALUE_PROFILE_POSITION_MODE : 
            activMode[nodeID-1] = PROFILE_POSITION;
            break;
            
        case VALUE_POSITION_MODE : 
            activMode[nodeID-1] = POSITION;
            break;
            
        case VALUE_CURRENT_MODE : 
            activMode[nodeID-1] = CURRENT;
            break;
            
        case VALUE_VELOCITY_MODE : 
            activMode[nodeID-1] = VELOCITY;
            break;
        
        default :                        
    }
    
    wait_us(100);        
}

void shutdownControlword(const int8_t nodeID)
{
    CANMessage canmsg;
 
    //Shutdown Controlword Firmware Spec 8.2.84 bit wise 0xxx x110 so 0x0006
    data[0] = WRITING_OBJECT_2_BYTE;
    data[1] = 0x40;
    data[2] = 0x60;
    data[3] = 0x00;
    data[4] = 0x06;     //Low Byte
    data[5] = 0x00;     //High Byte
    data[6] = 0x00;
    data[7] = 0x00;
    cantoepos.write(CANMessage(COB_ID_SDO_CLIENT_TO_SERVER_DEFAULT + nodeID, data, 6));
    while (!(cantoepos.read(canmsg))) 
    {
        wait_us(1);
    }
    wait_ms(PAUSE);
}

void shutdownControlwordIT(const int8_t nodeID)
{
    CANMessage canmsg;
 
    //Shutdown Controlword Firmware Spec 8.2.84 bit wise 0xxx x110 so 0x0006
    data[0] = WRITING_OBJECT_2_BYTE;
    data[1] = 0x40;
    data[2] = 0x60;
    data[3] = 0x00;
    data[4] = 0x06;     //Low Byte
    data[5] = 0x00;     //High Byte
    data[6] = 0x00;
    data[7] = 0x00;
    cantoepos.write(CANMessage(COB_ID_SDO_CLIENT_TO_SERVER_DEFAULT + nodeID, data, 6));    
}

void switchOnEnableOperationControlword(const int8_t nodeID)
{
    CANMessage canmsg;
 
    //Switch On & Enable Operation Controlword Firmware Spec 8.2.84 bit wise 0xxx 1111 so 0x000F
    data[0] = WRITING_OBJECT_2_BYTE;
    data[1] = 0x40;
    data[2] = 0x60;
    data[3] = 0x00;
    data[4] = 0x0F;
    data[5] = 0x00;
    data[6] = 0x00;
    data[7] = 0x00;
    cantoepos.write(CANMessage(COB_ID_SDO_CLIENT_TO_SERVER_DEFAULT + nodeID, data, 6)); 
    while (!(cantoepos.read(canmsg))) 
    {
        wait_us(1);  
    }
    wait_ms(126);     //Tested 5ms not enough, 6ms seems ok
}

void switchOnEnableOperationControlwordIT(const int8_t nodeID)
{
    CANMessage canmsg;
 
    //Switch On & Enable Operation Controlword Firmware Spec 8.2.84 bit wise 0xxx 1111 so 0x000F
    data[0] = WRITING_OBJECT_2_BYTE;
    data[1] = 0x40;
    data[2] = 0x60;
    data[3] = 0x00;
    data[4] = 0x0F;
    data[5] = 0x00;
    data[6] = 0x00;
    data[7] = 0x00;
    cantoepos.write(CANMessage(COB_ID_SDO_CLIENT_TO_SERVER_DEFAULT + nodeID, data, 6));     
}

void faultResetControlword(const int8_t nodeID)
{
    CANMessage canmsg;
 
    //Fault reset Controlword Firmware Spec 8.2.84 bit wise 0xxx x110 so 0x0006
    data[0] = WRITING_OBJECT_2_BYTE;
    data[1] = 0x40;
    data[2] = 0x60;
    data[3] = 0x00;
    data[4] = 0x80;     //Low Byte
    data[5] = 0x00;     //High Byte
    data[6] = 0x00;
    data[7] = 0x00;
    cantoepos.write(CANMessage(COB_ID_SDO_CLIENT_TO_SERVER_DEFAULT + nodeID, data, 6));    
}

int8_t initEposBoard(const int8_t nodeID)
{
    CANMessage canmsg;
    
    //if(nodeID ==1) setNMT(nodeID, CS_RESET_NODE); 
    pc.printf("\t- initialise board ID = %d...", nodeID);  
       
    if(cantoepos.frequency(1000000) != 1)  return EPOS2_ERROR;
    
    //First reset the node, that also clears the red LED that sometimes appears at the end of the CAN bus
    //if(nodeID == 1) setNMT(nodeID, CS_RESET_COMMUNICATION);
    //wait_ms(100);
    //if(nodeID ==1) setNMT(nodeID, CS_RESET_NODE);    
    //wait_ms(100);
    
    //setObjectSDO(nodeID, OBJECT_MAXIMAL_FOLLOWING_ERROR, 0xFFFFFFFA);    
    setNMT(nodeID, CS_ENTER_PRE_OPERATIONAL);    
    wait_ms(100);
    
    //set parameters
    if(setObjectSDO(nodeID, OBJECT_MOTOR_TYPE, BRUSHED_DC_MOTOR) == EPOS2_OK) //do the first as a test if the nodeID exist and board is responding
    {
        setObjectSDO(nodeID, OBJECT_POLE_PAIR_NUMBER, 0x01);
        setObjectSDO(nodeID, OBJECT_MAXIMAL_MOTOR_SPEED, 0x000030D4);
        setObjectSDO(nodeID, OBJECT_THERMAL_TIME_CONSTANT_WINDING, 0x00B5);
        
        //Miscellaneous configuration : Set bit 0 to 1 to Disable sensor supervision by software, to prevent Position Sensor Breach Error
        setObjectSDO(nodeID, OBJECT_MISCELLANEOUS_CONFIGURATION, 0x0000);
        
        //set Position Mode parameters
        setObjectSDO(nodeID, OBJECT_MAXIMAL_FOLLOWING_ERROR, 0x0000C350);
        //setObjectSDO(nodeID, OBJECT_MAXIMAL_PROFILE_VELOCITY, 0x000061A8); //out of range ??
        setObjectSDO(nodeID, OBJECT_MAXIMAL_ACCELERATION, 0x00001332); //0x00000FFF ; 0x00002FFF
        
        //save all the parameters
        setObjectSDO(nodeID, OBJECT_STORE_PARAMETERS, SAVE_ALL);
        
        setNMT(nodeID, CS_RESET_COMMUNICATION);
        
        //setObjectSDO(nodeID, OBJECT_PROFILE_VELOCITY, 0x000003E8);
        //setObjectSDO(nodeID, OBJECT_PROFILE_ACCELERATION, 0x00002710);
        //setObjectSDO(nodeID, OBJECT_PROFILE_DECELERATION, 0x00002710);
        //setObjectSDO(nodeID, OBJECT_QUICKSTOP_DECELERATION, 0x00002710);
        //setObjectSDO(nodeID, OBJECT_MOTION_PROFILE_TYPE, 0x0001); //0= trapezoidal, 1 = sinusoidal
                              
        setNMT(nodeID, CS_ENTER_PRE_OPERATIONAL);
             
            //set RxPDO 1 for position mode
            setPDO(nodeID, RECEIVE_PDO_1_PARAMETER_INDEX, COB_ID_RECEIVE_PDO_1_SUBINDEX, COB_ID_RECEIVE_PDO_1_DISABLE + nodeID, 4);         
                setPDO(nodeID, MAPPED_OBJECT_RECEIVE_PDO_1_INDEX, NUMBER_OBJECTS_RECEIVE_PDO_SUBINDEX, 0x00, 1);      
                    setPDO(nodeID, MAPPED_OBJECT_RECEIVE_PDO_1_INDEX, MAPPED_OBJECT_1_RECEIVE_PDO_SUBINDEX, OBJECT_POSITION_MODE_SETTING_VALUE, 4);                
                    //setPDO(nodeID, MAPPED_OBJECT_RECEIVE_PDO_1_INDEX, MAPPED_OBJECT_1_RECEIVE_PDO_SUBINDEX, OBJECT_TARGET_POSITION, 4);                
                setPDO(nodeID, MAPPED_OBJECT_RECEIVE_PDO_1_INDEX, NUMBER_OBJECTS_RECEIVE_PDO_SUBINDEX, 0x01, 1); //enable RxPDO 1 with 1 object    
            setPDO(nodeID, RECEIVE_PDO_1_PARAMETER_INDEX, COB_ID_RECEIVE_PDO_1_SUBINDEX, COB_ID_RECEIVE_PDO_1_ENABLE + nodeID, 4); 
            
            //set RxPDO 2 for current mode
            setPDO(nodeID, RECEIVE_PDO_2_PARAMETER_INDEX, COB_ID_RECEIVE_PDO_2_SUBINDEX, COB_ID_RECEIVE_PDO_2_DISABLE + nodeID, 4);         
                setPDO(nodeID, MAPPED_OBJECT_RECEIVE_PDO_2_INDEX, NUMBER_OBJECTS_RECEIVE_PDO_SUBINDEX, 0x00, 1);                 
                    setPDO(nodeID, MAPPED_OBJECT_RECEIVE_PDO_2_INDEX, MAPPED_OBJECT_1_RECEIVE_PDO_SUBINDEX, OBJECT_CURRENT_MODE_SETTING_VALUE, 4);              
                setPDO(nodeID, MAPPED_OBJECT_RECEIVE_PDO_2_INDEX, NUMBER_OBJECTS_RECEIVE_PDO_SUBINDEX, 0x01, 1); //enable RxPDO 2 with 1 object    
            setPDO(nodeID, RECEIVE_PDO_2_PARAMETER_INDEX, COB_ID_RECEIVE_PDO_2_SUBINDEX, COB_ID_RECEIVE_PDO_2_ENABLE + nodeID, 4);
            
            //set RxPDO 3 for velocity mode
            setPDO(nodeID, RECEIVE_PDO_3_PARAMETER_INDEX, COB_ID_RECEIVE_PDO_3_SUBINDEX, COB_ID_RECEIVE_PDO_3_DISABLE + nodeID, 4);         
                setPDO(nodeID, MAPPED_OBJECT_RECEIVE_PDO_3_INDEX, NUMBER_OBJECTS_RECEIVE_PDO_SUBINDEX, 0x00, 1);                 
                    setPDO(nodeID, MAPPED_OBJECT_RECEIVE_PDO_3_INDEX, MAPPED_OBJECT_1_RECEIVE_PDO_SUBINDEX, OBJECT_VELOCITY_MODE_SETTING_VALUE, 4);              
                setPDO(nodeID, MAPPED_OBJECT_RECEIVE_PDO_3_INDEX, NUMBER_OBJECTS_RECEIVE_PDO_SUBINDEX, 0x01, 1); //enable RxPDO 3 with 1 object  
            setPDO(nodeID, RECEIVE_PDO_3_PARAMETER_INDEX, COB_ID_RECEIVE_PDO_3_SUBINDEX, COB_ID_RECEIVE_PDO_3_ENABLE + nodeID, 4);
            
            //TODO set RxPDO 4 for fast Mode change
            setPDO(nodeID, RECEIVE_PDO_4_PARAMETER_INDEX, COB_ID_RECEIVE_PDO_4_SUBINDEX, COB_ID_RECEIVE_PDO_4_DISABLE + nodeID, 4);         
                setPDO(nodeID, MAPPED_OBJECT_RECEIVE_PDO_4_INDEX, NUMBER_OBJECTS_RECEIVE_PDO_SUBINDEX, 0x00, 1);                 
                    setPDO(nodeID, MAPPED_OBJECT_RECEIVE_PDO_4_INDEX, MAPPED_OBJECT_1_RECEIVE_PDO_SUBINDEX, OBJECT_MODE_OF_OPERATION, 4);              
                setPDO(nodeID, MAPPED_OBJECT_RECEIVE_PDO_4_INDEX, NUMBER_OBJECTS_RECEIVE_PDO_SUBINDEX, 0x01, 1); //enable RxPDO 4 with 1 object  
            setPDO(nodeID, RECEIVE_PDO_4_PARAMETER_INDEX, COB_ID_RECEIVE_PDO_4_SUBINDEX, COB_ID_RECEIVE_PDO_4_ENABLE + nodeID, 4);
            
            //set TxPDO 1 to get position
            setPDO(nodeID, MAPPED_OBJECT_TRANSMIT_PDO_1_INDEX, NUMBER_OBJECTS_TRANSMIT_PDO_SUBINDEX, 0x00, 1);            
                setPDO(nodeID, MAPPED_OBJECT_TRANSMIT_PDO_1_INDEX, MAPPED_OBJECT_1_TRANSMIT_PDO_SUBINDEX, OBJECT_POSITION_ACTUAL_VALUE, 4);            
            setPDO(nodeID, MAPPED_OBJECT_TRANSMIT_PDO_1_INDEX, NUMBER_OBJECTS_TRANSMIT_PDO_SUBINDEX, 0x01, 1);   
            
            //set TxPDO 2 to get current
            setPDO(nodeID, MAPPED_OBJECT_TRANSMIT_PDO_2_INDEX, NUMBER_OBJECTS_TRANSMIT_PDO_SUBINDEX, 0x00, 1);            
                //setPDO(nodeID, MAPPED_OBJECT_TRANSMIT_PDO_2_INDEX, MAPPED_OBJECT_1_TRANSMIT_PDO_SUBINDEX, OBJECT_CURRENT_ACTUAL_VALUE, 4);  
                setPDO(nodeID, MAPPED_OBJECT_TRANSMIT_PDO_2_INDEX, MAPPED_OBJECT_1_TRANSMIT_PDO_SUBINDEX, OBJECT_CURRENT_ACTUAL_VALUE_AVERAGED, 4);                        
            setPDO(nodeID, MAPPED_OBJECT_TRANSMIT_PDO_2_INDEX, NUMBER_OBJECTS_TRANSMIT_PDO_SUBINDEX, 0x01, 1);   
            
            //set TxPDO 3 to get velocity
            setPDO(nodeID, MAPPED_OBJECT_TRANSMIT_PDO_3_INDEX, NUMBER_OBJECTS_TRANSMIT_PDO_SUBINDEX, 0x00, 1);            
                setPDO(nodeID, MAPPED_OBJECT_TRANSMIT_PDO_3_INDEX, MAPPED_OBJECT_1_TRANSMIT_PDO_SUBINDEX, OBJECT_VELOCITY_ACTUAL_VALUE_AVERAGED, 4);            
            setPDO(nodeID, MAPPED_OBJECT_TRANSMIT_PDO_3_INDEX, NUMBER_OBJECTS_TRANSMIT_PDO_SUBINDEX, 0x01, 1);   
            
            //set TxPDO 4 to get poti position (analog input 1)
            setPDO(nodeID, MAPPED_OBJECT_TRANSMIT_PDO_4_INDEX, NUMBER_OBJECTS_TRANSMIT_PDO_SUBINDEX, 0x00, 1);            
                setPDO(nodeID, MAPPED_OBJECT_TRANSMIT_PDO_4_INDEX, MAPPED_OBJECT_1_TRANSMIT_PDO_SUBINDEX, OBJECT_ANALOG_INPUT_1, 4);            
            setPDO(nodeID, MAPPED_OBJECT_TRANSMIT_PDO_4_INDEX, NUMBER_OBJECTS_TRANSMIT_PDO_SUBINDEX, 0x01, 1);   
                   
            //set TxPDO 1 to RTR mode (request only)       
            setPDO(nodeID, TRANSMIT_PDO_1_PARAMETER_INDEX, COB_ID_TRANSMIT_PDO_1_SUBINDEX, COB_ID_TRANSMIT_PDO_1_DISABLE + nodeID, 4);   
                setPDO(nodeID, TRANSMIT_PDO_1_PARAMETER_INDEX, TRANSMISSION_TYPE_TRANSMIT_PDO_1_SUBINDEX, PARAM_TRANSMISSION_TYPE_RTR, 1);                  
            setPDO(nodeID, TRANSMIT_PDO_1_PARAMETER_INDEX, COB_ID_TRANSMIT_PDO_1_SUBINDEX, COB_ID_TRANSMIT_PDO_1_ENABLE + nodeID, 4);  
            
            //set TxPDO 2 to RTR mode (request only)       
            setPDO(nodeID, TRANSMIT_PDO_2_PARAMETER_INDEX, COB_ID_TRANSMIT_PDO_2_SUBINDEX, COB_ID_TRANSMIT_PDO_2_DISABLE + nodeID, 4);   
                setPDO(nodeID, TRANSMIT_PDO_2_PARAMETER_INDEX, TRANSMISSION_TYPE_TRANSMIT_PDO_2_SUBINDEX, PARAM_TRANSMISSION_TYPE_RTR, 1);                  
            setPDO(nodeID, TRANSMIT_PDO_2_PARAMETER_INDEX, COB_ID_TRANSMIT_PDO_2_SUBINDEX, COB_ID_TRANSMIT_PDO_2_ENABLE + nodeID, 4);  
            
            //set TxPDO 3 to RTR mode (request only)       
            setPDO(nodeID, TRANSMIT_PDO_3_PARAMETER_INDEX, COB_ID_TRANSMIT_PDO_3_SUBINDEX, COB_ID_TRANSMIT_PDO_3_DISABLE + nodeID, 4);   
                setPDO(nodeID, TRANSMIT_PDO_3_PARAMETER_INDEX, TRANSMISSION_TYPE_TRANSMIT_PDO_3_SUBINDEX, PARAM_TRANSMISSION_TYPE_RTR, 1);                  
            setPDO(nodeID, TRANSMIT_PDO_3_PARAMETER_INDEX, COB_ID_TRANSMIT_PDO_3_SUBINDEX, COB_ID_TRANSMIT_PDO_3_ENABLE + nodeID, 4);  
            
            //set TxPDO 4 to RTR mode (request only)       
            setPDO(nodeID, TRANSMIT_PDO_4_PARAMETER_INDEX, COB_ID_TRANSMIT_PDO_4_SUBINDEX, COB_ID_TRANSMIT_PDO_4_DISABLE + nodeID, 4);   
                setPDO(nodeID, TRANSMIT_PDO_4_PARAMETER_INDEX, TRANSMISSION_TYPE_TRANSMIT_PDO_4_SUBINDEX, PARAM_TRANSMISSION_TYPE_RTR, 1);                  
            setPDO(nodeID, TRANSMIT_PDO_4_PARAMETER_INDEX, COB_ID_TRANSMIT_PDO_4_SUBINDEX, COB_ID_TRANSMIT_PDO_4_ENABLE + nodeID, 4);  
        
        //setNMT(nodeID, CS_RESET_COMMUNICATION);  
            //setObjectSDO(nodeID, OBJECT_MAXIMAL_FOLLOWING_ERROR, 0xFFFFFFFD);  
        setNMT(nodeID, CS_START_REMOTE_NODE);    
            //setObjectSDO(nodeID, OBJECT_MAXIMAL_FOLLOWING_ERROR, 0xFFFFFFFC);
                     
            setModeOfOperation(nodeID, VALUE_POSITION_MODE);//VALUE_PROFILE_POSITION_MODE); //this also initialise activMode variable
            
            shutdownControlword(nodeID);  
            switchOnEnableOperationControlword(nodeID);  
            //setObjectSDO(nodeID, OBJECT_MAXIMAL_FOLLOWING_ERROR, 0xFFFFFFFB);
            /*
            //test
            setObjectSDO(nodeID, OBJECT_MAXIMAL_FOLLOWING_ERROR, 0xFFFFFFFD); //try here
            //save all the parameters
            setObjectSDO(nodeID, STORE_PARAMETERS, SAVE_ALL);
            //test end
        */
        pc.printf("...OK\n\r");
    }
    else
    {
        pc.printf("...NOT RESPONDING\n\r");
        
        return EPOS2_ERROR;
    }
    
    return EPOS2_OK;
}

void setPosition(const int8_t nodeID, const int32_t position)
{    
/*         
    if(activMode[nodeID-1] != POSITION) 
    {
        setModeOfOperationPDO(nodeID, VALUE_POSITION_MODE);        
    }
*/
    data[0] = (position & 0x000000FF); //LSB
    data[1] = (position & 0x0000FF00) >> 8;
    data[2] = (position & 0x00FF0000) >> 16;
    data[3] = (position & 0xFF000000) >> 24;          
    cantoepos.write(CANMessage(COB_ID_RECEIVE_PDO_1_ENABLE + nodeID, data, 4)); 
    wait_us(100);            
}

void setCurrent(const int8_t nodeID, const int16_t current)
{      
/*
    if(activMode[nodeID-1] != CURRENT)
    {
        setModeOfOperationPDO(nodeID, VALUE_CURRENT_MODE);      
    }
  */      
    data[0] = (current & 0x00FF);
    data[1] = current >> 8; 
            
    cantoepos.write(CANMessage(COB_ID_RECEIVE_PDO_2_ENABLE + nodeID, data, 2)); //RxPDO 2 for Current
    wait_us(100);
}

void setVelocity(const int8_t nodeID, const int32_t velocity)
{     
/*
    if(activMode[nodeID-1] != VELOCITY) 
    {
        setModeOfOperationPDO(nodeID, VALUE_VELOCITY_MODE);      
    }
  */      
    data[0] = (velocity & 0x000000FF); //LSB
    data[1] = (velocity & 0x0000FF00) >> 8;
    data[2] = (velocity & 0x00FF0000) >> 16;
    data[3] = (velocity & 0xFF000000) >> 24;
    cantoepos.write(CANMessage(COB_ID_RECEIVE_PDO_3_ENABLE + nodeID, data, 4));     
    wait_us(100);     
}

void setForce(const int8_t nodeID, const int16_t force)
{
    //TODO
}

void getPosition(const int8_t nodeID)
{    
    cantoepos.write(CANMessage(COB_ID_TRANSMIT_PDO_1_ENABLE + nodeID)); //the answer frame is caught by interrupt
}

void getCurrent(const int8_t nodeID)
{
    cantoepos.write(CANMessage(COB_ID_TRANSMIT_PDO_2_ENABLE + nodeID));
}

void getVelocity(const int8_t nodeID)
{   
    cantoepos.write(CANMessage(COB_ID_TRANSMIT_PDO_3_ENABLE + nodeID));
}

int16_t getForce(const int8_t nodeID)
{
    //unsigned short force = 0x0000; //unsigned short is 16 bits
    int16_t force = 0x0000;
    float force_f = 0.0;
    
    //setMultiplexerChannel(nodeID);        
    /*
    if((nodeID > 0) && (nodeID <= 8))
    {
        force_f = an1.read();        
    }
    else if((nodeID > 8) && (nodeID <= NUMBER_EPOS2_BOARDS))
    {
        force_f = an3.read();  
    }
    */
    
    force_f = an6.read();
    force_f = force_f*10000;
    force = (int16_t)force_f;
    //pc.printf("%d\n", force); 
    
    return force;
}

void getPotiPosition(const int8_t nodeID)
{   
    cantoepos.write(CANMessage(COB_ID_TRANSMIT_PDO_4_ENABLE + nodeID)); //TODO change it to get analog input on epos2
}

void getIncEnc1CntIdxPls(const int8_t nodeID)
{   
    //Get Incremental Encoder 1 Counter at Index Pulse Firmware Spec 8.2.45 Motor
    data[0] = 0x40; //or 0x00 //read, so no data
    data[1] = 0x21;
    data[2] = 0x20;
    data[3] = 0x00;
    data[4] = 0x00; //null data
    data[5] = 0x00;     
    data[6] = 0x00;
    data[7] = 0x00;
    cantoepos.write(CANMessage(COB_ID_SDO_CLIENT_TO_SERVER_DEFAULT + nodeID, data, 4));
}

void getStatusword(const int8_t nodeID)
{           
    //Get Incremental Encoder 1 Counter at Index Pulse Firmware Spec 8.2.45 Motor
    data[0] = 0x40; //receiving data : Byte 0 is 4B : see App Notes 10.4.2
    data[1] = 0x41;
    data[2] = 0x60;
    data[3] = 0x00;
    data[4] = 0x00; //null data
    data[5] = 0x00;     
    data[6] = 0x00;
    data[7] = 0x00;
    cantoepos.write(CANMessage(COB_ID_SDO_CLIENT_TO_SERVER_DEFAULT + nodeID, data, 4));
}
